package ru.yandex.direct.grid.processing.service.organizations;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneService;
import ru.yandex.direct.core.entity.domain.DomainUtils;
import ru.yandex.direct.core.entity.organization.model.OrganizationCreateRequest;
import ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish;
import ru.yandex.direct.core.entity.organizations.OrganizationsClientTranslatableException;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.organization.GdOrganizationsByUrlInput;
import ru.yandex.direct.grid.processing.model.organization.GdOrganizationsContainer;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.model.organizations.mutation.GdCreateOrganization;
import ru.yandex.direct.grid.processing.model.organizations.mutation.GdCreateOrganizationPayload;
import ru.yandex.direct.grid.processing.service.organizations.loader.CanOverridePhoneDataLoader;
import ru.yandex.direct.grid.processing.service.organizations.loader.OperatorCanEditDataLoader;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.organizations.swagger.OrganizationInfo;
import ru.yandex.direct.organizations.swagger.OrganizationsClientException;
import ru.yandex.direct.organizations.swagger.model.PubApiExtraCompanyField;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.UrlUtils;

import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.organizations.swagger.OrganizationsClient.getLanguageByName;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.misc.io.http.UrlUtils.extractHost;
import static ru.yandex.misc.io.http.UrlUtils.stripWwwFromHost;

/**
 * Сервис для работы с организациями справочника.
 */
@Service
@ParametersAreNonnullByDefault
public class OrganizationsDataService {
    private static final Logger logger = LoggerFactory.getLogger(OrganizationsDataService.class);

    private final OrganizationService organizationService;
    private final ClientPhoneService clientPhoneService;
    private final GridValidationService gridValidationService;

    private final OperatorCanEditDataLoader operatorCanEditDataLoader;
    private final CanOverridePhoneDataLoader canOverridePhoneDataLoader;

    private static final Pattern SLASHES_AT_THE_BEGINNING_PATTERN = Pattern.compile("^/+");
    private static final Pattern SLASHES_AT_THE_END_PATTERN = Pattern.compile("/+$");

    @Autowired
    public OrganizationsDataService(OrganizationService organizationService,
                                    ClientPhoneService clientPhoneService,
                                    GridValidationService gridValidationService,
                                    OperatorCanEditDataLoader operatorCanEditDataLoader,
                                    CanOverridePhoneDataLoader canOverridePhoneDataLoader) {
        this.organizationService = organizationService;
        this.clientPhoneService = clientPhoneService;
        this.gridValidationService = gridValidationService;
        this.operatorCanEditDataLoader = operatorCanEditDataLoader;
        this.canOverridePhoneDataLoader = canOverridePhoneDataLoader;
    }

    /**
     * Возвращает данные по организациям на основе переданного списка пермалинков и языка.
     * Отправляет запрос в Телефонию и добавляет для организаций номера Телефонии в базу, если их еще нет.
     */
    public List<GdOrganization> getOrganizations(User subjectUser,
                                                 GdOrganizationsContainer input,
                                                 Set<ModelProperty<?, ?>> propertiesToFetch)
            throws OrganizationsClientTranslatableException {
        List<GdOrganization> organizationListInternal = getOrganizationListInternal(
                propertiesToFetch,
                fields -> organizationService.getOrganizationsInfo(input.getPermalinkIds(), input.getLanguage(), fields)
        );
        List<Long> permalinkIds = mapList(organizationListInternal, GdOrganization::getPermalinkId);
        clientPhoneService.getAndSaveTelephonyPhones(subjectUser.getClientId(), permalinkIds);
        return organizationListInternal;
    }

    /**
     * Возвращает данные по организациям на основе переданного списка пермалинков и языка.
     */
    public List<GdOrganization> getOrganizationsByUrl(GdOrganizationsByUrlInput input,
                                                      Set<ModelProperty<?, ?>> propertiesToFetch)
            throws OrganizationsClientTranslatableException {
        return getOrganizationListInternal(propertiesToFetch, fields -> getByTextWithOffsetAndLimit(input, fields));
    }

    /**
     * Вернуть "главную" организацию (главная организация сети, самая приоритетная организация),
     * подходящую под ответ. Сейчас метод возвращает организацию только если нашлась ровно одна организация
     * (она по дефолту "главная"), иначе возвращается null.
     *
     * @return информация о главной организации во внутреннем негридовом представлении
     */
    public OrganizationInfo getMainOrganizationInfoByUrl(String url,
                                                         LanguageOuterClass.Language language)
            throws OrganizationsClientTranslatableException {
        var input = new GdOrganizationsByUrlInput()
                .withUrl(url)
                .withLanguage(language);
        Set<PubApiExtraCompanyField> requestedFields = OrganizationsConverter
                .getRequestedSpravFields(GdOrganization.allModelProperties());
        var list = getByTextWithOffsetAndLimit(input, requestedFields);
        return list.size() != 1 ? null : list.get(0);
    }

    public List<GdOrganization> getAllClientOrganizations(User subjectUser,
                                                          Set<ModelProperty<?, ?>> propertiesToFetch) {
        try {
            LanguageOuterClass.Language language = getLanguageByName(
                    LocaleContextHolder.getLocale().getLanguage()).orElse(LanguageOuterClass.Language.EN);
            return getOrganizationListInternal(propertiesToFetch, fields ->
                    organizationService
                            .getApiClientOrganizations(subjectUser.getClientId(), subjectUser.getChiefUid(),
                                    language, null, fields)
                            .stream()
                            .filter(org -> org.getStatusPublish() == OrganizationStatusPublish.PUBLISHED)
                            .collect(Collectors.toList())
            );
        } catch (OrganizationsClientTranslatableException | InterruptedRuntimeException e) {
            logger.error(e.getMessage());
            return List.of();
        }
    }

    /*
     ** fieldsToFetch будет использоваться в дальнейшем
     */
    private <T> List<GdOrganization> getOrganizationListInternal(Set<ModelProperty<?, ?>> fieldsToFetch,
                                                                 Function<Set<PubApiExtraCompanyField>,
                                                                         List<? extends OrganizationInfo>> retriever)
            throws OrganizationsClientTranslatableException {
        Set<PubApiExtraCompanyField> requestedFields = OrganizationsConverter.getRequestedSpravFields(fieldsToFetch);
        return mapList(retriever.apply(requestedFields), OrganizationsConverter::toGdOrganization);
    }

    private List<OrganizationInfo> getByTextWithOffsetAndLimit(GdOrganizationsByUrlInput input,
                                                               Set<PubApiExtraCompanyField> requestedFields)
            throws OrganizationsClientTranslatableException {
        if (StringUtils.isBlank(input.getUrl())) {
            return Collections.emptyList();
        }
        String preparedUrl = prepareUrl(input.getUrl()); // урл без протокола, www и без слешей
        String preparedUrlForSearch; // урл для поиска в Справочнике
        String searchedHost = extractHost(preparedUrl); // хост урла
        if (DomainUtils.isSocialNetworkDomain(searchedHost)) {
            preparedUrlForSearch = preparedUrl;
        } else {
            preparedUrlForSearch = searchedHost + "/*";
        }
        List<OrganizationInfo> resultRaw = organizationService.getOrganizationsInfoByText(preparedUrlForSearch,
                input.getLanguage(), requestedFields);

        List<OrganizationInfo> filteredResult = resultRaw.stream()
                .filter(t -> t.getUrls() != null && t.getUrls().stream().anyMatch(
                        orgUrl -> orgUrl != null && isSameUrl(orgUrl, preparedUrl)))
                .collect(Collectors.toList());
        // если нашли организации по точному совпадению урлов - берем их
        // иначе отфильтровываем организации по совпадению хостов урлов
        if (filteredResult.isEmpty() && searchedHost != null) {
            filteredResult = resultRaw.stream()
                    .filter(t -> t.getUrls() != null && t.getUrls().stream().anyMatch(
                            orgUrl -> orgUrl != null && isSameHost(orgUrl, searchedHost)))
                    .collect(Collectors.toList());
        }
        LimitOffset limitOffset = normalizeLimitOffset(input.getLimitOffset());
        return filteredResult.stream()
                .sorted(Comparator.comparingLong(OrganizationInfo::getPermalinkId))
                .skip(limitOffset.offset())
                .limit(limitOffset.limit())
                .collect(Collectors.toList());
    }

    private boolean isSameUrl(String organizationUrl,
                              String preparedUrl) {
        if (!organizationUrl.contains(preparedUrl)) {
            return false;
        }
        String preparedOrganizationUrl = prepareUrl(organizationUrl);
        return preparedOrganizationUrl.equals(preparedUrl);
    }

    private boolean isSameHost(String organizationUrl,
                               String searchedHost) {
        if (!organizationUrl.contains(searchedHost)) {
            return false;
        }
        String preparedOrganizationUrl = prepareUrl(organizationUrl);
        return extractHost(preparedOrganizationUrl).equals(searchedHost);
    }

    private static String prepareUrl(String url) {
        return trimTrailingSlashes(stripWwwFromHost(UrlUtils.trimProtocol(url)));
    }

    private static String trimTrailingSlashes(String input) {
        String result = input;
        result = SLASHES_AT_THE_BEGINNING_PATTERN.matcher(result).replaceAll("");
        result = SLASHES_AT_THE_END_PATTERN.matcher(result).replaceAll("");
        return result;
    }

    public CompletableFuture<Boolean> getOperatorCanEdit(long permalinkId) {
        return operatorCanEditDataLoader.get().load(permalinkId).thenApply(canEdit -> nvl(canEdit, false));
    }

    public CompletableFuture<Boolean> getCanOverridePhone(long permalinkId) {
        return canOverridePhoneDataLoader.get().load(permalinkId).thenApply(canOverride -> nvl(canOverride, false));
    }

    public GdCreateOrganizationPayload createOnlineOrganization(
            Long operatorUid,
            GdCreateOrganization input
    ) {
        Result<Long> result;
        try {
            result = organizationService.createOnlineOrganization(operatorUid, convertToCore(input));
        } catch (OrganizationsClientException e) {
            return new GdCreateOrganizationPayload()
                    .withIsSpravAvailable(false);
        }

        if (result.isSuccessful()) {
            return new GdCreateOrganizationPayload()
                    .withIsSpravAvailable(true)
                    .withPermalinkId(result.getResult());
        }
        GdValidationResult validationResult = gridValidationService.getValidationResult(result, emptyPath());
        return new GdCreateOrganizationPayload()
                .withIsSpravAvailable(true)
                .withValidationResult(validationResult);
    }

    private OrganizationCreateRequest convertToCore(GdCreateOrganization input) {
        return new OrganizationCreateRequest()
                .withName(input.getName())
                .withServiceAreaGeoIds(input.getServiceAreaGeoIds())
                .withRubrics(input.getRubrics())
                .withUrl(input.getUrl())
                .withPhoneFormatted(input.getPhoneFormatted());
    }
}
