package ru.yandex.direct.grid.processing.service.organizations;

import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.organizations.OrganizationsClientTranslatableException;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.organization.GdOrganizationsByUrlInput;
import ru.yandex.direct.grid.processing.model.organization.GdOrganizationsContainer;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganizationAccess;
import ru.yandex.direct.grid.processing.model.organizations.mutation.GdCreateOrganization;
import ru.yandex.direct.grid.processing.model.organizations.mutation.GdCreateOrganizationPayload;
import ru.yandex.direct.grid.processing.processor.util.GridRequestedFields;
import ru.yandex.direct.grid.processing.processor.util.GridRequestedFieldsHolder;
import ru.yandex.direct.model.ModelProperty;

/**
 * Сервис, отдающий информацию об организациях справочника
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class OrganizationsGraphQlService {
    private final OrganizationsDataService organizationsDataService;

    @Autowired
    public OrganizationsGraphQlService(OrganizationsDataService organizationsDataService) {
        this.organizationsDataService = organizationsDataService;
    }

    /**
     * GraphQL запрос. Получить данные об организациях по их пермалинкам
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "organizations")
    public List<GdOrganization> getOrganizations(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdOrganizationsContainer input,
            @GridRequestedFields GridRequestedFieldsHolder requestedFieldsHolder)
            throws OrganizationsClientTranslatableException {
        Set<ModelProperty<?, ?>> propertySet =
                requestedFieldsHolder.getRequestedModelFields(GdOrganization.allModelProperties());
        User subjectUser = context.getSubjectUser();
        return organizationsDataService.getOrganizations(subjectUser, input, propertySet);
    }

    /**
     * GraphQL запрос. Получить данные об организациях по ссылке на сайт
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "organizationsByUrl")
    public List<GdOrganization> getOrganizationsByUrl(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdOrganizationsByUrlInput input,
            @GridRequestedFields GridRequestedFieldsHolder requestedFieldsHolder)
            throws OrganizationsClientTranslatableException {
        Set<ModelProperty<?, ?>> propertySet =
                requestedFieldsHolder.getRequestedModelFields(GdOrganization.allModelProperties());
        return organizationsDataService.getOrganizationsByUrl(input, propertySet);
    }

    /**
     * GraphQL подзапрос. Возвращает {@code true}, если оператор может редактировать организацию
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "operatorCanEdit")
    public CompletableFuture<Boolean> operatorCanEdit(@GraphQLContext GdOrganizationAccess organizationAccess) {
        if (organizationAccess.getPermalinkId() == null) {
            return CompletableFuture.completedFuture(false);
        }
        return organizationsDataService.getOperatorCanEdit(organizationAccess.getPermalinkId());
    }

    /**
     * GraphQL подзапрос. Возвращает {@code true}, если можно переопределять телефон на свой
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "canOverridePhone")
    public CompletableFuture<Boolean> canOverridePhone(@GraphQLContext GdOrganizationAccess organizationAccess) {
        if (organizationAccess.getPermalinkId() == null) {
            return CompletableFuture.completedFuture(false);
        }
        return organizationsDataService.getCanOverridePhone(organizationAccess.getPermalinkId());
    }

    /**
     * Создание online организации
     */
    @GraphQLNonNull
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "createOnlineOrganization")
    public GdCreateOrganizationPayload createOnlineOrganization(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCreateOrganization input
    ) {
        Long operatorUid = context.getOperator().getUid();
        return organizationsDataService.createOnlineOrganization(operatorUid, input);
    }
}
