package ru.yandex.direct.grid.processing.service.organizations.loader;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CanOverridePhoneDataLoader extends GridBatchingDataLoader<Long, Boolean> {
    private static final Logger logger = LoggerFactory.getLogger(CanOverridePhoneDataLoader.class);

    public CanOverridePhoneDataLoader(GridContextProvider gridContextProvider,
                                      OrganizationService organizationService) {
        this.dataLoader = mappedDataLoader(gridContextProvider,
                getBatchLoadFunction(organizationService, () -> retrieveClientId(gridContextProvider)));
    }

    private MappedBatchLoaderWithContext<Long, Boolean> getBatchLoadFunction(
            OrganizationService organizationService, Supplier<ClientId> clientIdSupplier) {
        return (permalinkIds, environment) -> {
            ClientId clientId = clientIdSupplier.get();
            if (clientId == null) {
                if (!CollectionUtils.isEmpty(permalinkIds)) {
                    logger.error("clientId cannot be null");
                }
                return CompletableFuture.completedFuture(Map.<Long, Boolean>of());
            }
            Map<Long, Boolean> hasAccessByPermalinkId = organizationService.hasAccessNoThrow(clientId, permalinkIds);
            var isEditableByPermalink = listToMap(permalinkIds, identity(),
                    id -> hasAccessByPermalinkId.getOrDefault(id, false));
            return CompletableFuture.completedFuture(isEditableByPermalink);
        };
    }

    //Дальше идёт костыль, так как clientId  получают разными способами.
    // Других пока нет, так что текущий костыль должен быть исчерпывающим
    private static ClientId retrieveClientId(GridContextProvider gridContextProvider) {
        ClientId clientId = ClientId.fromNullableLong(
                ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                        GridGraphQLContext::getQueriedClient),
                        GdClientInfo::getId));
        if (clientId == null) {
            clientId = ifNotNull(ifNotNull(gridContextProvider.getGridContext(),
                    GridGraphQLContext::getSubjectUser),
                    User::getClientId);
        }
        return clientId;
    }
}
