package ru.yandex.direct.grid.processing.service.organizations.loader;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class OperatorCanEditDataLoader extends GridBatchingDataLoader<Long, Boolean> {
    private static final Logger logger = LoggerFactory.getLogger(OperatorCanEditDataLoader.class);

    public OperatorCanEditDataLoader(GridContextProvider gridContextProvider,
                                     OrganizationService organizationService) {
        this.dataLoader = mappedDataLoader(gridContextProvider,
                getBatchLoadFunction(organizationService, () -> retrieveOperatorUid(gridContextProvider)));
    }

    private MappedBatchLoaderWithContext<Long, Boolean> getBatchLoadFunction(
            OrganizationService organizationService, Supplier<Long> uidSupplier) {
        return (permalinkIds, environment) -> {
            Long uid = uidSupplier.get();
            if (uid == null) {
                if (!CollectionUtils.isEmpty(permalinkIds)) {
                    logger.error("uid cannot be null");
                }
                return CompletableFuture.completedFuture(Map.<Long, Boolean>of());
            }
            var editableOrganizations = organizationService
                    .operatorEditableOrganizations(uidSupplier.get(), permalinkIds);
            var isEditableByPermalink = listToMap(permalinkIds, identity(), id -> editableOrganizations.contains(id));
            return CompletableFuture.completedFuture(isEditableByPermalink);
        };

    }

    private static Long retrieveOperatorUid(GridContextProvider gridContextProvider) {
        return ifNotNull(ifNotNull(
                gridContextProvider.getGridContext(), GridGraphQLContext::getOperator),
                User::getUid);
    }
}
