package ru.yandex.direct.grid.processing.service.page;

import java.util.List;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService;
import ru.yandex.direct.grid.core.util.ordering.OrderingProcessor;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.page.GdPage;
import ru.yandex.direct.grid.processing.model.page.GdPageContext;
import ru.yandex.direct.grid.processing.model.page.GdPageOrderBy;
import ru.yandex.direct.grid.processing.model.page.GdPageOrderByField;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendation;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiRemovePagesFromBlackList;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationWithKpi;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService.RECOMMENDATIONS_CLICKS_EXTRACTOR;
import static ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService.RECOMMENDATIONS_COST_EXTRACTOR;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;

/**
 * Сервис, возвращающий данные о рекомендациях по площадкам
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class PageGraphQlService {
    private final GridValidationService gridValidationService;

    /**
     * Обработчик запросов на сортировку результата
     */
    private static final OrderingProcessor<GdPageOrderByField, GdPage> PAGE_ORDERING_PROCESSOR =
            OrderingProcessor.<GdPageOrderByField, GdPage>builder(false)
                    .withFieldComparator(GdPageOrderByField.CLICKS,
                            recommendations(RECOMMENDATIONS_CLICKS_EXTRACTOR))
                    .withFieldComparator(GdPageOrderByField.COST,
                            recommendations(RECOMMENDATIONS_COST_EXTRACTOR))
                    .build();

    private final GridRecommendationService gridRecommendationService;

    private static <T> Function<GdPage, T> recommendations(Function<List<GdRecommendation>, T> extractor) {
        return c -> c.getRecommendations() == null ? null : extractor.apply(c.getRecommendations());
    }

    @Autowired
    public PageGraphQlService(GridValidationService gridValidationService,
                              GridRecommendationService gridRecommendationService) {

        this.gridValidationService = gridValidationService;
        this.gridRecommendationService = gridRecommendationService;
    }

    /**
     * GraphQL подзапрос. Получает информацию о рекомендациях по площадкам клиента
     *
     * @param clientContext клиент, к которому относятся рекомендации
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "forbiddenPages")
    public GdPageContext getForbiddenPages(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @Nullable @GraphQLArgument(name = "orderBy") List<@GraphQLNonNull GdPageOrderBy> orderByList,
            @Nullable @GraphQLArgument(name = "limitOffset") GdLimitOffset limitOffset,
            @Nullable @GraphQLArgument(name = "cacheKey") String cacheKey) {
        gridValidationService.validateLimitOffset(limitOffset);
        final LimitOffset range = normalizeLimitOffset(limitOffset);

        final GdClientInfo client = context.getQueriedClient();

        final Long clientId = client.getId();

        final List<GdRecommendation> recommendations = gridRecommendationService.getPageRecommendationsGroupedByPage(clientId, context.getOperator());

        final List<GdPage> rowset = recommendations.stream()
                .filter(r -> GdRecommendationWithKpi.class.isAssignableFrom(r.getClass()))
                .map(GdRecommendationWithKpi.class::cast)
                .filter(r -> r.getKpi() != null)
                .map(r -> new GdPage()
                        .withName(((GdRecommendationKpiRemovePagesFromBlackList) r.getKpi()).getPageName())
                        .withRecommendations(singletonList(r)))
                .sorted(PAGE_ORDERING_PROCESSOR.comparator(orderByList))
                .skip(range.offset())
                .limit(range.limit())
                .collect(toList());

        return new GdPageContext()
                .withTotalCount(rowset.size())
                .withRowset(rowset);
    }

}
