package ru.yandex.direct.grid.processing.service.payment;

import java.math.BigDecimal;
import java.math.MathContext;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.exception.BalanceClientException;
import ru.yandex.direct.balance.client.model.response.GetCardBindingURLResponse;
import ru.yandex.direct.balance.client.model.response.GetClientPersonsResponseItem;
import ru.yandex.direct.core.entity.adgroup.model.StatusAutobudgetShow;
import ru.yandex.direct.core.entity.autobudget.model.AutobudgetHourlyProblem;
import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetHourlyAlertRepository;
import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils;
import ru.yandex.direct.core.entity.campaign.DayBudgetChangeLogRecord;
import ru.yandex.direct.core.entity.campaign.model.CampaignDayBudgetOptions;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetNotificationStatus;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo;
import ru.yandex.direct.core.entity.client.model.ClientSpent;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.repository.ClientSpentRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.eventlog.model.EventCampaignAndTimeData;
import ru.yandex.direct.core.entity.eventlog.model.EventLogType;
import ru.yandex.direct.core.entity.eventlog.repository.EventLogRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.mailnotification.model.CampaignEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.payment.model.AutopaySettings;
import ru.yandex.direct.core.entity.payment.model.CardInfo;
import ru.yandex.direct.core.entity.payment.repository.AutopaySettingsRepository;
import ru.yandex.direct.core.entity.payment.service.AutopayService;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeHelper;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeValidationContainer;
import ru.yandex.direct.core.entity.statistics.model.Period;
import ru.yandex.direct.core.entity.statistics.service.OrderStatService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.model.UserOptions;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.core.service.integration.balance.model.CreateAndPayRequestResult;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.currencies.CurrencyRub;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsDayBudgetShowMode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.model.payment.GdAutopayStatus;
import ru.yandex.direct.grid.processing.model.payment.GdClientPaymentCard;
import ru.yandex.direct.grid.processing.model.payment.GdGetAutopaySettingsPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetCardBindingUrl;
import ru.yandex.direct.grid.processing.model.payment.GdGetCardBindingUrlPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetClientPaymentCardsPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetPaymentFormUrl;
import ru.yandex.direct.grid.processing.model.payment.GdGetPaymentFormUrlPayload;
import ru.yandex.direct.grid.processing.model.payment.GdPayForAll;
import ru.yandex.direct.grid.processing.model.payment.GdPayForAllPayload;
import ru.yandex.direct.grid.processing.model.payment.GdSetUpDayBudget;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.intapi.client.model.response.PayForAllResponse;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.StringUtils;
import ru.yandex.direct.utils.TimeProvider;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.lessThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class PaymentDataService {
    private static final Logger LOGGER = LoggerFactory.getLogger(PaymentDataService.class);

    private final ShardHelper shardHelper;
    private final BalanceService balanceService;
    private final IntApiClient intApiClient;
    private final AutopayService autopayService;
    private final CampaignService campaignService;
    private final ClientService clientService;
    private final WalletRepository walletRepository;
    private final AutopaySettingsRepository autopaySettingsRepository;
    private final GridValidationResultConversionService validationResultConversionService;
    private final PromocodeHelper promocodeHelper;
    private final CampaignRepository campaignRepository;
    private final ClientRepository clientRepository;
    private final EventLogRepository eventLogRepository;
    private final TimeProvider timeProvider = new TimeProvider();
    private final OrderStatService orderStatService;
    private final UserRepository userRepository;
    private final FeatureService featureService;
    private final AutobudgetHourlyAlertRepository autobudgetHourlyAlertRepository;
    private final MailNotificationEventService mailNotificationEventService;
    private final ClientSpentRepository clientSpentRepository;

    private static final long OVERWRITTEN_RECOMMENDATION_SUM_MIN_RUB = 10000;
    private static final long MAX_DAY_BUDGET_DAILY_CHANGE_COUNT = 3;
    private static final String LOGGER_NAME = "PPCLOG_CMD.log";
    private static final Logger changeLogger = LoggerFactory.getLogger(LOGGER_NAME);

    @SuppressWarnings("checkstyle:ParameterNumber")
    @Autowired
    public PaymentDataService(ShardHelper shardHelper,
                              BalanceService balanceService,
                              IntApiClient intApiClient,
                              AutopayService autopayService,
                              CampaignService campaignService,
                              ClientService clientService,
                              WalletRepository walletRepository,
                              AutopaySettingsRepository autopaySettingsRepository,
                              GridValidationResultConversionService validationResultConversionService,
                              PromocodeHelper promocodeHelper,
                              CampaignRepository campaignRepository,
                              ClientRepository clientRepository,
                              EventLogRepository eventLogRepository,
                              OrderStatService orderStatService,
                              UserRepository userRepository,
                              FeatureService featureService,
                              AutobudgetHourlyAlertRepository autobudgetHourlyAlertRepository,
                              MailNotificationEventService mailNotificationEventService,
                              ClientSpentRepository clientSpentRepository
    ) {
        this.shardHelper = shardHelper;
        this.balanceService = balanceService;
        this.intApiClient = intApiClient;
        this.autopayService = autopayService;
        this.campaignService = campaignService;
        this.clientService = clientService;
        this.walletRepository = walletRepository;
        this.autopaySettingsRepository = autopaySettingsRepository;
        this.validationResultConversionService = validationResultConversionService;
        this.promocodeHelper = promocodeHelper;
        this.campaignRepository = campaignRepository;
        this.clientRepository = clientRepository;
        this.eventLogRepository = eventLogRepository;
        this.orderStatService = orderStatService;
        this.userRepository = userRepository;
        this.featureService = featureService;
        this.autobudgetHourlyAlertRepository = autobudgetHourlyAlertRepository;
        this.mailNotificationEventService = mailNotificationEventService;
        this.clientSpentRepository = clientSpentRepository;
    }

    /**
     * Получить информацию о банковских картах клиента
     */
    GdGetClientPaymentCardsPayload getClientPaymentCards(Long uid, ClientId clientId) {
        CurrencyCode currency = clientService.getWorkCurrency(clientId).getCode();
        List<CardInfo> clientCards = balanceService.getClientPaymentCards(uid, currency.name());
        List<GdClientPaymentCard> gdClientCardsList = clientCards.stream()
                .map(x -> new GdClientPaymentCard()
                        .withCardId(x.getCardId())
                        .withMaskedPan(x.getMaskedNumber()))
                .collect(toList());
        return new GdGetClientPaymentCardsPayload().withCards(gdClientCardsList);
    }

    /**
     * Получить адрес страницы баланса для редиректа
     */
    GdPayForAllPayload payForAll(
        Long uid,
        Long operatorUid,
        RbacRole role,
        String login,
        GdPayForAll input,
        String requestUrl,
        Locale locale
    ) {
        PayForAllResponse response = intApiClient.payForAll(
            role.toString().toLowerCase(),
            uid,
            login,
            input.getCid(),
            operatorUid,
            input.getSum(),
            requestUrl,
            input.getWithNds(),
            locale
        );

        ItemValidationBuilder<PayForAllResponse, Defect> validator = ItemValidationBuilder.of(response);

        validator.item(response.getErrorCode(), "error_code").check(isNull());
        validator.item(response.getError(), "error").check(isNull());

        return new GdPayForAllPayload()
                .withUrl(response.getUrl())
                .withValidationResult(
                        new GridValidationResultConversionService().buildGridValidationResult(validator.getResult())
                );
    }

    /**
     * Получить ссылку на страницу привязки карты (без оплаты)
     */
    GdGetCardBindingUrlPayload getCardBindingUrl(Long operatorUid, ClientId clientId, GdGetCardBindingUrl input) {
        CurrencyCode currency = clientService.getWorkCurrency(clientId).getCode();
        GetCardBindingURLResponse cardBinding = balanceService.getCardBinding(operatorUid, currency, null,
                input.getIsMobile());
        String url = cardBinding.getBindingUrl();
        return new GdGetCardBindingUrlPayload().withBindingUrl(url);
    }

    /**
     * Получить информацию о настройках автопополнения
     */
    public GdGetAutopaySettingsPayload getAutopaySettings(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        CurrencyCode clientCurrency = clientService.getWorkCurrency(clientId).getCode();

        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency);
        var result = new GdGetAutopaySettingsPayload()
                .withAutopayStatus(GdAutopayStatus.OFF)
                .withAllowBindUnbind(false);

        if (walletCid == null) {
            return result;
        }
        RbacRole role = clientService.massGetRolesByClientId(List.of(clientId)).get(clientId);
        Boolean isAutopayOn = walletRepository.isAutopayEnabled(shard, walletCid);

        result
                .withAllowBindUnbind(role == RbacRole.CLIENT)
                .withAutopayStatus(isAutopayOn ? GdAutopayStatus.ON : GdAutopayStatus.OFF);

        if (result.getAutopayStatus().equals(GdAutopayStatus.OFF)) {
            return result;
        }

        AutopaySettings autopaySettings = autopaySettingsRepository.getAutopaySettings(shard, walletCid);

        result
            .withCardId(autopaySettings.getCardId())
            .withPaymentSum(autopaySettings.getPaymentSum())
            .withRemainingSum(autopaySettings.getRemainingSum())
            .withIsLegal(Boolean.FALSE);

        var personId = autopaySettings.getPersonId();
        if (personId != null) {
            try {
                var person = balanceService.getPerson(clientId, personId);
                if (person != null) {
                    result.withIsLegal(person.getType().equals(GetClientPersonsResponseItem.LEGAL_PERSON_TYPE));
                }
            }
            catch (BalanceClientException e) {
                // test balance throws here for some clients on TC (since we have different clientIds for prod and test)
                LOGGER.warn("Failed to get person from balance: " + e.getMessage());
            }
        }
        return result;
    }

    /**
     * Удалить настройки автопополнения
     */
    public void removeAutopay(User user) {
        ClientId clientId = user.getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        autopayService.removeAutopay(shard, clientId);
    }

    /**
     * Получить ссылку на страницу оплаты
     */
    GdGetPaymentFormUrlPayload getPaymentFormUrl(User user, Long operatorUid, GdGetPaymentFormUrl input) {
        BigDecimal sum = input.getPaymentSum();
        checkArgument(sum.compareTo(BigDecimal.ZERO) > 0, "sum is negative");

        input.setPromocode(StringUtils.nullIfBlank(input.getPromocode()));

        ClientId clientId = user.getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        CurrencyCode clientCurrency = clientService.getWorkCurrency(clientId).getCode();

        Long walletCid = walletRepository.getClientWalletId(shard, clientId, clientCurrency, false);
        checkNotNull(walletCid, "no client wallet");

        Long personIdToPay = getPersonId(user, operatorUid, input.getIsLegalPerson());

        PromocodeValidationContainer promocodeValidationContainer = promocodeHelper
                .preparePromocodeValidationContainer(clientId);
        var vr = validatePayment(input, promocodeValidationContainer);
        if (vr != null && vr.hasAnyErrors()) {
            LOGGER.error("errors in grid payment validation: " + vr.flattenErrors().get(0).toString());
            return new GdGetPaymentFormUrlPayload().withValidationResult(
                    validationResultConversionService.buildGridValidationResultIfErrorsOrWarnings(vr, path()));
        }

        boolean denyPromocode = (input.getPromocode() != null) &&
                promocodeHelper.doDenyPromocode(promocodeValidationContainer);
        CreateAndPayRequestResult createAndPayRequestResult = balanceService.createAndPayRequest(
                operatorUid, clientId, walletCid, clientCurrency, sum, personIdToPay,
                null, true, input.getPromocode(), denyPromocode);

        return new GdGetPaymentFormUrlPayload()
                .withPaymentUrl(createAndPayRequestResult.getPaymentUrl(input.getIsMobile()))
                .withAmount(createAndPayRequestResult.getAmount())
                .withValidationResult(validationResultConversionService.buildGridValidationResultIfErrorsOrWarnings(
                        createAndPayRequestResult.getPaymentUrlValidationResult(), path()));
    }

    ItemValidationBuilder<GdSetUpDayBudget, Defect> setUpDayBudget(
        User user, GdSetUpDayBudget input, Long operatorUid) {

        var currency = clientService.getWorkCurrency(user.getClientId());

        ItemValidationBuilder<GdSetUpDayBudget, Defect> validator = ItemValidationBuilder.of(input);

        var clientId = user.getClientId();
        var shard = shardHelper.getShardByClientIdStrictly(clientId);

        var walletId = walletRepository.getClientWalletId(shard, clientId, currency.getCode(), false);

        validator.item(walletId, "wallet").check(notNull());
        if (walletId == null) {
            return validator;
        }

        var optionsList = campaignRepository.getDayBudgetOptions(shard, singletonList(walletId));
        var options = optionsList.get(walletId);

        var newDayBudget = input.getSum();
        var oldDayBudget = options.getDayBudget();
        if (!input.getState()) {
            newDayBudget = BigDecimal.ZERO;
        }
        var oldBudgetShowModeIsDefault = options.getDayBudgetShowMode() == CampaignsDayBudgetShowMode.default_;

        if (oldDayBudget.compareTo(newDayBudget) == 0 && oldBudgetShowModeIsDefault) return validator;

        validateDayBudget(currency, input, validator, options, newDayBudget);

        if (validator.getResult().hasAnyErrors()) {
            return validator;
        }

        var campaignIds = campaignRepository
                .getCampaignsByWalletIds(shard, singletonList(walletId), false)
                .stream().map(WalletCampaign::getId).collect(toList());

        campaignRepository.updateDailyBudgetAndResetShowMode(shard, walletId, newDayBudget);

        if (oldDayBudget.compareTo(newDayBudget) != 0) {
            var event = CampaignEvent.changedDayBudgetEvent(
                    operatorUid, user.getChiefUid(), walletId, currency.getCode(), oldDayBudget, newDayBudget);

            mailNotificationEventService.queueEvents(operatorUid, clientId, singletonList(event));

            logDayBudgetChange(walletId, operatorUid, oldDayBudget, newDayBudget, currency.getCode());
        }

        // сбрасываем установленные ранее отметки о приостановке показов автобюджетом если включили дневной бюджет
        // или сменился режим показов
        if (
            (oldDayBudget.compareTo(BigDecimal.ZERO) == 0 && newDayBudget.compareTo(BigDecimal.ZERO) > 0) ||
            (newDayBudget.compareTo(BigDecimal.ZERO) > 0 && !oldBudgetShowModeIsDefault)
        ) {
            campaignRepository.updateStatusAutoBudgetShow(shard, campaignIds, StatusAutobudgetShow.YES);
        }

        resetAutobudgetAlerts(shard, campaignIds, oldDayBudget, newDayBudget);

        resetStopTime(currency, shard, walletId, options, newDayBudget, oldDayBudget);

        options.withDayBudgetDailyChangeCount(options.getDayBudgetDailyChangeCount() + 1);

        campaignRepository.updateDayBudgetOptions(shard, options);

        return validator;
    }


    private void resetAutobudgetAlerts(
            int shard,
            List<Long> campaignIds,
            BigDecimal oldDayBudget,
            BigDecimal newDayBudget) {

        if (
            oldDayBudget.compareTo(BigDecimal.ZERO) == 0 ||
            newDayBudget.compareTo(BigDecimal.ZERO) != 0 && newDayBudget.compareTo(oldDayBudget) <= 0
        ) {
            return;
        }

        var alertMap = autobudgetHourlyAlertRepository.getAlerts(shard, campaignIds);
        var campaignsWithAlerts = new ArrayList<Long>();
        for (var entry : alertMap.entrySet()) {
            if (entry.getValue().getProblems().contains(AutobudgetHourlyProblem.WALLET_DAILY_BUDGET_REACHED)) {
                campaignsWithAlerts.add(entry.getKey());
            }
        }
        autobudgetHourlyAlertRepository.freezeAlerts(shard, campaignsWithAlerts);
    }

    private void logDayBudgetChange(
            Long walletId,
            Long operatorUid,
            BigDecimal oldDayBudget,
            BigDecimal newDayBudget,
            CurrencyCode currencyCode) {

        var record = new DayBudgetChangeLogRecord(Trace.current().getTraceId());
        record.setCids(List.of(walletId));
        record.setPath(DayBudgetChangeLogRecord.LOG_DAY_BUDGET_CHANGE_CMD_NAME);
        record.setOperatorId(operatorUid);

        Map<String, Object> params = record.getParam();
        params.put("cid", walletId);
        params.put("old_day_budget", Objects.requireNonNull(oldDayBudget).toPlainString());
        params.put("new_day_budget", Objects.requireNonNull(newDayBudget).toPlainString());
        params.put("currency", Objects.requireNonNull(currencyCode).name());

        changeLogger.info("{} {}", record.getPrefixLogTime(), JsonUtils.toJson(record));
    }

    private void validateDayBudget(Currency currency, GdSetUpDayBudget input,
                           ItemValidationBuilder<GdSetUpDayBudget, Defect> v, CampaignDayBudgetOptions options,
                           BigDecimal newSum) {

        if (input.getState()) {
            v.item(
                options.getDayBudgetDailyChangeCount(),
                "dayBudgetDailyChangeCount"
            ).check(lessThan(MAX_DAY_BUDGET_DAILY_CHANGE_COUNT));

            v.item(newSum, GdSetUpDayBudget.SUM.name())
                    .check(notGreaterThan(currency.getMaxDailyBudgetAmount()))
                    .check(notLessThan(currency.getMinWalletDayBudget()));
        }
    }

    /**
     * Сбрасываем options[DayBudgetStopTime] если дневной бюджет сбросили или увеличили,
     * (и он превышает потраченное за день)
     */
    private void resetStopTime(
            Currency currency,
            int shard,
            Long walletId,
            CampaignDayBudgetOptions options,
            BigDecimal newSum,
            BigDecimal oldSum) {

        boolean resetStopTime = oldSum.compareTo(BigDecimal.ZERO) > 0 && newSum.compareTo(BigDecimal.ZERO) == 0;

        var dayBudgetStopTime = options.getDayBudgetStopTime();
        if (oldSum.compareTo(BigDecimal.ZERO) > 0 && newSum.compareTo(oldSum) > 0 && dayBudgetStopTime != null) {
            LocalDateTime now = timeProvider.now();
            List<Period> periodList = singletonList(new Period(
                    "day", now.minus(1, ChronoUnit.DAYS).toLocalDate(), now.toLocalDate()));

            var spentMoney = getMoneySpentForWallet(shard, walletId, periodList, currency.getCode())
                    .get("day").bigDecimalValue();

            if (newSum.subtract(spentMoney).compareTo(Currencies.EPSILON) > 0) {
                resetStopTime = false;
            }
        }

        if (resetStopTime) {
            options.setDayBudgetStopTime(null);
            options.setDayBudgetNotificationStatus(DayBudgetNotificationStatus.READY);
        }
    }


    private ValidationResult<GdGetPaymentFormUrl, Defect> validatePayment(GdGetPaymentFormUrl request,
                                                                          PromocodeValidationContainer
                                                                                  promocodeValidationContainer) {
        ItemValidationBuilder<GdGetPaymentFormUrl, Defect> v = ItemValidationBuilder.of(request, Defect.class);

        v.item(request.getPromocode(), "promocode")
                .check(fromPredicate(t -> promocodeHelper.isApplicablePromocode(t, promocodeValidationContainer),
                        Defects.promocodeDomainOrClientDoesNotMatch()), When.isValid());

        return v.getResult();
    }

    private Long getPersonId(User user, Long operatorUid, boolean isLegalPerson) {
        Long personIdToPay = balanceService.getOrCreatePerson(user, operatorUid, isLegalPerson);
        checkNotNull(personIdToPay, "no person to pay");
        return personIdToPay;
    }

    @Nullable BigDecimal getDaysSinceMoneyOut(int shard, @Nullable Long walletId, ClientId clientId) {
        if (walletId == null) {
            return null;
        }

        Client client = clientService.getClient(clientId);
        LocalDate today = LocalDateTime.now().toLocalDate();
        WalletCampaign walletCampaign = campaignRepository.getWalletsByWalletCampaignIds(
                shard, Set.of(walletId)).stream().findFirst().get();
        Money walletRestMoney = campaignService.getWalletsRestMoneyByWalletCampaignIds(
                shard, Set.of(walletId)).get(walletId).getRest();
        BigDecimal walletDebt = campaignRepository.getWalletsDebt(shard, Set.of(walletId)).getOrDefault(
                walletId, BigDecimal.ZERO).negate();
        List<ClientAutoOverdraftInfo> clientOverdraftInfoList = clientRepository.getClientsAutoOverdraftInfo(
                shard, Set.of(clientId));
        boolean isOverdraftEnabled = false;
        Money autoOverdraftAddition = Money.valueOf(BigDecimal.ZERO, walletCampaign.getCurrency());
        if (!clientOverdraftInfoList.isEmpty()) {
            ClientAutoOverdraftInfo clientOverdraftInfo = clientOverdraftInfoList.get(0);
            autoOverdraftAddition = Money.valueOf(
                    AutoOverdraftUtils.calculateAutoOverdraftAddition(
                            walletCampaign.getCurrency(), walletCampaign.getSum(), walletDebt, clientOverdraftInfo),
                    walletCampaign.getCurrency());
            isOverdraftEnabled = clientOverdraftInfo.getAutoOverdraftLimit().compareTo(BigDecimal.ZERO) > 0 &&
                    clientOverdraftInfo.getOverdraftLimit().compareTo(BigDecimal.ZERO) > 0 &&
                    !AutoOverdraftUtils.CLIENT_IDS_WITH_TWO_ACTIVE_WALLETS.contains(clientId.asLong()) &&
                    !client.getIsBrand() &&
                    AutoOverdraftUtils.AVAILABLE_CURRENCY_CODES.contains(client.getWorkCurrency());
        }
        Money moneyOutLimit = Money.valueOf(
                Currencies.getCurrency(walletCampaign.getCurrency()).getMoneyOutLimit(), walletCampaign.getCurrency());
        BigDecimal daysSinceMoneyOut = null;
        if (walletRestMoney.add(autoOverdraftAddition).lessThan(moneyOutLimit)) {
            if (isOverdraftEnabled) {
                EventCampaignAndTimeData lastEvent = eventLogRepository.getLastEventByCliendIdAndCampaignId(
                        shard, clientId, walletId, EventLogType.MONEY_OUT_WALLET_WITH_AO);
                if (!Objects.isNull(lastEvent)) {
                    daysSinceMoneyOut = BigDecimal.valueOf(
                            (int) ChronoUnit.DAYS.between(lastEvent.getEventTime().toLocalDate(), today));
                }
            } else {
                EventCampaignAndTimeData lastEvent = eventLogRepository.getLastEventByCliendIdAndCampaignId(
                        shard, clientId, walletId, EventLogType.MONEY_OUT_WALLET);
                if (!Objects.isNull(lastEvent)) {
                    daysSinceMoneyOut = BigDecimal.valueOf(
                            (int) ChronoUnit.DAYS.between(lastEvent.getEventTime().toLocalDate(), today));
                }
            }
        }
        return daysSinceMoneyOut;
    }

    BigDecimal getAverageDailyBudget(
            int shard,
            CurrencyCode currencyCode,
            ClientId clientId,
            @Nullable Long walletCampaignId) {

        final int daysInWeek = 7;

        if (walletCampaignId == null) {
            return BigDecimal.ZERO;
        }

        LocalDateTime now = timeProvider.now();
        List<Period> periodList = singletonList(new Period(
                "week", now.minus(daysInWeek, ChronoUnit.DAYS).toLocalDate(), now.toLocalDate()));

        // Пытаемся найти предвычисленные значения в MySQL таблице
        final Optional<ClientSpent> clientSpent = currencyCode == CurrencyCode.RUB
                ? clientSpentRepository.getClientSpent(shard, clientId)
                : Optional.empty();

        if (clientSpent.isPresent() && clientSpent.get().getActive28DaysSum() != null) {
            return clientSpent.get().getActive28DaysSum().divide(BigDecimal.valueOf(28), MathContext.DECIMAL128);
        }

        // Иначе идем в статистику
        var spentMoney = getMoneySpentForWallet(shard, walletCampaignId, periodList, currencyCode).get("week");
        if (spentMoney.lessThanOrEqual(Money.valueOf(Currencies.EPSILON, currencyCode))) {
            return BigDecimal.valueOf(0);
        }
        return spentMoney.divide(daysInWeek).bigDecimalValue();
    }

    private Map<String, Money> getMoneySpentForWallet(
            int shard,
            Long walletCampaignId,
            List<Period> periodList,
            CurrencyCode currencyCode) {

        var campaigns = campaignRepository
                .getCampaignsByWalletIds(shard, singletonList(walletCampaignId), false);

        var orderIds = mapList(campaigns, WalletCampaign::getOrderId);
        return orderStatService.getOrdersSumSpent(orderIds, periodList, currencyCode);
    }

    BigDecimal getPaymentSumOffer(
            int shard,
            ClientId clientId,
            Long uid,
            Currency currency,
            @Nullable Long walletId) {

        UserOptions options = null;
        try {
            options = userRepository.getUserOptions(shard, uid);
        } catch (RuntimeException e) {
            LOGGER.error("Error on reading user options", e);
        }

        if (options != null && options.paymentSum != null && options.paymentSum > 0) {
            var walletCampaigns = campaignRepository.getWalletsWithCampaignsByWalletCampaignIds(
                    shard, singletonList(walletId), true).getPair(walletId);
            boolean changeRecurringPaymentOffer = featureService.isEnabledForClientId(
                    clientId, FeatureName.CHANGE_RECURRING_PAYMENT_SUM_OFFER);

            if (changeRecurringPaymentOffer && walletCampaigns != null) {
                var wallet = walletCampaigns.first;

                if (wallet.getStrategy().getDayBudget().compareTo(BigDecimal.ZERO) > 0) {
                    return wallet.getStrategy().getDayBudget().multiply(BigDecimal.valueOf(7));
                }

                return StreamEx.of(walletCampaigns.second)
                        .filter(campaign -> campaign.getStrategy().getAutobudget() == CampaignsAutobudget.YES)
                        .map(campaign -> campaign.getStrategy().getStrategyData().getSum())
                        .nonNull()
                        .max(BigDecimal::compareTo)
                        .orElse(BigDecimal.valueOf(options.paymentSum));
            }

        }

        boolean changeFirstPaymentMinSumRecommendation = featureService.isEnabledForClientId(
                clientId, FeatureName.CHANGE_FIRST_PAYMENT_MIN_SUM_RECOMMENDATION);

        if (changeFirstPaymentMinSumRecommendation && currency.equals(CurrencyRub.getInstance())) {
            return BigDecimal.valueOf(OVERWRITTEN_RECOMMENDATION_SUM_MIN_RUB);
        }

        return currency.getRecommendationSumMid();
    }
}
