package ru.yandex.direct.grid.processing.service.payment;

import java.util.Locale;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.balance.client.exception.BalanceClientException;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.payment.model.AutopayParams;
import ru.yandex.direct.core.entity.payment.model.AutopaySettingsPaymethodType;
import ru.yandex.direct.core.entity.payment.model.PaymentParams;
import ru.yandex.direct.core.entity.payment.service.PaymentService;
import ru.yandex.direct.core.entity.payment.service.PaymentValidationService;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeHelper;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeValidationContainer;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.currencies.CurrencyRub;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.exception.GdExceptions;
import ru.yandex.direct.grid.processing.exception.GridPublicException;
import ru.yandex.direct.grid.processing.model.payment.GdGetAutopaySettingsPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetCardBindingUrl;
import ru.yandex.direct.grid.processing.model.payment.GdGetCardBindingUrlPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetClientPaymentCardsPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetPaymentFormDataPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetPaymentFormUrl;
import ru.yandex.direct.grid.processing.model.payment.GdGetPaymentFormUrlPayload;
import ru.yandex.direct.grid.processing.model.payment.GdGetPaymentLinks;
import ru.yandex.direct.grid.processing.model.payment.GdGetPaymentLinksPayload;
import ru.yandex.direct.grid.processing.model.payment.GdPayForAll;
import ru.yandex.direct.grid.processing.model.payment.GdPayForAllPayload;
import ru.yandex.direct.grid.processing.model.payment.GdRemoveAutopayPayload;
import ru.yandex.direct.grid.processing.model.payment.GdSetUpAutopay;
import ru.yandex.direct.grid.processing.model.payment.GdSetUpAutopayPayload;
import ru.yandex.direct.grid.processing.model.payment.GdSetUpDayBudget;
import ru.yandex.direct.grid.processing.model.payment.GdSetUpDayBudgetPayload;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.rbac.RbacRole;

import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.UNDER_WALLET;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class PaymentGraphQlService {

    private static final Logger LOGGER = LoggerFactory.getLogger(PaymentGraphQlService.class);

    private final PaymentDataService paymentDataService;
    private final WalletRepository walletRepository;
    private final ShardHelper shardHelper;
    private final BalanceService balanceService;
    private final ClientService clientService;
    private final PromocodeHelper promocodeHelper;
    private final FeatureService featureService;
    private final PaymentValidationService paymentValidationService;
    private final PaymentService paymentService;
    private final UserService userService;

    /*
     Only RUB is currently supported for new payment
     */
    private final Currency supportedCurrency = CurrencyRub.getInstance();

    private static final Set<CampaignType> CAMPAIGN_TYPES = ImmutableSet.<CampaignType>builder()
            .addAll(UNDER_WALLET)
            .add(CampaignType.WALLET)
            .build();


    @Autowired
    public PaymentGraphQlService(
        PaymentDataService paymentDataService,
        WalletRepository walletRepository,
        ShardHelper shardHelper,
        BalanceService balanceService,
        ClientService clientService,
        PromocodeHelper promocodeHelper,
        FeatureService featureService,
        PaymentValidationService paymentValidationService,
        PaymentService paymentService,
        UserService userService
    ) {
        this.paymentDataService = paymentDataService;
        this.walletRepository = walletRepository;
        this.shardHelper = shardHelper;
        this.balanceService = balanceService;
        this.clientService = clientService;
        this.promocodeHelper = promocodeHelper;
        this.featureService = featureService;
        this.paymentValidationService = paymentValidationService;
        this.paymentService = paymentService;
        this.userService = userService;
    }

    /**
     * Получение информации о настройках автопополнения
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "getAutopaySettings")
    public GdGetAutopaySettingsPayload getAutopaySettings(
            @GraphQLRootContext GridGraphQLContext context) {
        return paymentDataService.getAutopaySettings(context.getSubjectUser().getClientId());
    }

    /**
     * Удаление настроек автопополнения
     */
    @GraphQLNonNull
    @GraphQLMutation(name = "removeAutopay")
    public GdRemoveAutopayPayload removeAutopay(
            @GraphQLRootContext GridGraphQLContext context) {
        paymentDataService.removeAutopay(context.getSubjectUser());
        return new GdRemoveAutopayPayload();
    }

    /**
     * Ручка для настройки автопополнения
     */
    @GraphQLNonNull
    @GraphQLMutation(name = "setUpAutopay")
    public GdSetUpAutopayPayload setUpAutopay(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSetUpAutopay input) {
        var params = new PaymentService.SetUpAutopayParams();
        params.isLegalPerson = input.getIsLegalPerson();
        params.paymentSum = input.getPaymentSum();
        params.isMobile = input.getIsMobile();
        params.remainingSum = input.getRemainingSum();
        params.cardId = input.getCardId();

        var result = paymentService.setUpAutopay(context.getSubjectUser(), context.getOperator().getUid(), params);

        var gridValidationResult = new GridValidationResultConversionService()
                .buildGridValidationResult(result.validationResult);

        return new GdSetUpAutopayPayload()
            .withBindingUrl(result.bindingUrl)
            .withValidationResult(gridValidationResult);
    }


    /**
     * Ручка для настройки дневного бюджета
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "setUpDayBudget")
    public GdSetUpDayBudgetPayload setUpDayBudget(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSetUpDayBudget input) {

        var subjectUser = context.getSubjectUser();
        var operatorUid = context.getOperator().getUid();

        var result = paymentDataService.setUpDayBudget(
                subjectUser, input, operatorUid);

        return new GdSetUpDayBudgetPayload().withValidationResult(
            new GridValidationResultConversionService().buildGridValidationResult(result.getResult())
        );
    }

    /**
     * Получает информацию о банковских картах клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "getClientPaymentCards")
    public GdGetClientPaymentCardsPayload getClientPaymentCards(
            @GraphQLRootContext GridGraphQLContext context) {
        checkCardsAccess(context);
        User operator = context.getOperator();
        return paymentDataService.getClientPaymentCards(operator.getUid(), operator.getClientId());
    }

    /**
     * Получает информацию о редиректе в баланс
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "payForAll")
    public GdPayForAllPayload payForAll(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdPayForAll input) {
        Long uid = context.getSubjectUser().getUid();
        RbacRole role = context.getOperator().getRole();
        Long operatorUid = context.getOperator().getUid();

        Locale locale = HttpUtil.getCurrentLocale().orElse(new Locale("ru"));

        return paymentDataService.payForAll(
            uid,
            operatorUid,
            role,
            context.getSubjectUser().getLogin(),
            input,
            HttpUtil.getRequest().getRequestURL().toString(),
            locale
        );
    }

    /**
     * Получение ссылки на страницу оплаты.
     * Ручка создает плательщика, если его еще нет.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "getPaymentFormUrl")
    public GdGetPaymentFormUrlPayload getPaymentFormUrl(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetPaymentFormUrl input) {
        return paymentDataService.getPaymentFormUrl(context.getSubjectUser(), context.getOperator().getUid(), input);
    }

    /**
     * Получение ссылки на страницу привязки карты (без оплаты).
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "getCardBindingUrl")
    public GdGetCardBindingUrlPayload getCardBindingUrl(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetCardBindingUrl input) {
        return paymentDataService.getCardBindingUrl(context.getOperator().getUid(),
                context.getSubjectUser().getClientId(), input);
    }

    /**
     * Флаг возможности оплаты от юрлица.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "canPayAsLegalEntity")
    public boolean canPayAsLegalEntity(@GraphQLRootContext GridGraphQLContext context) {
        var subjectUser = context.getSubjectUser();

        try {
            return balanceService.getOrCreatePerson(subjectUser, context.getOperator().getUid(), true) != null;
        } catch (BalanceClientException e) {
            // test balance throws here for some clients on TC (since we have different clientIds for prod and test)
            LOGGER.warn("Failed to get person from balance: " + e.getMessage());
            return false;
        }
    }

    /**
     * Получение данных для отображения формы оплаты.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "getPaymentFormData")
    public GdGetPaymentFormDataPayload getPaymentFormData(
            @GraphQLRootContext GridGraphQLContext context) {
        var subjectUser = context.getSubjectUser();
        var clientId = subjectUser.getClientId();
        var uid = subjectUser.getUid();

        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        var currency = clientService.getWorkCurrency(clientId);
        var walletId = walletRepository.getActualClientWalletId(shard, clientId, currency.getCode());

        return new GdGetPaymentFormDataPayload()
                .withAverageDailyBudgetStat(
                        paymentDataService.getAverageDailyBudget(shard, currency.getCode(), clientId, walletId))
                .withPaymentSumOffer(
                        paymentDataService.getPaymentSumOffer(shard, clientId, uid, currency, walletId))
                .withDaysSinceMoneyOut(paymentDataService.getDaysSinceMoneyOut(shard, walletId, clientId));
    }

    @GraphQLNonNull
    @GraphQLQuery(
            name = "getPaymentLinks",
            description = "Возвращает paymentUrl, в случае если запрошен автоплатеж и нет привязанных карт - возвращает bindingUrl"
    )
    public GdGetPaymentLinksPayload getPaymentLinks(@GraphQLRootContext GridGraphQLContext context,
                                  @GraphQLNonNull @GraphQLArgument(name = "input") GdGetPaymentLinks input) {
        User subjectUser = context.getSubjectUser();
        Long operatorUid = context.getOperator().getUid();
        Long subjectUserUid = subjectUser.getUid();

        CurrencyCode clientCurrency = clientService.getWorkCurrency(subjectUser.getClientId()).getCode();

        PromocodeValidationContainer promocodeValidationContainer = promocodeHelper
                .preparePromocodeValidationContainer(subjectUser.getClientId());
        boolean useExtendedCodes = featureService.isEnabledForClientId(subjectUser.getClientId(),
                FeatureName.USE_EXTENDED_ERROR_CODES_FOR_PROMOCODES_ERRORS);

        if (subjectUserUid.equals(operatorUid)) {
            userService.setOfferAccepted(subjectUserUid);
        }

        var paymentParams = new PaymentParams()
                .withSum(input.getSum())
                .withPaymentSum(input.getPaymentSum())
                .withRemainingSum(input.getRemainingSum())
                .withPromocode(input.getPromocode())
                .withIsMobile(input.getIsMobile());

        var validationResult = paymentValidationService.validate(
                paymentParams,
                clientCurrency,
                promocodeValidationContainer,
                useExtendedCodes);

        if (validationResult.hasAnyErrors()) {
            LOGGER.error("errors in web payment validation: " + validationResult.flattenErrors().get(0).toString());

            var gridValidationResult = new GridValidationResultConversionService()
                    .buildGridValidationResult(validationResult);

            return new GdGetPaymentLinksPayload()
                    .withValidationResult(gridValidationResult);
        }

        AutopayParams autopayParams = new AutopayParams()
                .withPaymentSum(input.getPaymentSum())
                .withRemainingSum(input.getRemainingSum())
                .withCardId(input.getCardId())
                .withPaymentType(AutopaySettingsPaymethodType.CARD);

        var result = paymentService.enableAutopayIfPossibleAndGetUrls(subjectUser, operatorUid, clientCurrency,
                input.getSum(), input.getIsLegal(), autopayParams,
                input.getPromocode(), promocodeValidationContainer, input.getIsMobile(), null);

        var gridValidationResult = new GridValidationResultConversionService()
                .buildGridValidationResult(result.validation);

        return new GdGetPaymentLinksPayload()
                .withValidationResult(gridValidationResult)
                .withBindingUrl(result.cardBindingUrl)
                .withPaymentUrl(result.paymentUrl)
                .withAmount(result.amount);
    }

    private void checkCardsAccess(@GraphQLRootContext GridGraphQLContext context) {
        User operator = context.getOperator();

        if (!operator.getUid().equals(context.getSubjectUser().getUid())) {
            throw new GridPublicException(GdExceptions.ACCESS_DENIED, "Operator can't get user cards",
                    SecurityTranslations.INSTANCE.accessDenied());
        }
    }

}
