package ru.yandex.direct.grid.processing.service.placement;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.placements.model1.BlockSize;
import ru.yandex.direct.core.entity.placements.model1.IndoorBlock;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementFormat;
import ru.yandex.direct.core.entity.placements.model1.PlacementPhoto;
import ru.yandex.direct.core.entity.placements.model1.PlacementsFilter;
import ru.yandex.direct.core.entity.region.RegionDesc;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictEntryWithLongKey;
import ru.yandex.direct.grid.processing.model.inventori.GdBlockSize;
import ru.yandex.direct.grid.processing.model.placement.GdIndoorPlacementBlock;
import ru.yandex.direct.grid.processing.model.placement.GdOutdoorPlacementBlock;
import ru.yandex.direct.grid.processing.model.placement.GdPiPage;
import ru.yandex.direct.grid.processing.model.placement.GdPlacement;
import ru.yandex.direct.grid.processing.model.placement.GdPlacementBlock;
import ru.yandex.direct.grid.processing.model.placement.GdPlacementContext;
import ru.yandex.direct.grid.processing.model.placement.GdPlacementFilter;
import ru.yandex.direct.grid.processing.model.placement.GdPlacementFormat;
import ru.yandex.direct.grid.processing.model.placement.GdPlacementPhoto;
import ru.yandex.direct.grid.processing.model.placement.GdRegionDesc;
import ru.yandex.direct.grid.processing.service.campaign.RegionDescriptionLocalizer;
import ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.localization.LocalizationMapper;
import ru.yandex.direct.web.core.entity.placement.model.PlacementsResponse;
import ru.yandex.direct.web.core.entity.placement.service.PlacementsService;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class PlacementDataService {

    private static final BiFunction<Integer, String, GdStringDictEntryWithLongKey> INTEGER_STRING_MAP_2_GD_STRING_DICT_ENTRY_WITH_LONG_KEY =
            (k, v) -> new GdStringDictEntryWithLongKey().withKey(k.longValue()).withValue(v);
    private final PlacementsService placementService;
    private final AdGroupRepository adGroupRepository;
    private final RegionDescriptionLocalizer localizer;
    private final ShardHelper shardHelpler;

    private static final LocalizationMapper<Map<Language, String>, StringHolder> addressTranslationMapper =
            LocalizationMapper.builder()
                    .addEnTranslation((Map<Language, String> map) -> map.get(Language.EN))
                    .addRuTranslation(map -> map.get(Language.RU))
                    .addUaTranslation(map -> map.get(Language.UK))
                    .addTrTranslation(map -> map.get(Language.TR))
                    .translateTo(StringHolder::setItem)
                    .createBy(StringHolder::new)
                    .withDefaultLanguage(Language.EN)
                    .build();

    @Autowired
    public PlacementDataService(PlacementsService placementService, AdGroupRepository adGroupRepository,
                                RegionDescriptionLocalizer localizer,
                                ShardHelper shardHelpler) {
        this.placementService = placementService;
        this.adGroupRepository = adGroupRepository;
        this.localizer = localizer;
        this.shardHelpler = shardHelpler;
    }

    public GdPlacementContext getPlacements(ClientId clientId, @GraphQLNonNull GdPlacementFilter input) {

        if (!isEmpty(input.getAdGroupIds())) {
            int shard = shardHelpler.getShardByClientId(clientId);
            Set<Long> pageIds = adGroupRepository.getAdGroupsPageTargetByAdGroupId(shard, input.getAdGroupIds()).values().stream()
                    .flatMap(Collection::stream)
                    .map(PageBlock::getPageId)
                    .collect(Collectors.toSet());
            input.setPageIds(new ArrayList<>(pageIds));
        }

        PlacementsResponse response = placementService.getPlacements(toPlacementsFilter(input), clientId);

        List<GdPlacement> placements = mapList(response.getPlacements(), this::toGdPlacement);
        List<GdPlacementBlock> placementBlocks = mapList(response.getPlacementBlocks(), this::toGdPlacementBlock);

        List<GdRegionDesc> regionDictionary =
                EntryStream.of(nvl(response.getRegionDictionary(), Collections.emptyMap())).values()
                        .map(this::toGdRegionDesc)
                        .toList();

        List<GdStringDictEntryWithLongKey> facilityDictionary = EntryStream.of(nvl(response.getFacilityDictionary(), Collections.emptyMap()))
                .mapKeyValue(INTEGER_STRING_MAP_2_GD_STRING_DICT_ENTRY_WITH_LONG_KEY).toList();

        List<GdStringDictEntryWithLongKey> zoneDictionary = EntryStream.of(nvl(response.getZoneDictionary(), Collections.emptyMap()))
                .mapKeyValue(INTEGER_STRING_MAP_2_GD_STRING_DICT_ENTRY_WITH_LONG_KEY).toList();

        return new GdPlacementContext()
                .withPlacements(placements)
                .withPlacementBlocks(placementBlocks)
                .withFacilityDictionary(facilityDictionary)
                .withRegionDictionary(regionDictionary)
                .withZoneDictionary(zoneDictionary);
    }

    private PlacementsFilter toPlacementsFilter(GdPlacementFilter input) {
        PlacementsFilter filter = new PlacementsFilter();
        filter.setFacilityType(input.getFacilityType());
        filter.setZoneCategory(input.getZoneCategory());
        filter.setPlacementType(input.getPlacementType());
        filter.setPageIds(input.getPageIds());
        filter.setGeoId(input.getGeoId());
        return filter;
    }

    private GdPlacement toGdPlacement(Placement p) {
        return new GdPlacement()
                .withId(p.getId())
                .withType(p.getType())
                .withDomain(p.getDomain())
                .withCaption(p.getCaption())
                .withLogin(p.getLogin())
                .withOperatorName(p.getOperatorName())
                .withIsYandexPage(p.isYandexPage())
                .withIsDeleted(p.isDeleted())
                .withIsTesting(p.isTesting())
                .withBlocks(mapList(p.getBlocks(), b -> toGdPlacementBlock(b)));
    }



    private GdRegionDesc toGdRegionDesc(RegionDesc rd) {
        GdRegionDesc localized = localizer.localize(rd);

        return localized
                .withNameEn(rd.getNameEn())
                .withNameTr(rd.getNameTr())
                .withNameUa(rd.getNameUa())
                .withNameRu(rd.getNameRu());
    }

    private GdPlacementBlock toGdPlacementBlock(Object block) {
        boolean isIndoor = block instanceof IndoorBlock;
        boolean isOutdoor = block instanceof OutdoorBlock;
        if (isIndoor) {
            IndoorBlock indoorBlock = (IndoorBlock) block;
            return new GdIndoorPlacementBlock()
                    .withPageId(indoorBlock.getPageId())
                    .withBlockId(indoorBlock.getBlockId())
                    .withBlockCaption(indoorBlock.getBlockCaption())
                    .withLastChange(indoorBlock.getLastChange())
                    .withIsDeleted(indoorBlock.isDeleted())
                    .withSizes(new HashSet<>(mapList(indoorBlock.getSizes(), this::toGdBlockSize)))
                    .withGeoId(indoorBlock.getGeoId())
                    .withAddress(localizeBlockAddress(indoorBlock.getAddressTranslations(), indoorBlock.getAddress()))
                    .withCoordinates(indoorBlock.getCoordinates())
                    .withResolution(toGdBlockSize(indoorBlock.getResolution()))
                    .withFacilityType(indoorBlock.getFacilityType())
                    .withZoneCategory(indoorBlock.getZoneCategory())
                    .withAspectRatio(indoorBlock.getAspectRatio())
                    .withPhotos(mapList(indoorBlock.getPhotos(), this::toGdPlacementPhoto))
                    .withHidden(nvl(indoorBlock.getHidden(),false));  // убрать этот кастыль про false когда ПИ сделают свой тикет https://st.yandex-team.ru/PI-17398
        } else if (isOutdoor) {
            OutdoorBlock outdoorBlock = (OutdoorBlock) block;
            return new GdOutdoorPlacementBlock()
                    .withPageId(outdoorBlock.getPageId())
                    .withBlockId(outdoorBlock.getBlockId())
                    .withBlockCaption(outdoorBlock.getBlockCaption())
                    .withLastChange(outdoorBlock.getLastChange())
                    .withIsDeleted(outdoorBlock.isDeleted())
                    .withSizes(new HashSet<>(mapList(outdoorBlock.getSizes(), this::toGdBlockSize)))
                    .withGeoId(outdoorBlock.getGeoId())
                    .withAddress(localizeBlockAddress(outdoorBlock.getAddressTranslations(), outdoorBlock.getAddress()))
                    .withCoordinates(outdoorBlock.getCoordinates())
                    .withResolution(toGdBlockSize(outdoorBlock.getResolution()))
                    .withFacilityType(outdoorBlock.getFacilityType())
                    .withDirection(outdoorBlock.getDirection())
                    .withWidth(outdoorBlock.getWidth())
                    .withHeight(outdoorBlock.getHeight())
                    .withDuration(outdoorBlock.getDuration())
                    .withPhotos(mapList(outdoorBlock.getPhotos(), this::toGdPlacementPhoto))
                    .withHidden(outdoorBlock.getHidden());
        }
        throw new IllegalArgumentException("Unexpected (none of [IndoorBlock, OutdoorBlock]) class of placement block:" + block.getClass());
    }

    private String localizeBlockAddress(@Nullable Map<Language, String> translations, String defaultValue) {
        Map<Language, String> translationMap = new HashMap<>(
                Optional
                        .ofNullable(translations)
                        .orElse(Collections.emptyMap())
        );

        translationMap.put(Language.RU, defaultValue);
        Locale locale = LocaleContextHolder.getLocale();
        return addressTranslationMapper.localize(translationMap, locale, defaultValue).getItem();
    }


    private GdPlacementPhoto toGdPlacementPhoto(PlacementPhoto p) {
        return new GdPlacementPhoto()
                .withFormats(mapList(p.getFormats(), this::toGdPlacementFormat));
    }

    private GdPlacementFormat toGdPlacementFormat(PlacementFormat f) {
        return new GdPlacementFormat()
                .withHeight(f.getHeight())
                .withWidth(f.getWidth())
                .withPath(f.getPath());
    }

    private GdBlockSize toGdBlockSize(BlockSize blockSize) {
        return new GdBlockSize()
                .withHeight(blockSize.getHeight())
                .withWidth(blockSize.getWidth());
    }

    private static final class StringHolder {
        private String item;

        public String getItem() {
            return item;
        }

        public void setItem(String item) {
            this.item = item;
        }
    }

    public List<GdPiPage> findPlacements(ClientId clientId, String filterString) {
        var placements = placementService.findPlacements(filterString, clientId);
        return mapList(placements, PricePackageDataConverter::toGdPiPage);
    }

    public List<GdPiPage> getPlacementsByIds(List<Long> ids) {
        var placements = placementService.getPlacementsByIds(ids);
        return mapList(placements, PricePackageDataConverter::toGdPiPage);
    }
}
