package ru.yandex.direct.grid.processing.service.placement;

import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.placement.GdFindPlacementFilter;
import ru.yandex.direct.grid.processing.model.placement.GdGetPlacementFilter;
import ru.yandex.direct.grid.processing.model.placement.GdPiPage;
import ru.yandex.direct.grid.processing.model.placement.GdPlacementContext;
import ru.yandex.direct.grid.processing.model.placement.GdPlacementFilter;
import ru.yandex.direct.rbac.RbacRole;

import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;

/**
 * Сервис, возвращающий данные о рекламных блоках
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class PlacementGraphQlService {
    private static final String PLACEMENTS_RESOLVER_NAME = "placements";
    private static final String PLACEMENTS_SUGGEST_RESOLVER_NAME = "findPlacements";
    private static final String GET_PLACEMENT_SUGGEST_RESOLVER_NAME = "getPlacements";

    private final PlacementDataService placementDataService;

    @Autowired
    public PlacementGraphQlService(PlacementDataService productDataService) {
        this.placementDataService = productDataService;
    }

    /**
     * GraphQL подзапрос. Получает справочник рекламных блоков
     *
     * @param input параметры фильтра и вывода
     */
    @GraphQLNonNull
    @GraphQLQuery(name = PLACEMENTS_RESOLVER_NAME)
    public GdPlacementContext getPlacements(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdPlacementFilter input) {
        return placementDataService.getPlacements(context.getSubjectUser().getClientId(), input);
    }

    /**
     * GraphQL подзапрос. Отдаёт список пейджей под условие поиска для саджеста.
     *
     * @param input параметры фильтра
     */
    @GraphQLNonNull
    @GraphQLQuery(name = PLACEMENTS_SUGGEST_RESOLVER_NAME)
    public List<GdPiPage> findPlacements(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdFindPlacementFilter input) {

        if (!hasOneOfRoles(context.getOperator(), RbacRole.SUPER, RbacRole.SUPERREADER,
                RbacRole.MANAGER, RbacRole.SUPPORT)) {
            return Collections.emptyList();
        }

        return placementDataService.findPlacements(context.getSubjectUser().getClientId(), input.getDomain());
    }

    /**
     * GraphQL подзапрос. Отдаёт пейдж по id для саджеста.
     *
     * @param input параметры фильтра
     */
    @GraphQLNonNull
    @GraphQLQuery(name = GET_PLACEMENT_SUGGEST_RESOLVER_NAME)
    public List<GdPiPage> getPlacements(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetPlacementFilter input) {
        return placementDataService.getPlacementsByIds(input.getIds());
    }
}
