package ru.yandex.direct.grid.processing.service.pricepackage;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.markupcondition.repository.MarkupConditionRepository;
import ru.yandex.direct.core.entity.pricepackage.model.MarkupCondition;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetMarkupConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetMarkupConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdMarkupConditionsOperator;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddMarkupConditionPayloadItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddMarkupConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddMarkupConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeleteMarkupConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeleteMarkupConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdateMarkupConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdateMarkupConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdateMarkupConditionsPayloadItem;
import ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter;

import static ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter.toCoreMarkupConditionTargeting;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class MarkupConditionDataService {

    private final MarkupConditionRepository markupConditionRepository;

    @Autowired
    public MarkupConditionDataService(
            MarkupConditionRepository markupConditionRepository) {
        this.markupConditionRepository = markupConditionRepository;
    }

    public GdGetMarkupConditionsPayload getMarkupConditions(GdGetMarkupConditions input) {
        List<MarkupCondition> allMarkupConditions = markupConditionRepository.getAllMarkupConditions();
        return new GdGetMarkupConditionsPayload()
                .withTotalCount(allMarkupConditions.size())
                .withRowset(PricePackageDataConverter.toGdMarkupConditions(allMarkupConditions));
    }

    public GdAddMarkupConditionsPayload addMarkupConditions(GdAddMarkupConditions input) {
        List<MarkupCondition> markupConditions = mapList(input.getAddItems(), item -> new MarkupCondition()
                .withIsArchived(false)
                .withGroup(item.getGroup())
                .withPriority(item.getPriority())
                .withOperator(GdMarkupConditionsOperator.toSource(item.getOperator()))
                .withDescription(item.getDescription())
                .withTargeting(toCoreMarkupConditionTargeting(item.getTargeting())));
        List<Long> conditionIds = markupConditionRepository.addMarkupConditions(markupConditions);
        return new GdAddMarkupConditionsPayload()
                .withAddedItems(mapList(conditionIds, id -> new GdAddMarkupConditionPayloadItem().withId(id)));
    }

    public GdUpdateMarkupConditionsPayload updateMarkupConditions(GdUpdateMarkupConditions input) {
        List<MarkupCondition> markupConditions = mapList(input.getUpdateItems(), item -> new MarkupCondition()
                .withId(item.getId())
                .withGroup(item.getGroup())
                .withPriority(item.getPriority())
                .withIsArchived(item.getIsArchived())
                .withOperator(GdMarkupConditionsOperator.toSource(item.getOperator()))
                .withDescription(item.getDescription())
                .withTargeting(toCoreMarkupConditionTargeting(item.getTargeting())));

        markupConditionRepository.updateMarkupConditions(markupConditions);

        return new GdUpdateMarkupConditionsPayload()
                .withUpdatedItems(mapList(input.getUpdateItems(), item -> new GdUpdateMarkupConditionsPayloadItem()
                        .withId(item.getId())));
    }

    public GdDeleteMarkupConditionsPayload deleteMarkupConditions(GdDeleteMarkupConditions input) {
        markupConditionRepository.deleteMarkupConditions(input.getConditionIds());
        return new GdDeleteMarkupConditionsPayload().withDeletedConditionIds(input.getConditionIds());
    }
}
