package ru.yandex.direct.grid.processing.service.pricepackage;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.canvas.client.CanvasClient;
import ru.yandex.direct.canvas.client.model.video.PresetResponse;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.SortOrder;
import ru.yandex.direct.core.entity.creative.repository.BannerStorageDictRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.repository.PlacementRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCategory;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForClient;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageOrderBy;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageOrderByField;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackagesFilter;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsFixed;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository.PricePackagesWithTotalCount;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageAddOperation;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageAddOperationFactory;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageDeleteOperation;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageDeleteOperationFactory;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageUpdateOperationFactory;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeType;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackageCategoriesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesAvailableForClientPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesForClient;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackage;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageForClient;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackagesCreativeTemplatesItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackagePayloadItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackagesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeletePricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeletePricePackagesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdMutationTargetingsCustom;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdMutationTargetingsFixed;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdPricePackageClientInput;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackagesItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackagesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackagesPayloadItem;
import ru.yandex.direct.grid.processing.service.campaign.RegionDescriptionLocalizer;
import ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter;
import ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverterContext;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.web.core.entity.inventori.service.CryptaService;
import ru.yandex.direct.web.core.model.retargeting.CryptaGoalWeb;
import ru.yandex.direct.web.core.model.retargeting.WebGoalType;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.creative.repository.CreativeMappings.convertVideoType;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toGdCreativeType;
import static ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter.toGdPricePackagesHtml5PresetsCreativeTemplatesItem;
import static ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter.toGdPricePackagesVideoPresetsCreativeTemplatesItem;
import static ru.yandex.direct.grid.processing.util.GeoTreeUtils.geoToFrontendGeo;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.operation.Applicability.PARTIAL;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class PricePackageDataService {

    private final PricePackageAddOperationFactory addOperationFactory;
    private final PricePackageDeleteOperationFactory deleteOperationFactory;
    private final PricePackageUpdateOperationFactory updateOperationFactory;
    private final PricePackageRepository repository;
    private final GridValidationService gridValidationService;
    private final PricePackageService pricePackageService;
    private final RegionDescriptionLocalizer localizer;
    private final ShardHelper shardHelper;
    private final BannerStorageDictRepository bannerStorageDictRepository;
    private final CanvasClient canvasClient;
    private final CryptaService cryptaService;
    private final PlacementRepository placementRepository;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final FeatureService featureService;
    private final PpcProperty<List<Long>> bsTemplateWhiteListProp;

    @Autowired
    public PricePackageDataService(PricePackageAddOperationFactory addOperationFactory,
                                   PricePackageDeleteOperationFactory deleteOperationFactory,
                                   PricePackageUpdateOperationFactory updateOperationFactory,
                                   PricePackageRepository repository,
                                   GridValidationService gridValidationService,
                                   PricePackageService pricePackageService,
                                   RegionDescriptionLocalizer localizer,
                                   ShardHelper shardHelper,
                                   BannerStorageDictRepository bannerStorageDictRepository,
                                   CanvasClient canvasClient, CryptaService cryptaService,
                                   PlacementRepository placementRepository,
                                   SspPlatformsRepository sspPlatformsRepository,
                                   FeatureService featureService,
                                   PpcPropertiesSupport ppcPropertiesSupport) {
        this.addOperationFactory = addOperationFactory;
        this.deleteOperationFactory = deleteOperationFactory;
        this.updateOperationFactory = updateOperationFactory;
        this.repository = repository;
        this.gridValidationService = gridValidationService;
        this.pricePackageService = pricePackageService;
        this.localizer = localizer;
        this.shardHelper = shardHelper;
        this.bannerStorageDictRepository = bannerStorageDictRepository;
        this.canvasClient = canvasClient;
        this.cryptaService = cryptaService;
        this.placementRepository = placementRepository;
        this.sspPlatformsRepository = sspPlatformsRepository;
        this.featureService = featureService;
        bsTemplateWhiteListProp = ppcPropertiesSupport.get(PpcPropertyNames.PRICE_PACKAGE_BS_TEMPLATE_WHITE_LIST,
                Duration.ofMinutes(2));
    }


    public GdAddPricePackagesPayload addPricePackages(GdAddPricePackages input, GridGraphQLContext context) {
        var converter = new PricePackageDataConverter(createPricePackageDataConverterContext());
        List<PricePackage> pricePackagesToAdd = converter.toCorePricePackages(input);
        PricePackageAddOperation addOperation = addOperationFactory.newInstance(
                PARTIAL, pricePackagesToAdd, context.getOperator());
        MassResult<Long> result = addOperation.prepareAndApply();

        GdValidationResult validationResult = getGdValidationResult(result, GdAddPricePackages.ADD_ITEMS);
        List<GdAddPricePackagePayloadItem> addedItems =
                getResults(result, PricePackageDataConverter::toGdAddPricePackagePayloadItem);
        return new GdAddPricePackagesPayload()
                .withAddedItems(addedItems)
                .withValidationResult(validationResult);
    }

    public GdGetPricePackagesPayload getPricePackages(GdGetPricePackages input) {
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());

        PricePackageOrderBy orderBy = (input.getOrderBy() == null || input.getOrderBy().size() == 0) ?
                new PricePackageOrderBy()
                        .withField(PricePackageOrderByField.PACKAGE_ID)
                        .withOrder(SortOrder.ASC) :
                PricePackageDataConverter.toCorePricePackageOrderBy(input.getOrderBy().get(0));

        PricePackagesWithTotalCount pricePackagesWithTotalCount = repository.getPricePackages(
                PricePackageDataConverter.toCorePricePackageFilter(input.getFilter()),
                orderBy, range);


        List<GdPricePackage> rowset = toGdPricePackages(pricePackagesWithTotalCount.getPricePackages(),
                pagesToPlacements(mapList(pricePackagesWithTotalCount.getPricePackages(),
                        PricePackageForClient.class::cast)));

        return new GdGetPricePackagesPayload()
                .withRowset(rowset)
                .withTotalCount(pricePackagesWithTotalCount.getTotalCount());
    }

    private Map<Long, Placement> pagesToPlacements(List<PricePackageForClient> pricePackages) {
        List<Long> pageIds =
                pricePackages.stream()
                        .filter(e -> e.getAllowedPageIds() != null)
                        .flatMap(e -> e.getAllowedPageIds().stream())
                        .collect(Collectors.toList());

        return placementRepository.getPlacements(pageIds);
    }

    List<GdPricePackage> toGdPricePackages(List<PricePackage> pricePackages, Map<Long, Placement> pagesToPlacements) {
        return PricePackageDataConverter.toGdPricePackages(pricePackages, pagesToPlacements,
                geo -> geoToFrontendGeo(geo, pricePackageService.getGeoTree(), localizer),
                cryptaSegmentTypeSupplier(pricePackages.stream()
                        .flatMap(it -> extractCryptaSegmentIds(it.getTargetingsCustom(),
                                it.getTargetingsFixed()).stream())
                        .collect(Collectors.toSet())
                ),
                creativeTemplatesSupplier());
    }

    GdPricePackageForClient toGdPricePackageForClient(PricePackageForClient pricePackageForClient, Long operatorUid) {
        List<Long> pageIds = nvl(pricePackageForClient.getAllowedPageIds(), emptyList());
        Map<Long, Placement> pagesToPlacements = placementRepository.getPlacements(pageIds);
        boolean isBrandLiftHiddenEnabled = featureService.isEnabled(operatorUid, FeatureName.BRAND_LIFT_HIDDEN);
        return PricePackageDataConverter.toGdPricePackageForClient(pricePackageForClient,
                pagesToPlacements,
                geo -> geoToFrontendGeo(geo, pricePackageService.getGeoTree(), localizer),
                cryptaSegmentTypeSupplier(extractCryptaSegmentIds(pricePackageForClient.getTargetingsCustom(),
                        pricePackageForClient.getTargetingsFixed())),
                creativeTemplatesSupplier(),
                isBrandLiftHiddenEnabled);
    }

    private static List<Long> extractCryptaSegmentIds(TargetingsCustom targetingsCustom,
                                                      TargetingsFixed targetingsFixed) {
        List<Long> allSegmentIds = new ArrayList<>();
        if (targetingsFixed != null && targetingsFixed.getCryptaSegments() != null) {
            allSegmentIds.addAll(targetingsFixed.getCryptaSegments());
        }
        if (targetingsCustom != null
                && targetingsCustom.getRetargetingCondition() != null
                && targetingsCustom.getRetargetingCondition().getCryptaSegments() != null) {
            allSegmentIds.addAll(targetingsCustom.getRetargetingCondition().getCryptaSegments());
        }
        return allSegmentIds;
    }

    private Function<Long, WebGoalType> cryptaSegmentTypeSupplier(Collection<Long> cryptaSegmentIds) {
        if (cryptaSegmentIds.isEmpty()) {//настройки крипты на заданы, экономим поход в базу
            return id -> null;
        }
        var cryptaSegments = cryptaService.getSegments();
        var map = listToMap(cryptaSegments, CryptaGoalWeb::getId, CryptaGoalWeb::getType);
        return id -> map.get(id);
    }

    GdMutationTargetingsFixed toGdMutationTargetingsFixed(TargetingsFixed targetingsFixed) {
        return PricePackageDataConverter.toGdMutationTargetingsFixed(targetingsFixed,
                geo -> geoToFrontendGeo(geo, pricePackageService.getGeoTree(), localizer));
    }

    GdMutationTargetingsCustom toGdMutationTargetingsCustom(TargetingsCustom targetingsCustom) {
        return PricePackageDataConverter.toGdMutationTargetingsCustom(targetingsCustom,
                geo -> geoToFrontendGeo(geo, pricePackageService.getGeoTree(), localizer));
    }

    public GdGetPricePackagesAvailableForClientPayload getPricePackagesForClient(ClientId clientId,
                                                                                 Long operatorUid,
                                                                                 GdGetPricePackagesForClient input) {
        PricePackagesFilter filter = PricePackageDataConverter.toCorePricePackagesForClientFilter(input.getFilter());
        List<PricePackageForClient> activePricePackagesForClient =
                pricePackageService.getActivePricePackagesForClient(clientId, filter);
        boolean isBrandLiftHiddenEnabled = featureService.isEnabled(operatorUid, FeatureName.BRAND_LIFT_HIDDEN);
        return new GdGetPricePackagesAvailableForClientPayload()
                .withRowset(PricePackageDataConverter.toGdPricePackagesForClient(
                        activePricePackagesForClient, pagesToPlacements(activePricePackagesForClient),
                        geo -> geoToFrontendGeo(geo, pricePackageService.getGeoTree(), localizer),
                        cryptaSegmentTypeSupplier(
                                activePricePackagesForClient.stream()
                                        .flatMap(it -> extractCryptaSegmentIds(
                                                it.getTargetingsCustom(), it.getTargetingsFixed()).stream())
                                        .collect(Collectors.toSet())
                        ),
                        creativeTemplatesSupplier(),
                        isBrandLiftHiddenEnabled));
    }

    public Map<Long, GdPricePackageForClient> getPricePackagesForClient(Collection<Long> pricePackageIds,
                                                                        Long operatorUid) {
        Map<Long, PricePackage> pricePackages = repository.getPricePackages(pricePackageIds);
        return EntryStream.of(pricePackages)
                .mapValues(pricePackage -> toGdPricePackageForClient(pricePackage, operatorUid))
                .toMap();
    }

    public GdGetPricePackageCategoriesPayload getPricePackageCategories() {
        List<PricePackageCategory> pricePackagesCategories = repository.getAllPricePackageCategories();
        return new GdGetPricePackageCategoriesPayload()
                .withRowset(pricePackagesCategories
                        .stream()
                        .map(PricePackageDataConverter::toGdPricePackageCategory)
                        .collect(Collectors.toList()));
    }

    public GdDeletePricePackagesPayload deletePricePackages(GdDeletePricePackages input) {
        PricePackageDeleteOperation deleteOperation = deleteOperationFactory.newInstance(PARTIAL,
                input.getPackageIds());
        MassResult<Long> massResult = deleteOperation.prepareAndApply();
        List<Long> deletedPackageIds = getResults(massResult, r -> r);
        GdValidationResult validationResult = getGdValidationResult(massResult, GdDeletePricePackages.PACKAGE_IDS);
        return new GdDeletePricePackagesPayload()
                .withDeletedPackageIds(deletedPackageIds)
                .withValidationResult(validationResult);
    }

    public GdUpdatePricePackagesPayload updatePricePackages(GdUpdatePricePackages input, User operator) {
        var converter = new PricePackageDataConverter(createPricePackageDataConverterContext());
        Map<String, Long> loginToClientId = getLoginToClientId(input);
        List<ModelChanges<PricePackage>> modelChanges = new ArrayList<>();
        List<LocalDateTime> userTimestamps = new ArrayList<>();
        var packageIdList = input.getUpdateItems().stream()
                .map(GdUpdatePricePackagesItem::getId)
                .collect(Collectors.toList());
        Map<Long, PricePackage> oldPackages = repository.getPricePackages(packageIdList);
        for (GdUpdatePricePackagesItem updateItem : input.getUpdateItems()) {
            modelChanges.add(converter.toCoreModelChange(updateItem, loginToClientId,
                    oldPackages.getOrDefault(updateItem.getId(), null)));
            userTimestamps.add(updateItem.getLastSeenUpdateTime());
        }
        var updateOperation = updateOperationFactory.newInstance(PARTIAL, modelChanges, userTimestamps, operator);

        MassResult<Long> result = updateOperation.prepareAndApply();

        GdValidationResult validationResult = getGdValidationResult(result, GdUpdatePricePackages.UPDATE_ITEMS);
        List<GdUpdatePricePackagesPayloadItem> updatedItems =
                getResults(result, PricePackageDataConverter::toGdUpdatePricePackagesPayloadItem);
        return new GdUpdatePricePackagesPayload()
                .withUpdatedItems(updatedItems)
                .withValidationResult(validationResult);
    }

    private Map<String, Long> getLoginToClientId(GdUpdatePricePackages input) {
        List<String> clientLogins = StreamEx.of(input.getUpdateItems())
                .map(GdUpdatePricePackagesItem::getClients)
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .map(GdPricePackageClientInput::getLogin)
                .filter(Objects::nonNull)
                .distinct()
                .toList();
        return shardHelper.getClientIdsByLogins(clientLogins);
    }

    /**
     * Достать результат валидации из MassResult и сконвертировать в GdValidationResult
     *
     * @param result        - результат операции
     * @param modelProperty - имя modelProperty будет использоваться в качестве префикса для всех путей в валидации
     * @return сконвертированный результат валидации
     */
    private GdValidationResult getGdValidationResult(MassResult result, ModelProperty modelProperty) {
        return gridValidationService.getValidationResult(result, path(field(modelProperty)));
    }

    public HashMap<GdCreativeType, List<GdPricePackagesCreativeTemplatesItem>> getCreativeTemplates() {
        List<Long> bsTemplateWhiteList = bsTemplateWhiteListProp.getOrDefault(emptyList());
        var allBannerStorageTemplates = bannerStorageDictRepository.getAllTemplates();
        var bannerStorageTemplates = filterList(allBannerStorageTemplates.values(),
                it -> !nvl(it.getTemplateTypeName(), "").equalsIgnoreCase("SMART")
                        && bsTemplateWhiteList.contains(it.getId())
        );
        var creativeTemplates = new HashMap<GdCreativeType, List<GdPricePackagesCreativeTemplatesItem>>();
        creativeTemplates.put(GdCreativeType.BANNERSTORAGE,
                mapList(bannerStorageTemplates,
                        PricePackageDataConverter::toGdPricePackagesBannerStorageCreativeTemplatesItem));
        PresetResponse presetResponse = canvasClient.getCreativeTemplates();
        presetResponse.getVideoPresets().forEach(preset -> {
                    GdCreativeType gdCreativeType = toGdCreativeType(convertVideoType(preset.getPresetId()));
                    var item = toGdPricePackagesVideoPresetsCreativeTemplatesItem(preset, gdCreativeType);
                    if (!creativeTemplates.containsKey(gdCreativeType)) {
                        creativeTemplates.put(gdCreativeType, new ArrayList<>());
                    }
                    creativeTemplates.get(gdCreativeType).add(item);
                }
        );

        creativeTemplates.put(GdCreativeType.CANVAS,
                mapList(presetResponse.getCanvasPresets(),
                        item -> toGdPricePackagesVideoPresetsCreativeTemplatesItem(item, GdCreativeType.CANVAS)));

        creativeTemplates.put(GdCreativeType.HTML5_CREATIVE,
                mapList(nvl(presetResponse.getHtml5Presets(), emptyList()),
                        item -> toGdPricePackagesHtml5PresetsCreativeTemplatesItem(item,
                                GdCreativeType.HTML5_CREATIVE)));

        return creativeTemplates;
    }

    public Supplier<List<GdPricePackagesCreativeTemplatesItem>> creativeTemplatesSupplier() {
        return () -> getCreativeTemplates()
                .values()
                .stream()
                .flatMap(it -> it.stream())
                .collect(Collectors.toList());
    }

    private PricePackageDataConverterContext createPricePackageDataConverterContext() {
        return new PricePackageDataConverterContext(sspPlatformsRepository.getAllSspPlatforms());
    }
}
