package ru.yandex.direct.grid.processing.service.pricepackage;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.abac.Attribute;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackageCategoriesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesAvailableForClientPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesFilterInternal;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesForClient;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageClient;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageRetargetingSubCategory;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackagesCreativeTemplateList;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackagesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeletePricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeletePricePackagesPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackagesPayload;
import ru.yandex.direct.web.annotations.RequiredAttributes;

import static ru.yandex.direct.utils.CommonUtils.nvl;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class PricePackageGraphQlService {

    private final PricePackageDataService pricePackageDataService;
    private final PricePackageClientLoginDataLoader clientLoginDataLoader;

    @Autowired
    public PricePackageGraphQlService(PricePackageDataService pricePackageDataService,
                                      PricePackageClientLoginDataLoader clientLoginDataLoader) {
        this.pricePackageDataService = pricePackageDataService;
        this.clientLoginDataLoader = clientLoginDataLoader;
    }

    @GraphQLQuery(name = "getPricePackages", description = "Получение прайсовых пакетов")
    @RequiredAttributes(Attribute.CAN_OPEN_PRICE_PACKAGE_GRID)
    public GdGetPricePackagesPayload getPricePackages(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetPricePackages input) {
        return pricePackageDataService.getPricePackages(input);
    }

    @GraphQLQuery(name = "login", description = "Логин клиента привязанного к прайсовому пакету")
    public CompletableFuture<String> getPricePackagesClientLogin(
            @GraphQLContext GdPricePackageClient pricePackageClient) {
        return clientLoginDataLoader.get().load(pricePackageClient.getClientId());
    }

    @PreAuthorizeRead
    @GraphQLQuery(name = "getPricePackagesAvailableForClient",
            description = "Получение прайсовых пакетов доступных текущему клиенту для создания кампании")
    public GdGetPricePackagesAvailableForClientPayload getPricePackagesAvailableForClient(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLArgument(name = "input") GdGetPricePackagesForClient input) {
        return pricePackageDataService.getPricePackagesForClient(
                context.getSubjectUser().getClientId(),
                context.getOperator().getUid(),
                nvl(input, new GdGetPricePackagesForClient().withFilter(new GdGetPricePackagesFilterInternal())));
    }

    @GraphQLQuery(name = "getPricePackageCategories",
            description = "Получение прайсовых пакетов доступных текущему клиенту для создания кампании")
    public GdGetPricePackageCategoriesPayload getPricePackageCategories(
            @GraphQLRootContext GridGraphQLContext context) {
        return pricePackageDataService.getPricePackageCategories();
    }

    @GraphQLMutation(name = "addPricePackages", description = "Создание прайсовых пакетов")
    @RequiredAttributes(Attribute.MANAGE_PRICE_PACKAGES)
    public GdAddPricePackagesPayload addPricePackages(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddPricePackages input) {
        return pricePackageDataService.addPricePackages(input, context);
    }

    @GraphQLMutation(name = "updatePricePackages", description = "Обновление прайсовых пакетов")
    @RequiredAttributes({Attribute.MANAGE_PRICE_PACKAGES, Attribute.APPROVE_PRICE_PACKAGES,
            Attribute.MANAGE_PRICE_PACKAGE_CLIENTS})
    public GdUpdatePricePackagesPayload updatePricePackages(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdatePricePackages input) {
        return pricePackageDataService.updatePricePackages(input, context.getOperator());
    }

    @GraphQLMutation(name = "deletePricePackages", description = "Удаление прайсовых пакетов")
    @RequiredAttributes(Attribute.MANAGE_PRICE_PACKAGES)
    public GdDeletePricePackagesPayload deletePricePackages(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdDeletePricePackages input) {
        return pricePackageDataService.deletePricePackages(input);
    }

    @PreAuthorizeRead
    @GraphQLQuery(name = "getCreativeTemplateList", description = "Список доступных шаблонов")
    public GdPricePackagesCreativeTemplateList getCreativeTemplateList(
            @GraphQLRootContext GridGraphQLContext context) {
        return new GdPricePackagesCreativeTemplateList()
                .withCreativeTemplates(
                        StreamEx.of(pricePackageDataService.getCreativeTemplates().values())
                                .flatMap(it -> it.stream())
                                .toList());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "fakePricePackageRetargetingSubCategoryQuery",
            description = "Метод для прокидывания на фронт типа данных GdPricePackageRetargetingSubCategory")
    public GdPricePackageRetargetingSubCategory fakePricePackageRetargetingSubCategoryQuery(
            @GraphQLRootContext GridGraphQLContext context) {
        return GdPricePackageRetargetingSubCategory.FAMILY;
    }
}
