package ru.yandex.direct.grid.processing.service.pricepackage;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.markupcondition.repository.MarkupConditionRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.model.pricepackage.GdCalcCampaignBudgetError;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackageAdGroupMarkupItem;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackageAdGroupMarkupsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackageCampaignBudget;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageAdGroupMarkup;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageCampaignBudgetPayload;
import ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter;
import ru.yandex.direct.regions.GeoTree;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.calcBudgetByVolume;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.calcPackagePrice;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.calcVolumeByCurrency;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Service
@ParametersAreNonnullByDefault
public class PricePackageMarkupDataService {
    private final PricePackageRepository pricePackageRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final ShardHelper shardHelper;
    private final MarkupConditionRepository markupConditionRepository;
    private final CampaignRepository campaignRepository;
    private final PricePackageService pricePackageService;
    private final ClientGeoService clientGeoService;
    private final FeatureService featureService;

    @Autowired
    public PricePackageMarkupDataService(
            ShardHelper shardHelper,
            PricePackageRepository pricePackageRepository,
            CampaignTypedRepository campaignTypedRepository,
            MarkupConditionRepository markupConditionRepository,
            CampaignRepository campaignRepository,
            PricePackageService pricePackageService,
            ClientGeoService clientGeoService,
            FeatureService featureService) {
        this.shardHelper = shardHelper;
        this.pricePackageRepository = pricePackageRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.markupConditionRepository = markupConditionRepository;
        this.campaignRepository = campaignRepository;
        this.pricePackageService = pricePackageService;
        this.clientGeoService = clientGeoService;
        this.featureService = featureService;
    }

    public GdGetPricePackageAdGroupMarkupsPayload getPricePackageMarkups(ClientId clientId,
            List<GdGetPricePackageAdGroupMarkupItem> items)
    {
        List<GdPricePackageAdGroupMarkup> markups = items.stream()
                .map(item -> {
                    Long campaignId = item.getCampaignId();
                    int shard = shardHelper.getShardByClientId(clientId);
                    Map<Long, CampaignType> campaignsTypeMap = campaignRepository.getCampaignsTypeMap(
                            shard, clientId, singleton(campaignId));
                    var packagePriceFunctionByCid =
                            RetargetingUtils.getPackagePriceFunctionByCampaignId(shard,
                                    campaignTypedRepository, pricePackageRepository,
                                    markupConditionRepository, campaignsTypeMap);

                    List<Long> regionIds = item.getRegionIds();
                    GeoTree geoTree = pricePackageService.getGeoTree();
                    List<Long> geo = clientGeoService.convertForSave(regionIds, geoTree);

                    return ifNotNull(packagePriceFunctionByCid.get(campaignId),
                            fn -> fn.apply(geo, item.getGoalIds(), item.getProjectParamConditions()));
                })
                .filter(Objects::nonNull)
                .map(priceWithMarkups -> new GdPricePackageAdGroupMarkup()
                        .withCpm(priceWithMarkups.getPrice())
                        .withPercent(priceWithMarkups.getPercent())
                        .withSeasonalMarkup(ifNotNull(priceWithMarkups.getPriceMarkup(), PricePackageDataConverter::toGdPriceMarkup))
                        .withTargetingMarkups(ifNotNull(priceWithMarkups.getTargetingMarkups(), PricePackageDataConverter::toGdTargetingMarkups))
                        .withMarkupConditions(ifNotNull(priceWithMarkups.getMarkupConditions(), PricePackageDataConverter::toGdMarkupConditions))
                )
                .collect(toList());

        return new GdGetPricePackageAdGroupMarkupsPayload().withRowset(markups);
    }

    public GdPricePackageCampaignBudgetPayload getPricePackageCampaignBudget(ClientId clientId,
            GdGetPricePackageCampaignBudget inputBudget) {
        CpmPriceCampaign camp = new CpmPriceCampaign()
                .withFlightOrderVolume(inputBudget.getOrderVolume())
                .withStartDate(inputBudget.getStartDate())
                .withEndDate(inputBudget.getEndDate());
        PricePackage pricePackage = pricePackageService.getPricePackage(inputBudget.getPackageId());
        var isBackendCalcEnabledByFeature = featureService.isEnabledForClientId(clientId,
                FeatureName.BACKEND_CPM_PRICE_CAMPAIGN_BUDGET_CALC_ENABLED);

        BigDecimal cpm = calcPackagePrice(camp, pricePackage, true, isBackendCalcEnabledByFeature);

        var result = new GdPricePackageCampaignBudgetPayload()
                .withCpm(cpm);
        if (cpm == null) {
            return result.withError(GdCalcCampaignBudgetError.IMPOSSIBLE_CHOOSE_SEASON_PERIOD);
        }
        if (inputBudget.getOrderVolume() != null) {
            return result.withBudget(calcBudgetByVolume(cpm, pricePackage.getCurrency(), inputBudget.getOrderVolume()));
        }
        return result.withOrderVolume(calcVolumeByCurrency(cpm, pricePackage.getCurrency(), inputBudget.getBudget()));
    }
}
