package ru.yandex.direct.grid.processing.service.pricepackage;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.pricepackage.model.BbKeyword;
import ru.yandex.direct.core.entity.pricepackage.model.ProjectParamCondition;
import ru.yandex.direct.core.entity.pricepackage.model.ProjectParamConjunction;
import ru.yandex.direct.core.entity.projectparam.repository.ProjectParamConditionRepository;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.pricepackage.GdBbKeyword;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetProjectParamConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetProjectParamConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.GdProjectParamCondition;
import ru.yandex.direct.grid.processing.model.pricepackage.GdProjectParamConjunction;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddProjectParamConditionPayloadItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddProjectParamConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddProjectParamConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeleteProjectParamConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdDeleteProjectParamConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdateProjectParamConditions;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdateProjectParamConditionsPayload;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdateProjectParamConditionsPayloadItem;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class ProjectParamConditionDataService {

    public static final int DEFAULT_LIMIT = 100;
    private final ProjectParamConditionRepository projectParamConditionRepository;

    @Autowired
    public ProjectParamConditionDataService(
            ProjectParamConditionRepository projectParamConditionRepository) {
        this.projectParamConditionRepository = projectParamConditionRepository;
    }

    public GdGetProjectParamConditionsPayload getProjectParamConditions(GdGetProjectParamConditions input) {
        int limit = ifNotNullOrDefault(input.getLimitOffset(), GdLimitOffset::getLimit, DEFAULT_LIMIT);
        int offset = ifNotNullOrDefault(input.getLimitOffset(), GdLimitOffset::getOffset, 0);
        String descriptionContains = nvl(input.getFilter().getDescriptionContains(), "");
        Set<Long> idIn = input.getFilter().getIdIn();

        List<ProjectParamCondition> filtered =
                projectParamConditionRepository.getAllProjectParamConditions().stream()
                        .filter(ppc -> ppc.getDescription().contains(descriptionContains))
                        .filter(ppc -> idIn == null || idIn.contains(ppc.getId()))
                        .collect(Collectors.toList());

        List<ProjectParamCondition> paged = filtered.stream()
                    .skip(offset)
                    .limit(limit)
                    .collect(Collectors.toList());

        return new GdGetProjectParamConditionsPayload()
                .withTotalCount(filtered.size())
                .withRowset(mapList(paged, mc -> new GdProjectParamCondition()
                        .withId(mc.getId())
                        .withIsArchived(mc.getIsArchived())
                        .withConjunctions(ifNotNull(mc.getConjunctions(),
                                conjunctions -> mapList(conjunctions, this::toGdProjectParamConjunction)))
                        .withDescription(mc.getDescription())));
    }

    private GdProjectParamConjunction toGdProjectParamConjunction(ProjectParamConjunction targeting) {
        return new GdProjectParamConjunction()
                .withBbKeywords(ifNotNull(targeting.getBbKeywords(),
                        keywords -> mapList(keywords, this::toGdBbKeyword)));
    }

    private GdBbKeyword toGdBbKeyword(BbKeyword keyword) {
        return new GdBbKeyword()
                .withKeyword(keyword.getKeyword())
                .withValue(keyword.getValue());
    }

    public GdAddProjectParamConditionsPayload addProjectParamConditions(GdAddProjectParamConditions input) {
        List<ProjectParamCondition> projectParamConditions =
                mapList(input.getAddItems(), item -> new ProjectParamCondition()
                        .withIsArchived(false)
                        .withDescription(item.getDescription())
                        .withConjunctions(ifNotNull(item.getConjunctions(),
                                conjunctions -> mapList(conjunctions, this::toCoreProjectParamConjunction))));
        List<Long> conditionIds = projectParamConditionRepository.addProjectParamConditions(projectParamConditions);
        return new GdAddProjectParamConditionsPayload()
                .withAddedItems(mapList(conditionIds, id -> new GdAddProjectParamConditionPayloadItem().withId(id)));
    }

    public GdUpdateProjectParamConditionsPayload updateProjectParamConditions(GdUpdateProjectParamConditions input) {
        List<ProjectParamCondition> projectParamConditions =
                mapList(input.getUpdateItems(), item -> new ProjectParamCondition()
                        .withId(item.getId())
                        .withIsArchived(item.getIsArchived())
                        .withDescription(item.getDescription())
                        .withConjunctions(ifNotNull(item.getConjunctions(),
                                conjunctions -> mapList(conjunctions, this::toCoreProjectParamConjunction))));

        projectParamConditionRepository.updateProjectParamConditions(projectParamConditions);

        return new GdUpdateProjectParamConditionsPayload()
                .withUpdatedItems(mapList(input.getUpdateItems(), item -> new GdUpdateProjectParamConditionsPayloadItem()
                        .withId(item.getId())));
    }

    private ProjectParamConjunction toCoreProjectParamConjunction(GdProjectParamConjunction targeting) {
        return new ProjectParamConjunction()
                .withBbKeywords(ifNotNull(targeting.getBbKeywords(),
                        keywords -> mapList(keywords, this::toCoreBbKeyword)));
    }

    private BbKeyword toCoreBbKeyword(GdBbKeyword keyword) {
        return new BbKeyword()
                .withKeyword(keyword.getKeyword())
                .withValue(keyword.getValue());
    }

    public GdDeleteProjectParamConditionsPayload deleteProjectParamConditions(GdDeleteProjectParamConditions input) {
        projectParamConditionRepository.deleteProjectParamConditions(input.getConditionIds());
        return new GdDeleteProjectParamConditionsPayload().withDeletedConditionIds(input.getConditionIds());
    }
}
