package ru.yandex.direct.grid.processing.service.pricepackage.converter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.canvas.client.model.video.Preset;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.OsType;
import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.core.entity.creative.model.BannerStorageDictLayoutItem;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.pricepackage.model.MarkupCondition;
import ru.yandex.direct.core.entity.pricepackage.model.MarkupConditionTargeting;
import ru.yandex.direct.core.entity.pricepackage.model.PriceMarkup;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCampaignOptions;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCategory;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageClient;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForClient;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageOrderBy;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageOrderByField;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackagePlatform;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackagesFilter;
import ru.yandex.direct.core.entity.pricepackage.model.PriceRetargetingCondition;
import ru.yandex.direct.core.entity.pricepackage.model.ShowsFrequencyLimit;
import ru.yandex.direct.core.entity.pricepackage.model.StatusApprove;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingMarkup;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsFixed;
import ru.yandex.direct.core.entity.pricepackage.model.ViewType;
import ru.yandex.direct.core.entity.targettags.model.TargetTag;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.processing.model.GdDateRange;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeType;
import ru.yandex.direct.grid.processing.model.goal.GdGoalType;
import ru.yandex.direct.grid.processing.model.placement.GdPiPage;
import ru.yandex.direct.grid.processing.model.pricepackage.GdAvailableAdGroupType;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesFilter;
import ru.yandex.direct.grid.processing.model.pricepackage.GdGetPricePackagesForClientFilter;
import ru.yandex.direct.grid.processing.model.pricepackage.GdMarkupCondition;
import ru.yandex.direct.grid.processing.model.pricepackage.GdMarkupConditionTargeting;
import ru.yandex.direct.grid.processing.model.pricepackage.GdMarkupConditionsOperator;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPriceAllowedCreativeTemplates;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPriceCryptaSegment;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPriceMarkup;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackage;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageBidModifiers;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageCategory;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageClient;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageForClient;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageOrderBy;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackageOrderByField;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackagePlatform;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackagePlatformBidModifier;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPricePackagesCreativeTemplatesItem;
import ru.yandex.direct.grid.processing.model.pricepackage.GdPriceRetargetingCondition;
import ru.yandex.direct.grid.processing.model.pricepackage.GdShowsFrequencyLimit;
import ru.yandex.direct.grid.processing.model.pricepackage.GdStatusApprove;
import ru.yandex.direct.grid.processing.model.pricepackage.GdTargetingMarkup;
import ru.yandex.direct.grid.processing.model.pricepackage.GdTargetingsCustom;
import ru.yandex.direct.grid.processing.model.pricepackage.GdTargetingsFixed;
import ru.yandex.direct.grid.processing.model.pricepackage.GdViewType;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackageItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackagePayloadItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdAddPricePackages;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdMutationPriceRetargetingCondition;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdMutationTargetingsCustom;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdMutationTargetingsFixed;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdPricePackageClientInput;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackagesItem;
import ru.yandex.direct.grid.processing.model.pricepackage.mutation.GdUpdatePricePackagesPayloadItem;
import ru.yandex.direct.grid.processing.model.targettag.GdTargetTag;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.result.MappingPathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.web.core.model.retargeting.WebGoalType;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.product.service.ProductService.NOT_BUSINESS_UNIT_ID;
import static ru.yandex.direct.grid.processing.service.bidmodifier.BidModifierDataConverter.toGdInventoryType;
import static ru.yandex.direct.grid.processing.service.bidmodifier.BidModifierDataConverter.toInventoryType;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toCreativeType;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toGdCreativeType;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class PricePackageDataConverter {
    private final PricePackageDataConverterContext context;
    public static final PathNodeConverter PRICE_PACKAGE_CLIENT_PATH_CONVERTER =
            MappingPathNodeConverter.builder(PricePackageClient.class.getName())
                    .replace(PricePackageClient.CLIENT_ID.name(), GdPricePackageClientInput.LOGIN.name())
                    .build();

    public PricePackageDataConverter(PricePackageDataConverterContext context) {
        this.context = context;
    }

    public List<PricePackage> toCorePricePackages(GdAddPricePackages input) {
        return mapList(input.getAddItems(), this::toCorePricePackage);
    }

    public PricePackage toCorePricePackage(GdAddPricePackageItem item) {
        TargetingsFixed targetingsFixed = toCoreTargetingsFixed(item.getTargetingsFixed());
        TargetingsCustom targetingsCustom = toCoreTargetingsCustom(item.getTargetingsCustom());
        var availableAdGroupTypes = item.getAvailableAdGroupTypes() == null || item.getAvailableAdGroupTypes().isEmpty() ?
                Set.of(AdGroupType.CPM_YNDX_FRONTPAGE) ://значение по умолчанию
                toCoreAvailableAdGroupTypes(item.getAvailableAdGroupTypes());
        return new PricePackage()
                .withProductId(item.getProductId())
                .withAuctionPriority(item.getAuctionPriority())
                .withTitle(item.getTitle())
                .withTrackerUrl(item.getTrackerUrl())
                .withPrice(item.getPrice())
                .withCurrency(item.getCurrency())
                .withOrderVolumeMin(item.getOrderVolumeMin())
                .withOrderVolumeMax(item.getOrderVolumeMax())
                .withPriceMarkups(toCorePriceMarkups(item.getPriceMarkups()))
                .withTargetingMarkups(toCoreTargetingMarkups(item.getTargetingMarkups()))
                .withTargetingsFixed(targetingsFixed)
                .withTargetingsCustom(targetingsCustom)
                .withDateStart(item.getDateStart())
                .withDateEnd(item.getDateEnd())
                .withIsPublic(item.getIsPublic())
                .withIsSpecial(nvl(item.getIsSpecial(), false))
                .withIsCpd(nvl(item.getIsCpd(), false))
                .withClients(emptyList())
                .withCampaignAutoApprove(nvl(item.getCampaignAutoApprove(), false))
                .withIsDraftApproveAllowed(nvl(item.getIsDraftApproveAllowed(), false))
                .withAvailableAdGroupTypes(availableAdGroupTypes)
                .withIsFrontpage(item.getAvailableAdGroupTypes() == null
                        || item.getAvailableAdGroupTypes().isEmpty()
                        || toCoreIsFrontpage(item.getAvailableAdGroupTypes()))
                .withAllowedPageIds(item.getAllowedPageIds())
                .withAllowedProjectParamConditions(item.getAllowedProjectParamConditions())
                .withAllowedDomains(toCoreAllowedDomains(item.getAllowedDomains()))
                .withAllowedSsp(toCoreAllowedSsp(item.getAllowedDomains()))
                .withAllowedTargetTags(item.getAllowedTargetTags())
                .withAllowedOrderTags(ifNotNull(item.getAllowedOrderTags(), List::copyOf))
                .withCampaignOptions(toCorePricePackageCampaignOptions(item))
                .withBidModifiers(toCoreBidModifiers(item.getBidModifiers()))
                .withAllowedCreativeTemplates(toCoreAllowedCreativeTemplates(item.getAllowedCreativeTemplates()))
                .withCategoryId(item.getCategoryId());
    }

    public ModelChanges<PricePackage> toCoreModelChange(GdUpdatePricePackagesItem item,
                                                        Map<String, Long> loginToClientId,
                                                        PricePackage oldPricePackage) {
        TargetingsFixed targetingsFixed = toCoreTargetingsFixed(item.getTargetingsFixed());
        TargetingsCustom targetingsCustom = toCoreTargetingsCustom(item.getTargetingsCustom());
        List<PricePackageClient> clients = toCoreClients(item.getClients(), loginToClientId);

        return new ModelChanges<>(item.getId(), PricePackage.class)
                .processNotNull(item.getProductId(), PricePackage.PRODUCT_ID)
                .processNotNull(item.getTitle(), PricePackage.TITLE)
                .processNotNull(item.getTrackerUrl(), PricePackage.TRACKER_URL)
                .processNotNull(item.getPrice(), PricePackage.PRICE)
                .processNotNull(item.getCurrency(), PricePackage.CURRENCY)
                .processNotNull(item.getOrderVolumeMin(), PricePackage.ORDER_VOLUME_MIN)
                .processNotNull(item.getOrderVolumeMax(), PricePackage.ORDER_VOLUME_MAX)
                .processNotNull(toCorePriceMarkups(item.getPriceMarkups()), PricePackage.PRICE_MARKUPS)
                .processNotNull(targetingsFixed, PricePackage.TARGETINGS_FIXED)
                .processNotNull(targetingsCustom, PricePackage.TARGETINGS_CUSTOM)
                .processNotNull(toCoreTargetingMarkups(item.getTargetingMarkups()), PricePackage.TARGETING_MARKUPS)
                .processNotNull(GdStatusApprove.toSource(item.getStatusApprove()), PricePackage.STATUS_APPROVE)
                .processNotNull(item.getDateStart(), PricePackage.DATE_START)
                .processNotNull(item.getDateEnd(), PricePackage.DATE_END)
                .processNotNull(item.getIsPublic(), PricePackage.IS_PUBLIC)
                .processNotNull(item.getIsSpecial(), PricePackage.IS_SPECIAL)
                .processNotNull(item.getIsCpd(), PricePackage.IS_CPD)
                .processNotNull(item.getIsArchived(), PricePackage.IS_ARCHIVED)
                .processNotNull(clients, PricePackage.CLIENTS)
                .processNotNull(item.getCampaignAutoApprove(), PricePackage.CAMPAIGN_AUTO_APPROVE)
                .processNotNull(item.getIsDraftApproveAllowed(), PricePackage.IS_DRAFT_APPROVE_ALLOWED)
                .processNotNull(item.getAllowedPageIds(), PricePackage.ALLOWED_PAGE_IDS)
                .processNotNull(item.getAllowedProjectParamConditions(), PricePackage.ALLOWED_PROJECT_PARAM_CONDITIONS)
                .processNotNull(toCoreAllowedSsp(item.getAllowedDomains()), PricePackage.ALLOWED_SSP)
                .processNotNull(toCoreAllowedDomains(item.getAllowedDomains()), PricePackage.ALLOWED_DOMAINS)
                .processNotNull(item.getAllowedTargetTags(), PricePackage.ALLOWED_TARGET_TAGS)
                .processNotNull(ifNotNull(item.getAllowedOrderTags(), List::copyOf), PricePackage.ALLOWED_ORDER_TAGS)
                .processNotNull(toCoreAvailableAdGroupTypes(item.getAvailableAdGroupTypes()),
                        PricePackage.AVAILABLE_AD_GROUP_TYPES)
                .processNotNull(toCoreIsFrontpage(item.getAvailableAdGroupTypes()), PricePackage.IS_FRONTPAGE)
                .processNotNull(toCorePricePackageCampaignOptions(item, oldPricePackage), PricePackage.CAMPAIGN_OPTIONS)
                .processNotNull(toCoreBidModifiers(item.getBidModifiers()), PricePackage.BID_MODIFIERS)
                .processNotNull(toCoreAllowedCreativeTemplates(item.getAllowedCreativeTemplates()),
                        PricePackage.ALLOWED_CREATIVE_TEMPLATES)
                .processNotNull(item.getCategoryId(), PricePackage.CATEGORY_ID)
                .processNotNull(item.getAuctionPriority(), PricePackage.AUCTION_PRIORITY);
    }

    private static Boolean toCoreIsFrontpage(@Nullable Set<GdAvailableAdGroupType> gd) {
        if (gd == null) {
            return null;
        }
        return gd.contains(GdAvailableAdGroupType.CPM_PRICE_FRONTPAGE_VIDEO) || gd.contains(GdAvailableAdGroupType.CPM_PRICE);
    }

    private static Map<CreativeType, List<Long>> toCoreAllowedCreativeTemplates(
            GdPriceAllowedCreativeTemplates gd) {
        if (gd == null) {
            return null;
        }

        Map<CreativeType, List<Long>> creativeTemplates = new HashMap<>();
        gd.getCreativeTemplateIds().forEach(
                (gdCreativeType, ids) -> creativeTemplates.put(toCreativeType(gdCreativeType), ids)
        );
        return creativeTemplates;
    }

    private static PricePackageCampaignOptions toCorePricePackageCampaignOptions(GdAddPricePackageItem item) {
        return toCorePricePackageCampaignOptions(item.getAllowBrandSafety(), item.getShowsFrequencyLimit(),
                item.getAllowDisabledPlaces(), item.getAllowDisabledVideoPlaces(), item.getAllowBrandLift(),
                item.getAllowImage(), item.getAllowGallery(),
                null);
    }

    private static PricePackageCampaignOptions toCorePricePackageCampaignOptions(GdUpdatePricePackagesItem item,
                                                                                 PricePackage oldPricePackage) {
        //понять есть ли обновление
        if (item.getAllowBrandSafety() == null && item.getShowsFrequencyLimit() == null &&
                item.getAllowDisabledPlaces() == null && item.getAllowDisabledVideoPlaces() == null &&
                item.getAllowBrandLift() == null) {
            return null;
        }
        return toCorePricePackageCampaignOptions(item.getAllowBrandSafety(), item.getShowsFrequencyLimit(),
                item.getAllowDisabledPlaces(), item.getAllowDisabledVideoPlaces(), item.getAllowBrandLift(),
                item.getAllowImage(), item.getAllowGallery(),
                oldPricePackage);
    }

    private static PricePackageCampaignOptions toCorePricePackageCampaignOptions(
            Boolean allowBrandSafety, GdShowsFrequencyLimit showsFrequencyLimit, Boolean allowDisabledPlaces,
            Boolean allowDisabledVideoPlaces, Boolean allowBrandLift, Boolean allowImage, Boolean allowGallery,
            PricePackage oldPricePackage) {
        var pricePackageCampaignOptions = oldPricePackage == null || oldPricePackage.getCampaignOptions() == null
                ? new PricePackageCampaignOptions()
                : oldPricePackage.getCampaignOptions();
        if (allowBrandSafety != null) {
            pricePackageCampaignOptions.setAllowBrandSafety(allowBrandSafety);
        }
        pricePackageCampaignOptions.withShowsFrequencyLimit(showsFrequencyLimit == null ? null :
                new ShowsFrequencyLimit()
                        .withFrequencyLimit(showsFrequencyLimit.getFrequencyLimit())
                        .withFrequencyLimitDays(showsFrequencyLimit.getFrequencyLimitDays())
                        .withMinLimit(showsFrequencyLimit.getMinLimit())
                        .withFrequencyLimitIsForCampaignTime(showsFrequencyLimit.getFrequencyLimitIsForCampaignTime()));
        if (allowDisabledPlaces != null) {
            pricePackageCampaignOptions.setAllowDisabledPlaces(allowDisabledPlaces);
        }
        if (allowDisabledVideoPlaces != null) {
            pricePackageCampaignOptions.setAllowDisabledVideoPlaces(allowDisabledVideoPlaces);
        }
        if (allowBrandLift != null) {
            pricePackageCampaignOptions.setAllowBrandLift(allowBrandLift);
        }
        if (allowImage != null) {
            pricePackageCampaignOptions.setAllowImage(allowImage);
        }
        if (allowGallery != null) {
            pricePackageCampaignOptions.setAllowGallery(allowGallery);
        }
        return pricePackageCampaignOptions;
    }

    public static Set<AdGroupType> toCoreAvailableAdGroupTypes(@Nullable Set<GdAvailableAdGroupType> gd) {
        if (gd == null) {
            return null;
        }
        return mapSet(gd, it -> {
            switch (it) {
                case CPM_PRICE:
                    return AdGroupType.CPM_YNDX_FRONTPAGE;
                case CPM_PRICE_VIDEO:
                case CPM_PRICE_FRONTPAGE_VIDEO:
                    return AdGroupType.CPM_VIDEO;
                case CPM_PRICE_BANNER:
                    return AdGroupType.CPM_BANNER;
                case CPM_PRICE_AUDIO:
                    return AdGroupType.CPM_AUDIO;
                default:
                    throw new IllegalArgumentException("No conversion for " + it);
            }
        });
    }

    public static Set<GdAdGroupType> toGdAdGroupTypes(@Nullable Set<AdGroupType> core) {
        if (core == null) {
            return null;
        }
        return mapSet(core, it -> {
            switch (it) {
                case CPM_YNDX_FRONTPAGE:
                    return GdAdGroupType.CPM_PRICE;
                case CPM_VIDEO:
                    return GdAdGroupType.CPM_PRICE_VIDEO;
                case CPM_BANNER:
                    return GdAdGroupType.CPM_PRICE_BANNER;
                case CPM_AUDIO:
                    return GdAdGroupType.CPM_PRICE_AUDIO;
                default:
                    throw new IllegalArgumentException("No conversion for " + it);
            }
        });
    }

    public static Set<GdAvailableAdGroupType> toGdAvailableAdGroupTypes(PricePackageForClient pricePackage) {
        if (pricePackage.getAvailableAdGroupTypes() == null) {
            return null;
        }
        return mapSet(pricePackage.getAvailableAdGroupTypes(), it -> {
            switch (it) {
                case CPM_YNDX_FRONTPAGE:
                    return GdAvailableAdGroupType.CPM_PRICE;
                case CPM_VIDEO:
                    return pricePackage.getIsFrontpage() ?
                            GdAvailableAdGroupType.CPM_PRICE_FRONTPAGE_VIDEO : GdAvailableAdGroupType.CPM_PRICE_VIDEO;
                case CPM_BANNER:
                    return GdAvailableAdGroupType.CPM_PRICE_BANNER;
                case CPM_AUDIO:
                    return GdAvailableAdGroupType.CPM_PRICE_AUDIO;
                default:
                    throw new IllegalArgumentException("No conversion for " + it);
            }
        });
    }

    private static List<PriceMarkup> toCorePriceMarkups(@Nullable List<GdPriceMarkup> gdPriceMarkups) {
        if (gdPriceMarkups == null) {
            return null;
        }
        return mapList(gdPriceMarkups, gdPriceMarkup -> new PriceMarkup()
                .withDateStart(gdPriceMarkup.getDateStart())
                .withDateEnd(gdPriceMarkup.getDateEnd())
                .withPercent(gdPriceMarkup.getPercent()));
    }

    private static List<TargetingMarkup> toCoreTargetingMarkups(@Nullable List<GdTargetingMarkup> gdTargetingMarkups) {
        if (gdTargetingMarkups == null) {
            return null;
        }
        return mapList(gdTargetingMarkups, gdTargetingMarkup -> new TargetingMarkup()
                .withConditionId(gdTargetingMarkup.getConditionId())
                .withPercent(gdTargetingMarkup.getPercent()));
    }

    private static TargetingsFixed toCoreTargetingsFixed(@Nullable GdMutationTargetingsFixed gdTargetingsFixed) {
        if (gdTargetingsFixed == null) {
            return null;
        }
        List<ViewType> viewTypes = mapList(gdTargetingsFixed.getViewTypes(), GdViewType::toSource);
        return new TargetingsFixed()
                .withGeo(gdTargetingsFixed.getGeo())
                .withGeoType(gdTargetingsFixed.getGeoType())
                // не принимаем geoExpanded от фронта, заполним это поле сами позже
                .withViewTypes(viewTypes)
                .withCryptaSegments(gdTargetingsFixed.getCryptaSegments())
                .withAllowExpandedDesktopCreative(nvl(gdTargetingsFixed.getAllowExpandedDesktopCreative(), false))
                .withAllowPremiumDesktopCreative(nvl(gdTargetingsFixed.getAllowPremiumDesktopCreative(), false))
                .withHideIncomeSegment(nvl(gdTargetingsFixed.getHideIncomeSegment(), false));
    }

    private static TargetingsCustom toCoreTargetingsCustom(@Nullable GdMutationTargetingsCustom gdTargetingsCustom) {
        if (gdTargetingsCustom == null) {
            return null;
        }
        return new TargetingsCustom()
                .withGeo(gdTargetingsCustom.getGeo())
                .withGeoType(gdTargetingsCustom.getGeoType())
                .withGeoExpanded(gdTargetingsCustom.getGeoExpanded())
                .withRetargetingCondition(toCoreRetargetingCondition(gdTargetingsCustom.getRetargetingCondition()));
    }

    private static PriceRetargetingCondition toCoreRetargetingCondition(
            @Nullable GdMutationPriceRetargetingCondition retargetingCondition) {
        if (retargetingCondition == null) {
            return null;
        }
        return new PriceRetargetingCondition()
                .withAllowAudienceSegments(retargetingCondition.getAllowAudienceSegments())
                .withAllowMetrikaSegments(retargetingCondition.getAllowMetrikaSegments())
                .withLowerCryptaTypesCount(retargetingCondition.getLowerCryptaTypesCount())
                .withUpperCryptaTypesCount(retargetingCondition.getUpperCryptaTypesCount())
                .withCryptaSegments(retargetingCondition.getCryptaSegments());
    }

    public static List<GdPriceMarkup> toGdPriceMarkups(@Nullable List<PriceMarkup> priceMarkups) {
        if (priceMarkups == null) {
            return null;
        }
        return mapList(priceMarkups, PricePackageDataConverter::toGdPriceMarkup);
    }

    public static GdPriceMarkup toGdPriceMarkup(PriceMarkup priceMarkup) {
        return new GdPriceMarkup()
                .withDateStart(priceMarkup.getDateStart())
                .withDateEnd(priceMarkup.getDateEnd())
                .withPercent(priceMarkup.getPercent());
    }

    public static GdMarkupConditionTargeting toGdMarkupConditionTargeting(MarkupConditionTargeting targeting) {
        return new GdMarkupConditionTargeting()
                .withGeoType(targeting.getGeoType())
                .withGeo(targeting.getGeo())
                .withGoalTypes(ifNotNull(targeting.getGoalTypes(),
                        goalTypes -> mapList(goalTypes, GdGoalType::fromSource)))
                .withProjectParamConditions(targeting.getProjectParamConditions())
                .withCryptaSegments(targeting.getCryptaSegments());
    }


    public static MarkupConditionTargeting toCoreMarkupConditionTargeting(GdMarkupConditionTargeting targeting) {
        return new MarkupConditionTargeting()
                .withGeoType(targeting.getGeoType())
                .withGeo(targeting.getGeo())
                .withGoalTypes(ifNotNull(targeting.getGoalTypes(),
                        gdGoalTypes -> mapList(gdGoalTypes, GdGoalType::toSource)))
                .withProjectParamConditions(targeting.getProjectParamConditions())
                .withCryptaSegments(targeting.getCryptaSegments());
    }

    public static GdMarkupCondition toGdMarkupCondition(MarkupCondition mc) {
        return new GdMarkupCondition()
                .withId(mc.getId())
                .withIsArchived(mc.getIsArchived())
                .withGroup(mc.getGroup())
                .withPriority(mc.getPriority())
                .withOperator(GdMarkupConditionsOperator.fromSource(mc.getOperator()))
                .withTargeting(toGdMarkupConditionTargeting(mc.getTargeting()))
                .withDescription(mc.getDescription());
    }

    public static List<GdMarkupCondition> toGdMarkupConditions(List<MarkupCondition> allMarkupConditions) {
        return mapList(allMarkupConditions, PricePackageDataConverter::toGdMarkupCondition);
    }

    public static List<GdTargetingMarkup> toGdTargetingMarkups(@Nullable List<TargetingMarkup> targetingMarkups) {
        if (targetingMarkups == null) {
            return null;
        }
        return mapList(targetingMarkups, targetingMarkup -> new GdTargetingMarkup()
                .withConditionId(targetingMarkup.getConditionId())
                .withPercent(targetingMarkup.getPercent()));
    }

    public static GdTargetingsFixed toGdTargetingsFixed(@Nullable TargetingsFixed targetingsFixed,
                                                        UnaryOperator<List<Long>> geoToFrontendGeo,
                                                        Function<Long, WebGoalType> cryptaSegmentsSupplier) {
        if (targetingsFixed == null) {
            return null;
        }
        List<GdViewType> viewTypes = mapList(targetingsFixed.getViewTypes(), GdViewType::fromSource);

        // т.к. при geoType = 10 и geo != null geoExpanded некорректно вычисляется как пустой список, меняем его на geo.
        // т.о. либо при geoType != 10 он останется пустым списком, либо при geoType = 10 и пустом geo он также
        // останется пустым списком, либо при непустом geo мы некорректно пустой geoExpanded заменим на непустой список.
        // т.к. при geoType = 10 на фронте учитывается только наличие элементов в geoExpanded, этого достаточно.
        var isGeoExpandedEmpty = ifNotNullOrDefault(targetingsFixed.getGeoExpanded(), List::isEmpty, false);
        var geoExpanded = isGeoExpandedEmpty ? targetingsFixed.getGeo() : targetingsFixed.getGeoExpanded();

        return new GdTargetingsFixed()
                .withGeo(targetingsFixed.getGeo())
                .withGeoType(targetingsFixed.getGeoType())
                .withGeoExpanded(geoToFrontendGeo.apply(geoExpanded))
                .withViewTypes(viewTypes)
                .withCryptaSegments(toGdcryptaSegmentsList(targetingsFixed.getCryptaSegments(),
                        cryptaSegmentsSupplier))
                .withAllowExpandedDesktopCreative(nvl(targetingsFixed.getAllowExpandedDesktopCreative(), false))
                .withAllowPremiumDesktopCreative(nvl(targetingsFixed.getAllowPremiumDesktopCreative(), false))
                .withHideIncomeSegment(nvl(targetingsFixed.getHideIncomeSegment(), false));
    }

    public static GdMutationTargetingsFixed toGdMutationTargetingsFixed(@Nullable TargetingsFixed targetingsFixed,
                                                                        UnaryOperator<List<Long>> geoToFrontendGeo) {
        if (targetingsFixed == null) {
            return null;
        }
        List<GdViewType> viewTypes = mapList(targetingsFixed.getViewTypes(), GdViewType::fromSource);

        return new GdMutationTargetingsFixed()
                .withGeo(targetingsFixed.getGeo())
                .withGeoType(targetingsFixed.getGeoType())
                .withGeoExpanded(geoToFrontendGeo.apply(targetingsFixed.getGeoExpanded()))
                .withViewTypes(viewTypes)
                .withCryptaSegments(targetingsFixed.getCryptaSegments())
                .withAllowExpandedDesktopCreative(targetingsFixed.getAllowExpandedDesktopCreative())
                .withAllowPremiumDesktopCreative(targetingsFixed.getAllowPremiumDesktopCreative())
                .withHideIncomeSegment(targetingsFixed.getHideIncomeSegment());
    }

    public static GdTargetingsCustom toGdTargetingsCustom(@Nullable TargetingsCustom targetingsCustom,
                                                          UnaryOperator<List<Long>> geoToFrontendGeo,
                                                          Function<Long, WebGoalType> cryptaSegmentsSupplier) {
        if (targetingsCustom == null) {
            return null;
        }

        // т.к. при geoType = 10 и geo != null geoExpanded некорректно вычисляется как пустой список, меняем его на geo.
        // т.о. либо при geoType != 10 он останется пустым списком, либо при geoType = 10 и пустом geo он также
        // останется пустым списком, либо при непустом geo мы некорректно пустой geoExpanded заменим на непустой список.
        // т.к. при geoType = 10 на фронте учитывается только наличие элементов в geoExpanded, этого достаточно.
        var isGeoExpandedEmpty = ifNotNullOrDefault(targetingsCustom.getGeoExpanded(), List::isEmpty, false);
        var geoExpanded = isGeoExpandedEmpty ? targetingsCustom.getGeo() : targetingsCustom.getGeoExpanded();

        return new GdTargetingsCustom()
                .withGeo(targetingsCustom.getGeo())
                .withGeoType(targetingsCustom.getGeoType())
                .withGeoExpanded(geoToFrontendGeo.apply(geoExpanded))
                .withRetargetingCondition(toGdRetargetingCondition(targetingsCustom.getRetargetingCondition(),
                        cryptaSegmentsSupplier));
    }

    public static GdMutationPriceRetargetingCondition toGdMutationRetargetingCondition(
            @Nullable PriceRetargetingCondition retargetingCondition) {
        if (retargetingCondition == null) {
            return null;
        }
        return new GdMutationPriceRetargetingCondition()
                .withAllowAudienceSegments(retargetingCondition.getAllowAudienceSegments())
                .withAllowMetrikaSegments(retargetingCondition.getAllowMetrikaSegments())
                .withLowerCryptaTypesCount(retargetingCondition.getLowerCryptaTypesCount())
                .withUpperCryptaTypesCount(retargetingCondition.getUpperCryptaTypesCount())
                .withCryptaSegments(retargetingCondition.getCryptaSegments());
    }

    private static GdPriceRetargetingCondition toGdRetargetingCondition(
            @Nullable PriceRetargetingCondition retargetingCondition,
            Function<Long, WebGoalType> cryptaSegmentsSupplier) {
        if (retargetingCondition == null) {
            return null;
        }
        return new GdPriceRetargetingCondition()
                .withAllowAudienceSegments(retargetingCondition.getAllowAudienceSegments())
                .withAllowMetrikaSegments(retargetingCondition.getAllowMetrikaSegments())
                .withLowerCryptaTypesCount(retargetingCondition.getLowerCryptaTypesCount())
                .withUpperCryptaTypesCount(retargetingCondition.getUpperCryptaTypesCount())
                .withCryptaSegments(toGdcryptaSegmentsList(retargetingCondition.getCryptaSegments(),
                        cryptaSegmentsSupplier));
    }

    public static List<GdPriceCryptaSegment> toGdcryptaSegmentsList(@Nullable Collection<Long> ids,
                                                                    Function<Long, WebGoalType> cryptaSegmentsSupplier) {
        if (ids == null) {
            return null;
        }
        return ids.stream()
                .map(id -> new GdPriceCryptaSegment()
                        .withId(id)
                        .withType(cryptaSegmentsSupplier.apply(id))
                ).collect(Collectors.toList());
    }

    public static GdMutationTargetingsCustom toGdMutationTargetingsCustom(@Nullable TargetingsCustom targetingsCustom,
                                                                          UnaryOperator<List<Long>> geoToFrontendGeo) {
        if (targetingsCustom == null) {
            return null;
        }
        return new GdMutationTargetingsCustom()
                .withGeo(targetingsCustom.getGeo())
                .withGeoType(targetingsCustom.getGeoType())
                .withGeoExpanded(geoToFrontendGeo.apply(targetingsCustom.getGeoExpanded()))
                .withRetargetingCondition(toGdMutationRetargetingCondition(targetingsCustom.getRetargetingCondition()));
    }

    private static List<PricePackageClient> toCoreClients(@Nullable List<GdPricePackageClientInput> gdClients,
                                                          Map<String, Long> loginToClientId) {
        if (gdClients == null) {
            return null;
        }
        return mapList(gdClients, gdClient -> {
            checkArgument(gdClient.getClientId() == null || gdClient.getLogin() == null,
                    "You must specify either clientId or login, but not both at the same time");
            return new PricePackageClient()
                    .withClientId(nullableNvl(gdClient.getClientId(), loginToClientId.get(gdClient.getLogin())))
                    .withIsAllowed(gdClient.getIsAllowed());
        });
    }

    private static List<GdPricePackageClient> toGdClients(List<PricePackageClient> clients) {
        return mapList(clients, client -> new GdPricePackageClient()
                .withClientId(client.getClientId())
                .withIsAllowed(client.getIsAllowed()));
    }

    public static GdAddPricePackagePayloadItem toGdAddPricePackagePayloadItem(Long addedId) {
        return new GdAddPricePackagePayloadItem()
                .withId(addedId);

    }

    public static GdUpdatePricePackagesPayloadItem toGdUpdatePricePackagesPayloadItem(Long updatedId) {
        return new GdUpdatePricePackagesPayloadItem()
                .withId(updatedId);
    }

    public static List<GdPricePackage> toGdPricePackages(List<PricePackage> pricePackages,
                                                         Map<Long, Placement> pagesToPlacements,
                                                         UnaryOperator<List<Long>> geoToFrontendGeo,
                                                         Function<Long, WebGoalType> cryptaSegmentsSupplier,
                                                         Supplier<List<GdPricePackagesCreativeTemplatesItem>> creativeTemplatesSupplier) {
        return mapList(pricePackages, pricePackage -> {
                    var gdPricePackage = new GdPricePackage()
                            .withId(pricePackage.getId())
                            .withProductId(pricePackage.getProductId())
                            .withAuctionPriority(pricePackage.getAuctionPriority())
                            .withTitle(pricePackage.getTitle())
                            .withTrackerUrl(pricePackage.getTrackerUrl())
                            .withPrice(pricePackage.getPrice())
                            .withCurrency(pricePackage.getCurrency())
                            .withOrderVolumeMin(pricePackage.getOrderVolumeMin())
                            .withOrderVolumeMax(pricePackage.getOrderVolumeMax())
                            .withPriceMarkups(toGdPriceMarkups(pricePackage.getPriceMarkups()))
                            .withTargetingMarkups(toGdTargetingMarkups(pricePackage.getTargetingMarkups()))
                            .withTargetingsFixed(toGdTargetingsFixed(pricePackage.getTargetingsFixed(),
                                    geoToFrontendGeo, cryptaSegmentsSupplier))
                            .withTargetingsCustom(toGdTargetingsCustom(pricePackage.getTargetingsCustom(),
                                    geoToFrontendGeo, cryptaSegmentsSupplier))
                            .withStatusApprove(GdStatusApprove.fromSource(pricePackage.getStatusApprove()))
                            .withLastUpdateTime(pricePackage.getLastUpdateTime())
                            .withDateStart(pricePackage.getDateStart())
                            .withDateEnd(pricePackage.getDateEnd())
                            .withIsPublic(pricePackage.getIsPublic())
                            .withIsSpecial(pricePackage.getIsSpecial())
                            .withIsCpd(pricePackage.getIsCpd())
                            .withIsArchived(pricePackage.getIsArchived())
                            .withClients(toGdClients(pricePackage.getClients()))
                            .withCampaignAutoApprove(pricePackage.getCampaignAutoApprove())
                            .withIsDraftApproveAllowed(pricePackage.getIsDraftApproveAllowed())
                            .withAvailableAdGroupTypes(toGdAvailableAdGroupTypes(pricePackage))
                            .withAllowedPageIds(pricePackage.getAllowedPageIds())
                            .withAllowedProjectParamConditions(pricePackage.getAllowedProjectParamConditions())
                            .withAllowedDomains(toGdAllowedDomains(pricePackage.getAllowedSsp(),
                                    pricePackage.getAllowedDomains()))
                            .withAllowedTargetTags(pricePackage.getAllowedTargetTags())
                            .withAllowedOrderTags(ifNotNull(pricePackage.getAllowedOrderTags(), Set::copyOf))
                            .withBidModifiers(toGdBidModifiers(pricePackage.getBidModifiers()))
                            .withPlacements(nvl(pricePackage.getAllowedPageIds(), emptyList()).stream()
                                    .filter(pagesToPlacements::containsKey)
                                    .map(e -> toGdPiPage(pagesToPlacements.get(e))).collect(Collectors.toList()))
                            .withAllowedCreativeTemplates(
                                    toGdAllowedCreativeTemplates(pricePackage.getAllowedCreativeTemplates(),
                                            creativeTemplatesSupplier))
                            .withCategoryId(pricePackage.getCategoryId());
                    if (pricePackage.getCampaignOptions() != null) {
                        var options = pricePackage.getCampaignOptions();
                        gdPricePackage
                                .withAllowBrandSafety(options.getAllowBrandSafety())
                                .withAllowDisabledPlaces(options.getAllowDisabledPlaces())
                                .withAllowDisabledVideoPlaces(options.getAllowDisabledVideoPlaces())
                                .withAllowBrandLift(options.getAllowBrandLift())
                                .withAllowImage(options.getAllowImage())
                                .withAllowGallery(options.getAllowGallery())
                                .withShowsFrequencyLimit(options.getShowsFrequencyLimit() == null ? null :
                                        new GdShowsFrequencyLimit()
                                                .withFrequencyLimit(options.getShowsFrequencyLimit().getFrequencyLimit())
                                                .withFrequencyLimitDays(
                                                        options.getShowsFrequencyLimit().getFrequencyLimitDays())
                                                .withMinLimit(options.getShowsFrequencyLimit().getMinLimit())
                                                .withFrequencyLimitIsForCampaignTime(
                                                        options.getShowsFrequencyLimit().getFrequencyLimitIsForCampaignTime()));
                    }
                    return gdPricePackage;
                }
        );
    }

    public static GdPiPage toGdPiPage(Placement<? extends PlacementBlock> p) {
        return new GdPiPage()
                .withId(p.getId())
                .withDomain(p.getDomain())
                .withCaption(p.getCaption())
                .withIsYandexPage(p.isYandexPage())
                .withIsDeleted(p.isDeleted())
                .withMirrors(StreamEx.of(p.getMirrors()).append(p.getDomain()).distinct().collect(Collectors.toList()))
                .withIsTesting(p.isTesting());
    }

    public static GdTargetTag toGdTargetTag(TargetTag targetTag) {
        return new GdTargetTag()
                .withId(targetTag.getId())
                .withName(targetTag.getName())
                .withDescription(targetTag.getDescription());
    }

    public static List<GdPricePackageForClient> toGdPricePackagesForClient(List<PricePackageForClient> pricePackages,
                                                                           Map<Long, Placement> pagesToPlacements,
                                                                           UnaryOperator<List<Long>> geoToFrontendGeo,
                                                                           Function<Long, WebGoalType> cryptaSegmentsSupplier,
                                                                           Supplier<List<GdPricePackagesCreativeTemplatesItem>> creativeTemplatesSupplier,
                                                                           boolean isBrandLiftHiddenEnabled) {
        return mapList(pricePackages, pp -> toGdPricePackageForClient(pp, pagesToPlacements, geoToFrontendGeo,
                cryptaSegmentsSupplier, creativeTemplatesSupplier, isBrandLiftHiddenEnabled));
    }

    public static GdPricePackageForClient toGdPricePackageForClient(PricePackageForClient pricePackage,
                                                                    Map<Long, Placement> pagesToPlacements,
                                                                    UnaryOperator<List<Long>> geoToFrontendGeo,
                                                                    Function<Long, WebGoalType> cryptaSegmentsSupplier,
                                                                    Supplier<List<GdPricePackagesCreativeTemplatesItem>> creativeTemplatesSupplier,
                                                                    boolean isBrandLiftHiddenEnabled) {
        var gdPricePackage = new GdPricePackage()
                .withId(pricePackage.getId())
                .withTitle(pricePackage.getTitle())
                .withPrice(pricePackage.getPrice())
                .withCurrency(pricePackage.getCurrency())
                .withOrderVolumeMin(pricePackage.getOrderVolumeMin())
                .withOrderVolumeMax(pricePackage.getOrderVolumeMax())
                .withPriceMarkups(toGdPriceMarkups(pricePackage.getPriceMarkups()))
                .withTargetingMarkups(toGdTargetingMarkups(pricePackage.getTargetingMarkups()))
                .withTargetingsFixed(toGdTargetingsFixed(pricePackage.getTargetingsFixed(),
                        geoToFrontendGeo, cryptaSegmentsSupplier))
                .withTargetingsCustom(toGdTargetingsCustom(pricePackage.getTargetingsCustom(),
                        geoToFrontendGeo, cryptaSegmentsSupplier))
                .withDateStart(pricePackage.getDateStart())
                .withDateEnd(pricePackage.getDateEnd())
                .withIsSpecial(pricePackage.getIsSpecial())
                .withIsCpd(pricePackage.getIsCpd())
                .withCampaignAutoApprove(pricePackage.getCampaignAutoApprove())
                .withAvailableAdGroupTypes(toGdAvailableAdGroupTypes(pricePackage))
                .withAllowedPageIds(pricePackage.getAllowedPageIds())
                .withAllowedProjectParamConditions(pricePackage.getAllowedProjectParamConditions())
                .withAllowedDomains(toGdAllowedDomains(pricePackage.getAllowedSsp(), pricePackage.getAllowedDomains()))
                .withAllowedTargetTags(pricePackage.getAllowedTargetTags())
                .withAllowedOrderTags(ifNotNull(pricePackage.getAllowedOrderTags(), Set::copyOf))
                .withPlacements(nvl(pricePackage.getAllowedPageIds(), emptyList()).stream()
                        .filter(pagesToPlacements::containsKey)
                        .map(e -> toGdPiPage(pagesToPlacements.get(e))).collect(Collectors.toList()))
                .withBidModifiers(toGdBidModifiers(pricePackage.getBidModifiers()))
                .withAllowedCreativeTemplates(
                        toGdAllowedCreativeTemplates(pricePackage.getAllowedCreativeTemplates(),
                                creativeTemplatesSupplier))
                .withCategoryId(pricePackage.getCategoryId());
        if (pricePackage.getCampaignOptions() != null) {
            var options = pricePackage.getCampaignOptions();
            gdPricePackage
                    .withAllowBrandSafety(options.getAllowBrandSafety())
                    .withAllowDisabledPlaces(options.getAllowDisabledPlaces())
                    .withAllowDisabledVideoPlaces(options.getAllowDisabledVideoPlaces())
                    .withAllowBrandLift(options.getAllowBrandLift())
                    .withAllowImage(options.getAllowImage())
                    .withAllowGallery(options.getAllowGallery())
                    .withShowsFrequencyLimit(options.getShowsFrequencyLimit() == null ? null :
                            new GdShowsFrequencyLimit()
                                    .withFrequencyLimit(options.getShowsFrequencyLimit().getFrequencyLimit())
                                    .withFrequencyLimitDays(options.getShowsFrequencyLimit().getFrequencyLimitDays())
                                    .withMinLimit(options.getShowsFrequencyLimit().getMinLimit())
                                    .withFrequencyLimitIsForCampaignTime(
                                            options.getShowsFrequencyLimit().getFrequencyLimitIsForCampaignTime()));
        }
        // для операторов с фичей скрытого BL разрешаем заводить BL на любых пакетах
        if (isBrandLiftHiddenEnabled) {
            gdPricePackage.withAllowBrandLift(true);
        }
        if (!pricePackage.getAvailableAdGroupTypes().contains(AdGroupType.CPM_YNDX_FRONTPAGE)
                && !pricePackage.getIsFrontpage()
                && gdPricePackage.getTargetingsFixed() != null) {
            // если не на главной, то не показываем список площадок
            // аналогично CampaignWithPricePackageAddOperationSupport для создания кампании
            // прайсовое видео не может показываться на этих площадках
            gdPricePackage.getTargetingsFixed().setViewTypes(emptyList());
        }
        return gdPricePackage;
    }

    public static PricePackagesFilter toCorePricePackageFilter(GdGetPricePackagesFilter gdGetPricePackagesFilter) {
        return new PricePackagesFilter()
                .withPackageIdIn(gdGetPricePackagesFilter.getPackageIdIn())
                .withTitleContains(gdGetPricePackagesFilter.getTitleContains())
                .withBusinessUnitIdIn(toCoreBusinessUnitIdIn(gdGetPricePackagesFilter.getBusinessUnitIdIn()))
                .withIsNotBusinessUnit(containsNotBusinessUnitId(gdGetPricePackagesFilter.getBusinessUnitIdIn()))
                .withIsPublic(gdGetPricePackagesFilter.getIsPublic())
                .withIsArchived(gdGetPricePackagesFilter.getIsArchived())
                .withIsSpecial(gdGetPricePackagesFilter.getIsSpecial())
                .withClientIn(gdGetPricePackagesFilter.getClientIn())
                .withRegionIds(gdGetPricePackagesFilter.getRegionIds())
                .withActivityIntervals(toCoreActivityIntervals(gdGetPricePackagesFilter.getActivityIntervals()))
                .withMinPrice(gdGetPricePackagesFilter.getMinPrice())
                .withMaxPrice(gdGetPricePackagesFilter.getMaxPrice())
                .withCurrencyIn(gdGetPricePackagesFilter.getCurrencyIn())
                .withStatusApproveIn(toCoreStatusApproveIn(gdGetPricePackagesFilter.getStatusApproveIn()));
    }

    public static PricePackagesFilter toCorePricePackagesForClientFilter(
            GdGetPricePackagesForClientFilter gdGetPricePackagesForClientFilter) {
        return new PricePackagesFilter()
                .withIsSpecial(gdGetPricePackagesForClientFilter.getIsSpecial())
                .withRegionIds(gdGetPricePackagesForClientFilter.getRegionIds())
                .withPlatforms(toCorePlatforms(gdGetPricePackagesForClientFilter.getPlatforms()))
                .withFormats(toCoreAvailableAdGroupTypes(gdGetPricePackagesForClientFilter.getFormats()));
    }

    public static PricePackageOrderBy toCorePricePackageOrderBy(GdPricePackageOrderBy orderBy) {
        return new PricePackageOrderBy()
                .withField(toCorePricePackageOrderByField(orderBy.getField()))
                .withOrder(orderBy.getOrder());
    }

    private static PricePackageOrderByField toCorePricePackageOrderByField(GdPricePackageOrderByField field) {
        switch (field) {
            case TITLE:
                return PricePackageOrderByField.TITLE;
            case STATUS_APPROVE:
                return PricePackageOrderByField.STATUS_APPROVE;
            case PRICE:
                return PricePackageOrderByField.PRICE;
            default:
                throw new IllegalArgumentException("No conversion for " + field);
        }
    }

    private static Set<Long> toCoreBusinessUnitIdIn(@Nullable Set<Long> businessUnitIdIn) {
        if (businessUnitIdIn == null) {
            return null;
        }
        // Убираем фиктивный id=-1, если есть, обозначающий продукты без БЮ.
        return businessUnitIdIn.stream()
                .filter(buId -> !NOT_BUSINESS_UNIT_ID.equals(buId))
                .collect(Collectors.toSet());
    }

    private static boolean containsNotBusinessUnitId(@Nullable Set<Long> businessUnitIdIn) {
        return ifNotNullOrDefault(businessUnitIdIn, buIdIn -> buIdIn.contains(NOT_BUSINESS_UNIT_ID), false);
    }

    private static List<LocalDateRange> toCoreActivityIntervals(
            @Nullable List<GdDateRange> activityIntervals) {
        if (activityIntervals == null) {
            return null;
        }
        return activityIntervals.stream()
                .map(gdInterval -> new LocalDateRange()
                        .withFromInclusive(gdInterval.getMin())
                        .withToInclusive(gdInterval.getMax()))
                .collect(Collectors.toList());
    }

    private static Set<StatusApprove> toCoreStatusApproveIn(
            @Nullable Set<GdStatusApprove> statusApproveIn) {
        if (statusApproveIn == null) {
            return null;
        }
        return statusApproveIn.stream()
                .map(GdStatusApprove::toSource)
                .collect(Collectors.toSet());
    }

    private static Set<PricePackagePlatform> toCorePlatforms(@Nullable Set<GdPricePackagePlatform> platforms) {
        if (platforms == null) {
            return null;
        }
        return platforms.stream()
                .map(GdPricePackagePlatform::toSource)
                .collect(Collectors.toSet());
    }

    public static GdPricePackagesCreativeTemplatesItem toGdPricePackagesBannerStorageCreativeTemplatesItem(
            BannerStorageDictLayoutItem item) {
        return new GdPricePackagesCreativeTemplatesItem()
                .withCreativeType(GdCreativeType.BANNERSTORAGE)
                .withName(templateNameNotNull(item.getName(), item.getId()))
                .withId(item.getId());
    }

    /**
     * Имя не должно быть пустым. Если не задано, то возвращаем id.
     */
    private static String templateNameNotNull(String name, Long id) {
        if (name == null) {
            return "template " + id;
        }
        return name;
    }

    public static GdPricePackagesCreativeTemplatesItem toGdPricePackagesVideoPresetsCreativeTemplatesItem(
            Preset item, GdCreativeType gdCreativeType) {
        return new GdPricePackagesCreativeTemplatesItem()
                .withCreativeType(gdCreativeType)
                .withIsAdaptive(item.getIsAdaptive())
                .withName(templateNameNotNull(item.getPresetName(), item.getPresetId()))
                .withId(item.getPresetId());
    }

    public static GdPricePackagesCreativeTemplatesItem toGdPricePackagesHtml5PresetsCreativeTemplatesItem(
            Preset item, GdCreativeType gdCreativeType) {
        return new GdPricePackagesCreativeTemplatesItem()
                .withCreativeType(gdCreativeType)
                .withName(templateNameNotNull(item.getPresetName(), item.getPresetId()))
                .withId(item.getPresetId());
    }

    public static GdPriceAllowedCreativeTemplates toGdAllowedCreativeTemplates(Map<CreativeType, List<Long>> map,
                                                                               Supplier<List<GdPricePackagesCreativeTemplatesItem>> creativeTemplatesSupplier) {
        if (map == null) {
            return null;
        }
        List<GdPricePackagesCreativeTemplatesItem> dict = creativeTemplatesSupplier.get();
        Map<GdCreativeType, List<Long>> gdCreativeTemplates = new HashMap<>();
        List<GdPricePackagesCreativeTemplatesItem> creativeTemplates = new ArrayList<>();
        map.forEach(
                (creativeType, ids) -> {
                    var gdCreativeType = toGdCreativeType(creativeType);
                    gdCreativeTemplates.put(gdCreativeType, ids);
                    creativeTemplates.addAll(
                            dict.stream()
                                    .filter(it -> it.getCreativeType().equals(gdCreativeType) &&
                                            ids != null && ids.contains(it.getId()))
                                    .collect(Collectors.toList())
                    );
                }
        );
        return new GdPriceAllowedCreativeTemplates()
                .withCreativeTemplateIds(gdCreativeTemplates)
                .withCreativeTemplates(creativeTemplates);
    }

    private static List<BidModifier> toCoreBidModifiers(@Nullable GdPricePackageBidModifiers gdBidModifiers) {
        if (gdBidModifiers == null) {
            return null;
        }
        List<BidModifier> coreBidModifiers = new ArrayList<>();
        var inventoryModifiers = createInventoryBidModifier(gdBidModifiers);
        if (inventoryModifiers != null) {
            coreBidModifiers.add(inventoryModifiers);
        }

        var platformModifiers = createPlatformBidModifiers(gdBidModifiers);
        if (platformModifiers != null) {
            coreBidModifiers.addAll(platformModifiers);
        }

        return coreBidModifiers;
    }

    private static BidModifier createInventoryBidModifier(@Nullable GdPricePackageBidModifiers gdBidModifiers) {
        var gdBidModifierInventoryFixed = gdBidModifiers.getBidModifierInventoryFixed();
        var gdBidModifierInventoryAll = gdBidModifiers.getBidModifierInventoryAll();

        if (gdBidModifierInventoryAll != null && !gdBidModifierInventoryAll.isEmpty()) {
            List<BidModifierInventoryAdjustment> inventoryAdjustments = new ArrayList<>();
            gdBidModifierInventoryAll.forEach(gdType -> {
                inventoryAdjustments.add(new BidModifierInventoryAdjustment()
                        .withInventoryType(toInventoryType(gdType))
                        .withIsRequiredInPricePackage(gdBidModifierInventoryFixed != null
                                && gdBidModifierInventoryFixed.contains(gdType)));
            });
            return new BidModifierInventory()
                    .withType(BidModifierType.INVENTORY_MULTIPLIER)
                    .withInventoryAdjustments(inventoryAdjustments);
        }
        return null;
    }

    private static Set<BidModifier> createPlatformBidModifiers(@Nullable GdPricePackageBidModifiers gdBidModifiers) {

        var gdBidModifierPlatformFixed = gdBidModifiers.getBidModifierPlatformFixed();
        var gdBidModifierPlatformAll = gdBidModifiers.getBidModifierPlatformAll();

        if (gdBidModifierPlatformAll != null && !gdBidModifierPlatformAll.isEmpty()) {
            return mapSet(gdBidModifierPlatformAll,
                    platform -> createOnePlatformBidModifier(platform, gdBidModifierPlatformFixed.contains(platform)));
        }
        return null;
    }

    private static BidModifier createOnePlatformBidModifier(GdPricePackagePlatformBidModifier platform,
                                                            boolean isRequired) {
        switch (platform) {
            case DESKTOP:
                return new BidModifierDesktop()
                        .withType(BidModifierType.DESKTOP_MULTIPLIER)
                        .withDesktopAdjustment(new BidModifierDesktopAdjustment()
                                .withIsRequiredInPricePackage(isRequired));
            case MOBILE:
                return new BidModifierMobile()
                        .withType(BidModifierType.MOBILE_MULTIPLIER)
                        .withMobileAdjustment(new BidModifierMobileAdjustment()
                                .withIsRequiredInPricePackage(isRequired));
            case IOS:
                return new BidModifierMobile()
                        .withType(BidModifierType.MOBILE_MULTIPLIER)
                        .withMobileAdjustment(new BidModifierMobileAdjustment()
                                .withOsType(OsType.IOS)
                                .withIsRequiredInPricePackage(isRequired));
            case ANDROID:
                return new BidModifierMobile()
                        .withType(BidModifierType.MOBILE_MULTIPLIER)
                        .withMobileAdjustment(new BidModifierMobileAdjustment()
                                .withOsType(OsType.ANDROID)
                                .withIsRequiredInPricePackage(isRequired));
            default:
                return null;
        }
    }

    public static GdPricePackageBidModifiers toGdBidModifiers(@Nullable List<BidModifier> bidModifiers) {
        if (bidModifiers == null) {
            return null;
        }

        var gdBidModifiers = new GdPricePackageBidModifiers();
        gdBidModifiers.setBidModifierInventoryFixed(new HashSet<>());
        gdBidModifiers.setBidModifierInventoryAll(new HashSet<>());
        gdBidModifiers.setBidModifierPlatformFixed(new HashSet<>());
        gdBidModifiers.setBidModifierPlatformAll(new HashSet<>());

        for (BidModifier bidModifier : bidModifiers) {
            if (bidModifier.getType().equals(BidModifierType.INVENTORY_MULTIPLIER)) {
                // Все корректировки на инвентарь
                gdBidModifiers.setBidModifierInventoryAll(
                        listToSet(
                                ((BidModifierInventory) bidModifier).getInventoryAdjustments(),
                                adj -> toGdInventoryType(adj.getInventoryType())));

                // обозначем только зафиксированные на пакете инструменты
                gdBidModifiers.setBidModifierInventoryFixed(
                        listToSet(
                                filterList(((BidModifierInventory) bidModifier).getInventoryAdjustments(),
                                        adj -> nvl(adj.getIsRequiredInPricePackage(), true)),
                                adj -> toGdInventoryType(adj.getInventoryType())));
            }

            Set<GdPricePackagePlatformBidModifier> bidModifierPlatformAll = gdBidModifiers.getBidModifierPlatformAll();
            Set<GdPricePackagePlatformBidModifier> bidModifierPlatformFixed =
                    gdBidModifiers.getBidModifierPlatformFixed();

            if (bidModifier.getType().equals(BidModifierType.DESKTOP_MULTIPLIER)) {
                bidModifierPlatformAll.add(GdPricePackagePlatformBidModifier.DESKTOP);
                if (nvl(((BidModifierDesktop) bidModifier).getDesktopAdjustment().getIsRequiredInPricePackage(),
                        true)) {
                    bidModifierPlatformFixed.add(GdPricePackagePlatformBidModifier.DESKTOP);
                }
            }

            if (bidModifier.getType().equals(BidModifierType.MOBILE_MULTIPLIER)) {

                BidModifierMobileAdjustment mobileAdjustment = ((BidModifierMobile) bidModifier).getMobileAdjustment();
                OsType osType = mobileAdjustment.getOsType();
                boolean isRequired = nvl(mobileAdjustment.getIsRequiredInPricePackage(), true);

                if (osType == null) {
                    bidModifierPlatformAll.add(GdPricePackagePlatformBidModifier.MOBILE);
                    if (isRequired) {
                        bidModifierPlatformFixed.add(GdPricePackagePlatformBidModifier.MOBILE);
                    }
                }
                if (osType == OsType.IOS) {
                    bidModifierPlatformAll.add(GdPricePackagePlatformBidModifier.IOS);
                    if (isRequired) {
                        bidModifierPlatformFixed.add(GdPricePackagePlatformBidModifier.IOS);
                    }
                }
                if (osType == OsType.ANDROID) {
                    bidModifierPlatformAll.add(GdPricePackagePlatformBidModifier.ANDROID);
                    if (isRequired) {
                        bidModifierPlatformFixed.add(GdPricePackagePlatformBidModifier.ANDROID);
                    }
                }
            }
        }

        return gdBidModifiers;
    }

    public static Set<String> toGdAllowedDomains(List<String> allowedSsp, List<String> allowedDomains) {
        Set<String> set = new HashSet<>();
        if (allowedDomains != null) {
            set.addAll(allowedDomains);
        }
        if (allowedSsp != null) {
            set.addAll(allowedSsp);
        }
        return set;
    }

    private List<String> toCoreAllowedSsp(Set<String> gdAllowedDomains) {
        return filterAndMapList(gdAllowedDomains,
                it -> context.getSspPlatforms().stream().anyMatch(s -> s.equalsIgnoreCase(it)),
                it -> context.getSspPlatforms().stream().filter(s -> s.equalsIgnoreCase(it))
                        .findFirst().orElse(null));
    }

    private List<String> toCoreAllowedDomains(Set<String> gdAllowedDomains) {
        return filterList(gdAllowedDomains, it -> context.getSspPlatforms().stream()
                .noneMatch(s -> s.equalsIgnoreCase(it)));
    }

    public static GdPricePackageCategory toGdPricePackageCategory(PricePackageCategory category) {
        return new GdPricePackageCategory()
                .withId(category.getId())
                .withParentId(category.getParentId())
                .withOrder(category.getOrder())
                .withTitle(category.getTitle())
                .withDescription(category.getDescription());
    }

}
