package ru.yandex.direct.grid.processing.service.product;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.product.model.ConditionName;
import ru.yandex.direct.core.entity.product.model.Product;
import ru.yandex.direct.core.entity.product.model.ProductRestriction;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.processing.model.product.GdProduct;
import ru.yandex.direct.grid.processing.model.product.GdProductCalcType;
import ru.yandex.direct.grid.processing.model.product.GdProductContainer;
import ru.yandex.direct.grid.processing.model.product.GdProductRestriction;
import ru.yandex.direct.grid.processing.model.product.GdProductRestrictionCondition;
import ru.yandex.direct.grid.processing.model.product.GdProductType;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.feature.FeatureName.CONTENT_DURATION_BID_MODIFIER;
import static ru.yandex.direct.feature.FeatureName.CPM_GEO_PIN_PRODUCT_ENABLED;
import static ru.yandex.direct.feature.FeatureName.CPM_VIDEO_GROUPS_EDIT_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.CPM_YNDX_FRONTPAGE_PROFILE;
import static ru.yandex.direct.feature.FeatureName.DISABLE_CPM_AUDIO;
import static ru.yandex.direct.feature.FeatureName.DISABLE_CPM_VIDEO;
import static ru.yandex.direct.feature.FeatureName.EDIT_CPM_YNDX_FRONTPAGE_IN_DNA;
import static ru.yandex.direct.feature.FeatureName.ENABLE_NON_SKIPPABLE_VIDEO_15_SECONDS;
import static ru.yandex.direct.feature.FeatureName.ENABLE_NON_SKIPPABLE_VIDEO_20_SECONDS;
import static ru.yandex.direct.feature.FeatureName.IN_BANNER_CREATIVES_SUPPORT;
import static ru.yandex.direct.grid.model.entity.adgroup.AdGroupTypeConverter.toGdAdGroupType;
import static ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType.CPM_AUDIO;
import static ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType.CPM_GEOPRODUCT;
import static ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType.CPM_GEO_PIN;
import static ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType.CPM_OUTDOOR;
import static ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType.CPM_VIDEO;
import static ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType.CPM_YNDX_FRONTPAGE;
import static ru.yandex.direct.grid.processing.model.product.GdProductType.CPM_BANNER;
import static ru.yandex.direct.grid.processing.service.product.converter.ProductConverter.coreProductCalcTypeToGd;
import static ru.yandex.direct.grid.processing.service.product.converter.ProductConverter.coreProductTypeToGd;
import static ru.yandex.direct.grid.processing.service.product.converter.ProductConverter.coreProductUnitToGd;
import static ru.yandex.direct.grid.processing.service.product.converter.ProductConverter.gdProductCalcTypeToCore;
import static ru.yandex.direct.grid.processing.service.product.converter.ProductConverter.gdProductTypeToCore;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, возвращающий данные о продуктах
 */
@Service
@ParametersAreNonnullByDefault
public class ProductDataService {

    private static final Collection<String> CPM_YNDX_FRONTPAGE_USER_PROFILE_CONDITIONS =
            ImmutableList.of(ConditionName.SOCIAL_DEMO.name().toLowerCase(),
                    ConditionName.FAMILY.name().toLowerCase(), ConditionName.BEHAVIORS.name().toLowerCase(),
                    ConditionName.INTERESTS.name().toLowerCase(), ConditionName.AUDIENCE.name().toLowerCase());

    private final AdGroupService adGroupService;
    private final ProductService productService;
    private final FeatureService featureService;
    private final CampaignService campaignService;
    private final ClientService clientService;

    @Autowired
    public ProductDataService(AdGroupService adGroupService,
                              ProductService productService,
                              FeatureService featureService,
                              CampaignService campaignService,
                              ClientService clientService) {
        this.adGroupService = adGroupService;
        this.productService = productService;
        this.featureService = featureService;
        this.campaignService = campaignService;
        this.clientService = clientService;
    }

    /**
     * Возвращает продукты, выбранные фильтром.
     * Дополнительно, фильтрует продукты по квазивалютности клиента
     * (квазивалютному клиенту возвращаются только квазивалютные продукты,
     * неквазивалютному &mdash; только неквазивалютные)
     */
    List<GdProduct> getProducts(ClientId clientId, GdProductContainer input) {
        Long id = input.getFilter().getId();
        Long campaignId = input.getFilter().getCampaignId();
        GdProductCalcType calcType = input.getFilter().getCalcType();
        CurrencyCode currencyCode = input.getFilter().getCurrency();
        GdProductType productType = input.getFilter().getType();

        Client client = clientService.getClient(clientId);

        List<Product> products = productService.getProducts();
        Map<Long, List<ProductRestriction>> productRestrictions = productService.getProductRestrictions();

        @SuppressWarnings("ConstantConditions")
        List<GdProduct> gdProducts = products.stream()
                .filter(p -> id == null || p.getId().equals(id))
                .filter(p -> calcType == null || p.getCalcType() == gdProductCalcTypeToCore(calcType))
                .filter(p -> currencyCode == null || p.getCurrencyCode() == currencyCode)
                .filter(p -> productType == null || p.getType() == gdProductTypeToCore(productType))
                .filter(p -> filterQuasiCurrencyUnitName(p, client))
                .map(p -> toGdProduct(productRestrictions, p))
                .collect(toList());

        Set<String> availableFeatures = featureService.getEnabledForClientId(clientId);

        //Фильтруем ограничения зависящие от фич
        gdProducts.forEach(p -> p.setRestrictions(filterList(p.getRestrictions(),
                r -> isRestrictionAllowed(availableFeatures, r))));

        //Фильтруем условия зависящие от фич
        gdProducts.forEach(p -> p.getRestrictions().forEach(r -> r.setConditions(filterList(r.getConditions(),
                c -> isConditionAllowed(availableFeatures, r, c)))));

        //Делаем часть групп неактивными в зависимости от содержимого кампании
        if (campaignId != null && productType == CPM_BANNER) {
            disallowMixingGeoproductAndOtherGroups(clientId, campaignId, gdProducts);
            avoidUsingAnyGroupTypesExceptCpmVideoWithCpvStrategy(clientId, campaignId, gdProducts);
        }

        return gdProducts;
    }

    private boolean filterQuasiCurrencyUnitName(Product product, Client client) {
        boolean quasiCurrencyFlag = ProductService.getClientQuasiCurrencyFlag(client);
        return ProductService.checkUnitNameOnQuasiCurrency(quasiCurrencyFlag, product);
    }

    /**
     * Фильтрация для {@link GdProductType#CPM_BANNER}.
     * <p/>
     * Правила:<ul>
     * <li>Если есть хотя бы одна группа геопродукта - разрешен только геопродукт, остальные неактивны</li>
     * <li>Если есть хотя бы одна не-гео группа - геопродукт неактивен, остальные активны</li>
     * <li>Если кампания пустая - разрешены любые группы</li>
     * </ul>
     */
    void disallowMixingGeoproductAndOtherGroups(ClientId clientId,
                                                Long campaignId,
                                                List<GdProduct> gdProducts) {
        Set<AdGroupType> adGroupTypes = adGroupService
                .getSimpleAdGroupsByCampaignIds(clientId, Set.of(campaignId))
                .getOrDefault(campaignId, emptyList()).stream()
                .map(AdGroupSimple::getType)
                .collect(toSet());
        if (!adGroupTypes.isEmpty()) {
            boolean hasGeoGroup =
                    adGroupTypes.contains(AdGroupType.CPM_GEOPRODUCT) || adGroupTypes.contains(AdGroupType.CPM_GEO_PIN);
            gdProducts.forEach(p -> p.getRestrictions()
                    .forEach(r -> r.setActive(r.getActive() && (hasGeoGroup == List.of(CPM_GEOPRODUCT, CPM_GEO_PIN).contains(r.getGroupType())))));
        }
    }

    /**
     * Фильтрация для {@link GdProductType#CPM_BANNER}.
     * <p/>
     * Правила:<ul>
     * <li>Если на кампании CPV стратегия - разрешено только cpm_video, остальные неактивны</li>
     * </ul>
     */
    void avoidUsingAnyGroupTypesExceptCpmVideoWithCpvStrategy(ClientId clientId,
                                                              Long campaignId,
                                                              List<GdProduct> gdProducts) {
        boolean cpvStrategyUsed = campaignService.getCampaignsWithStrategies(clientId, singleton(campaignId))
                .stream()
                .anyMatch(c -> c.getStrategy().getStrategyName() == StrategyName.AUTOBUDGET_AVG_CPV
                        || c.getStrategy().getStrategyName() == StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD);
        if (cpvStrategyUsed) {
            gdProducts.forEach(p -> p.getRestrictions()
                    .forEach(r -> r.setActive(r.getGroupType() == CPM_VIDEO)));
        }
    }

    private boolean isRestrictionAllowed(Set<String> availableFeatures, GdProductRestriction r) {
        if ((!availableFeatures.contains(CPM_VIDEO_GROUPS_EDIT_FOR_DNA.getName()) || availableFeatures.contains(DISABLE_CPM_VIDEO.getName()))
                && r.getGroupType() == CPM_VIDEO) {
            return false;
        }

        if (availableFeatures.contains(DISABLE_CPM_AUDIO.getName())
                && r.getGroupType() == CPM_AUDIO) {
            return false;
        }

        if (!availableFeatures.contains(CPM_GEO_PIN_PRODUCT_ENABLED.getName())
                && r.getGroupType() == CPM_GEO_PIN) {
            return false;
        }

        if (!availableFeatures.contains(EDIT_CPM_YNDX_FRONTPAGE_IN_DNA.getName())
                && r.getGroupType() == CPM_YNDX_FRONTPAGE) {
            return false;
        }

        return true;
    }

    private boolean isConditionAllowed(Set<String> availableFeatures, GdProductRestriction r,
                                       GdProductRestrictionCondition c) {

        if (!availableFeatures.contains(IN_BANNER_CREATIVES_SUPPORT.getName())
                && r.getGroupType() == GdAdGroupType.CPM_BANNER && c.getName().equals("in_banner_tumbler")) {
            return false;
        }

        if (!availableFeatures.contains(ENABLE_NON_SKIPPABLE_VIDEO_15_SECONDS.getName())
                && !availableFeatures.contains(ENABLE_NON_SKIPPABLE_VIDEO_20_SECONDS.getName())
                && r.getGroupType() == CPM_VIDEO && c.getName().equals("non_skippable_video")) {
            return false;
        }

        if (c.getName().equalsIgnoreCase(ConditionName.CORRECTION_EXPRESS_CONTENT_DURATION.name())) {
            return availableFeatures.contains(CONTENT_DURATION_BID_MODIFIER.getName());
        }

        if (c.getName().equals("correction_traffic")) {
            return r.getGroupType() == CPM_OUTDOOR;
        }

        if (!availableFeatures.contains(CPM_YNDX_FRONTPAGE_PROFILE.getName()) && r.getGroupType() == CPM_YNDX_FRONTPAGE &&
                CPM_YNDX_FRONTPAGE_USER_PROFILE_CONDITIONS.contains(c.getName())) {
            return false;
        }

        return true;
    }

    private GdProduct toGdProduct(Map<Long, List<ProductRestriction>> productRestrictions, Product p) {
        return new GdProduct()
                .withId(p.getId())
                .withName(p.getProductName())
                .withPublicNameKey(p.getPublicNameKey())
                .withPublicDescriptionKey(p.getPublicDescriptionKey())
                .withType(coreProductTypeToGd(p.getType()))
                .withUnit(coreProductUnitToGd(p.getUnit()))
                .withUnitScale(p.getUnitScale())
                .withCalcType(coreProductCalcTypeToGd(p.getCalcType()))
                .withCurrency(p.getCurrencyCode())
                .withBusinessUnit(p.getBusinessUnit())
                .withBusinessUnitName(p.getBusinessUnitName())
                .withRestrictions(mapList(productRestrictions.getOrDefault(p.getId(), emptyList()),
                        r -> toGdProductRestriction(r, p)));
    }

    private GdProductRestriction toGdProductRestriction(ProductRestriction r, Product p) {
        return new GdProductRestriction()
                .withId(r.getId())
                .withGroupType(toGdAdGroupType(r.getGroupType(), p.getType()))
                .withPublicNameKey(r.getPublicNameKey())
                .withPublicDescriptionKey(r.getPublicDescriptionKey())
                .withConditions(r.getConditionJson() != null
                        ? Arrays.asList(JsonUtils.fromJson(r.getConditionJson(), GdProductRestrictionCondition[].class))
                        : emptyList())
                .withUnitCountMin(r.getUnitCountMin())
                .withUnitCountMax(r.getUnitCountMax())
                .withActive(true);
    }
}
