package ru.yandex.direct.grid.processing.service.product.converter;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.EnumBiMap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import ru.yandex.direct.core.entity.product.model.ProductCalcType;
import ru.yandex.direct.core.entity.product.model.ProductType;
import ru.yandex.direct.core.entity.product.model.ProductUnit;
import ru.yandex.direct.grid.processing.model.product.GdProductCalcType;
import ru.yandex.direct.grid.processing.model.product.GdProductType;
import ru.yandex.direct.grid.processing.model.product.GdProductUnit;

import static com.google.common.base.Preconditions.checkState;

@ParametersAreNonnullByDefault
public class ProductConverter {
    private static final BiMap<ProductCalcType, GdProductCalcType> PRODUCT_CALC_TYPE_MAP =
            Maps.unmodifiableBiMap(EnumBiMap.create(ImmutableMap.<ProductCalcType, GdProductCalcType>builder()
                    .put(ProductCalcType.CPM, GdProductCalcType.CPM)
                    .put(ProductCalcType.CPC, GdProductCalcType.CPC)
                    .build()));

    private static final BiMap<ProductType, GdProductType> PRODUCT_TYPE_MAP =
            Maps.unmodifiableBiMap(EnumBiMap.create(ImmutableMap.<ProductType, GdProductType>builder()
                    .put(ProductType.TEXT, GdProductType.TEXT)
                    .put(ProductType.MCB, GdProductType.MCB)
                    .put(ProductType.SOCDEM, GdProductType.SOCDEM)
                    .put(ProductType.ANTICONTEXT, GdProductType.ANTICONTEXT)
                    .put(ProductType.GEO, GdProductType.GEO)
                    .put(ProductType.MCB_PKG, GdProductType.MCB_PKG)
                    .put(ProductType.MCB_REGIONAL, GdProductType.MCB_REGIONAL)
                    .put(ProductType.MCB_TURKISH, GdProductType.MCB_TURKISH)
                    .put(ProductType.CPM_BANNER, GdProductType.CPM_BANNER)
                    .put(ProductType.CPM_DEALS, GdProductType.CPM_DEALS)
                    .put(ProductType.CPM_VIDEO, GdProductType.CPM_VIDEO)
                    .put(ProductType.CPM_OUTDOOR, GdProductType.CPM_OUTDOOR)
                    .put(ProductType.INTERNAL_DISTRIB, GdProductType.INTERNAL_DISTRIB)
                    .put(ProductType.INTERNAL_FREE, GdProductType.INTERNAL_FREE)
                    .put(ProductType.CPM_YNDX_FRONTPAGE, GdProductType.CPM_YNDX_FRONTPAGE)
                    .put(ProductType.CPM_INDOOR, GdProductType.CPM_INDOOR)
                    .put(ProductType.CPM_AUDIO, GdProductType.CPM_AUDIO)
                    .put(ProductType.CPM_PRICE, GdProductType.CPM_PRICE)
                    .put(ProductType.INTERNAL_AUTOBUDGET, GdProductType.INTERNAL_AUTOBUDGET)
                    .put(ProductType.AUTO_IMPORT, GdProductType.AUTO_IMPORT)
                    .build()));

    private static final BiMap<ProductUnit, GdProductUnit> PRODUCT_UNIT_MAP =
            Maps.unmodifiableBiMap(EnumBiMap.create(ImmutableMap.<ProductUnit, GdProductUnit>builder()
                    .put(ProductUnit.SHOWS, GdProductUnit.SHOWS)
                    .put(ProductUnit.CLICKS, GdProductUnit.CLICKS)
                    .put(ProductUnit.DAYS, GdProductUnit.DAYS)
                    .build()));

    private ProductConverter() {
    }

    public static ProductCalcType gdProductCalcTypeToCore(GdProductCalcType gdType) {
        checkState(PRODUCT_CALC_TYPE_MAP.inverse().containsKey(gdType), "Unexpected value: %s", gdType);
        return PRODUCT_CALC_TYPE_MAP.inverse().get(gdType);
    }

    public static GdProductCalcType coreProductCalcTypeToGd(ProductCalcType coreType) {
        checkState(PRODUCT_CALC_TYPE_MAP.containsKey(coreType), "Unexpected value: %s", coreType);
        return PRODUCT_CALC_TYPE_MAP.get(coreType);
    }

    public static ProductType gdProductTypeToCore(GdProductType gdType) {
        checkState(PRODUCT_TYPE_MAP.inverse().containsKey(gdType), "Unexpected value: %s", gdType);
        return PRODUCT_TYPE_MAP.inverse().get(gdType);
    }

    public static GdProductType coreProductTypeToGd(ProductType coreType) {
        checkState(PRODUCT_TYPE_MAP.containsKey(coreType), "Unexpected value: %s", coreType);
        return PRODUCT_TYPE_MAP.get(coreType);
    }

    public static GdProductUnit coreProductUnitToGd(ProductUnit coreType) {
        checkState(PRODUCT_UNIT_MAP.containsKey(coreType), "Unexpected value: %s", coreType);
        return PRODUCT_UNIT_MAP.get(coreType);
    }
}
