package ru.yandex.direct.grid.processing.service.promoextension

import org.dataloader.BatchLoaderEnvironment
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Scope
import org.springframework.context.annotation.ScopedProxyMode
import org.springframework.stereotype.Component
import org.springframework.web.context.WebApplicationContext
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData
import ru.yandex.direct.core.entity.promoextension.PromoExtensionRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtension
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider
import java.util.concurrent.CompletableFuture
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonKey
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate
import ru.yandex.direct.grid.processing.model.client.GdClientInfo
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionAction
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionAssociatedCampaign
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionPrimaryStatus
import ru.yandex.direct.grid.processing.service.client.ClientDataService
import ru.yandex.direct.grid.processing.service.promoextension.GdPromoExtensionConverter.toGdPromoExtension
import ru.yandex.direct.rbac.RbacService
import java.time.LocalDate

@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class PromoExtensionsDataLoader(
    gridContextProvider: GridContextProvider,
    private val promoExtensionRepository: PromoExtensionRepository,
) : GridBatchingDataLoader<Long, GdPromoExtension>() {
    companion object {
        val logger: Logger = LoggerFactory.getLogger(PromoExtensionsDataLoader::class.java)
    }

    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::getPromoExtensions)
    }

    private fun getPromoExtensions(
        ids: Set<Long>,
        environment: BatchLoaderEnvironment
    ): CompletableFuture<Map<Long, GdPromoExtension>> {
        if (ids.isEmpty()) {
            return CompletableFuture.completedFuture(emptyMap())
        }

        val context = environment.getContext<GridGraphQLContext>()
        val clientInfo = context.queriedClient
            ?: throw IllegalStateException("No queriedClient in graphql context to fetch promoextensions")

        val promoExtensions: List<GdPromoExtension> = promoExtensionRepository.getByIds(clientInfo.shard, ids)
            .map { toGdPromoExtension(it) }

        return CompletableFuture.completedFuture(promoExtensions.associateBy { it.id })
    }

    fun getAllClientPromoExtensions(clientId: ClientId): List<GdPromoExtension> {
        val result = promoExtensionRepository.getAllClientPromoExtensions(clientId)
            .map { toGdPromoExtension(it) }
        result.forEach { get().prime(it.id, it) }
        return result
    }

    fun getPromoExtensionById(id: Long): CompletableFuture<GdPromoExtension> = get().load(id)

    fun getMassPromoExtensionsByIds(
        ids: Set<Long>
    ): CompletableFuture<List<GdPromoExtension>> = get().loadMany(ids.toList()).thenApply { it.filterNotNull() }
}

@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class AssociatedCidsForPromoExtensionDataLoader(
    gridContextProvider: GridContextProvider,
    private val promoExtensionRepository: PromoExtensionRepository,
) : GridBatchingDataLoader<Long, List<Long>>() {
    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::getAssociatedCids)
    }

    private fun getAssociatedCids(
        promoExtensionIds: Set<Long>,
        environment: BatchLoaderEnvironment
    ): CompletableFuture<Map<Long, List<Long>>> {
        if (promoExtensionIds.isEmpty()) {
            return CompletableFuture.completedFuture(emptyMap())
        }
        val cidsByPromoIds = promoExtensionRepository.getCidsByPromoExtensionIds(promoExtensionIds)
        val result = promoExtensionIds.associateWith { cidsByPromoIds.get(it) ?: listOf() }
        return CompletableFuture.completedFuture(result)
    }
}

@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class AssociatedPromoExtensionCampaignsByCidDataLoader(
    gridContextProvider: GridContextProvider,
    private val campaignRepository: CampaignRepository,
) : GridBatchingDataLoader<Long, GdPromoExtensionAssociatedCampaign>() {
    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::getAssociatedCampaigns)
    }

    private fun getAssociatedCampaigns(
        cids: Set<Long>,
        environment: BatchLoaderEnvironment
    ): CompletableFuture<Map<Long, GdPromoExtensionAssociatedCampaign>> {
        if (cids.isEmpty()) {
            return CompletableFuture.completedFuture(emptyMap())
        }

        val context = environment.getContext<GridGraphQLContext>()
        val clientInfo = context.queriedClient
            ?: throw IllegalStateException("No queriedClient in graphql context to fetch promoextensions")

        val campaignsByIds = campaignRepository.getCampaignsSimple(clientInfo.shard, cids)
            .mapValues {
                GdPromoExtensionAssociatedCampaign(
                    cid = it.value.id,
                    name = it.value.name,
                )
            }
        return CompletableFuture.completedFuture(campaignsByIds)
    }
}

@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class PromoExtensionModerationRejectedReasonIdsDataLoader(
    gridContextProvider: GridContextProvider,
    private val moderationReasonService: ModerationReasonService,
) : GridBatchingDataLoader<Long, List<Long>>() {
    companion object {
        val logger: Logger = LoggerFactory.getLogger(PromoExtensionModerationRejectedReasonIdsDataLoader::class.java)
    }

    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::getModReasonIds)
    }

    private fun getModReasonIds(
        promoExtensionIds: Set<Long>,
        environment: BatchLoaderEnvironment
    ): CompletableFuture<Map<Long, List<Long>>> {
        if (promoExtensionIds.isEmpty()) {
            return CompletableFuture.completedFuture(emptyMap())
        }
        val context = environment.getContext<GridGraphQLContext>()
        val clientInfo = context.queriedClient
            ?: throw IllegalStateException("No queriedClient in graphql context to fetch promoextension mod reason ids")
        val keys = promoExtensionIds.map {
            ModerationReasonKey()
                .withObjectId(it)
                .withObjectType(ModerationReasonObjectType.PROMO_EXTENSION)
        }.toSet()
        val diagIdsByPromoExtensionIds = moderationReasonService
            .getRejectReasonDiagsByModerationKey(ClientId.fromLong(clientInfo.id), keys)
            .entries
            .associate { it.key.objectId to it.value.map { diag -> diag.id } }
        val result = promoExtensionIds.associateWith { diagIdsByPromoExtensionIds.get(it) ?: listOf() }
        return CompletableFuture.completedFuture(result)
    }
}

@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class PrimaryStatusPromoExtensionDataLoader(
    gridContextProvider: GridContextProvider,
    private val aggregatedStatusesViewService: AggregatedStatusesViewService,
    private val clientDataService: ClientDataService
) : GridBatchingDataLoader<GdPromoExtension, GdPromoExtensionPrimaryStatus>() {
    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::mapPromoToPrimaryStatus)
    }

    private fun mapPromoToPrimaryStatus(
        promoExtensions: Set<GdPromoExtension>,
        environment: BatchLoaderEnvironment
    ): CompletableFuture<Map<GdPromoExtension, GdPromoExtensionPrimaryStatus>> {
        val now = LocalDate.now()
        val context = environment.getContext<GridGraphQLContext>()
        val queriedClient = context.queriedClient
        val associatedCidsByPromoIds =
            (environment.keyContexts as Map<GdPromoExtension, Collection<Long>>).entries.associate { it.key.id to it.value }
        val aggregatedStatusesByCids = aggregatedStatusesViewService.getCampaignStatusesByIds(
            queriedClient!!.shard,
            associatedCidsByPromoIds.values.flatten().toSet()
        )
        return CompletableFuture.completedFuture(
            promoExtensions.associateWith { promoExtension ->
                getPrimaryStatus(
                    promoExtension,
                    associatedCidsByPromoIds[promoExtension.id]!!,
                    aggregatedStatusesByCids,
                    now
                )
            }
        )
    }

    private fun getPrimaryStatus(
        promoExtension: GdPromoExtension,
        associatedCidsForPromoExtension: Collection<Long>,
        campaignStatusesByIds: Map<Long, AggregatedStatusCampaignData>,
        now: LocalDate,
    ): GdPromoExtensionPrimaryStatus {
        return when (promoExtension.statusModerate) {
            PromoactionsStatusmoderate.Ready, PromoactionsStatusmoderate.Sending, PromoactionsStatusmoderate.Sent ->
                GdPromoExtensionPrimaryStatus.ON_MODERATION
            PromoactionsStatusmoderate.No -> GdPromoExtensionPrimaryStatus.DECLINED
            PromoactionsStatusmoderate.Yes ->
                if (promoExtension.finishDate?.isBefore(now) == true) {
                    GdPromoExtensionPrimaryStatus.FINISHED
                } else {
                    if (associatedCidsForPromoExtension.any { cid ->
                            campaignStatusesByIds[cid]?.status?.orElse(null)?.let { status ->
                                GdSelfStatusEnum.allRun().contains(status)
                            } == true
                        }
                    ) {
                        GdPromoExtensionPrimaryStatus.ACTIVE
                    } else {
                        GdPromoExtensionPrimaryStatus.NO_ACTIVE_CAMPAIGNS
                    }
                }
        }
    }
}

@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class AvailableActionsPromoExtensionDataLoader(
    gridContextProvider: GridContextProvider,
    private val promoExtensionRepository: PromoExtensionRepository,
    private val rbacService: RbacService,
) : GridBatchingDataLoader<Long, Set<GdPromoExtensionAction>>() {
    init {
        dataLoader = mappedDataLoader(gridContextProvider, this::mapPromoToAvailableActions)
    }

    private fun mapPromoToAvailableActions(
        promoExtensionIds: Set<Long>,
        environment: BatchLoaderEnvironment,
    ): CompletableFuture<Map<Long, Set<GdPromoExtensionAction>>> {
        val context = environment.getContext<GridGraphQLContext>()
        val queriedClient = context.queriedClient
        val clientPromoExtensions = promoExtensionRepository.getAllClientPromoExtensionIds(
            queriedClient.shard,
            ClientId.fromLong(queriedClient.id)
        )

        val associatedCidsByPromoIds =
            (environment.keyContexts as Map<Long, Collection<Long>>).entries.associate { it.key to it.value.isEmpty() }

        return CompletableFuture.completedFuture(
            promoExtensionIds.associateWith { promoExtensionId ->
                getAvailableActions(
                    promoExtensionId,
                    associatedCidsByPromoIds[promoExtensionId]!!,
                    clientPromoExtensions,
                    queriedClient,
                    context.operator.uid
                )
            }
        )
    }

    private fun getAvailableActions(
        promoExtensionId: Long,
        hasNoAssociatedCids: Boolean,
        clientPromoExtensionIds: Set<Long>,
        queriedClient: GdClientInfo,
        operatorUid: Long,
    ): Set<GdPromoExtensionAction> {
        val actions = mutableSetOf<GdPromoExtensionAction>()

        if (rbacService.canWrite(operatorUid, queriedClient.chiefUserId)
            && clientPromoExtensionIds.contains(promoExtensionId)
        ) {
            actions.add(GdPromoExtensionAction.EDIT_PROMO_EXTENSION)

            if (hasNoAssociatedCids) {
                actions.add(GdPromoExtensionAction.DELETE_PROMO_EXTENSION)
            }
        }
        return actions
    }
}
