package ru.yandex.direct.grid.processing.service.promoextension

import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLContext
import io.leangen.graphql.annotations.GraphQLMutation
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import java.util.Optional
import java.util.concurrent.CompletableFuture
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.core.entity.promoextension.PromoExtensionService
import ru.yandex.direct.core.entity.promoextension.model.PromoExtension
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.No
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.Ready
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.Sending
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.Sent
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.Yes
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.client.GdClient
import ru.yandex.direct.grid.processing.model.promoextension.GdAddPromoExtensionsInput
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionAction
import ru.yandex.direct.grid.processing.model.promoextension.GdDeletePromoExtensionsInput
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtension
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionAssociatedCampaign
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionContext
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionDeletePayload
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionFilter
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionPayload
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionPayloadItem
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionPrimaryStatus
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionStatus
import ru.yandex.direct.grid.processing.model.promoextension.GdPromoExtensionsContainer
import ru.yandex.direct.grid.processing.model.promoextension.GdUpdatePromoExtensionsInput
import ru.yandex.direct.grid.processing.model.promoextension.PreparedPromoExtensionQueryInput
import ru.yandex.direct.grid.processing.model.promoextension.PromoExtensionsCacheRecordInfo
import ru.yandex.direct.grid.processing.processor.util.GridRequestedFields
import ru.yandex.direct.grid.processing.processor.util.GridRequestedFieldsHolder
import ru.yandex.direct.grid.processing.service.cache.GridCacheService
import ru.yandex.direct.grid.processing.service.cache.util.CacheUtils
import ru.yandex.direct.grid.processing.service.client.ClientDataService
import ru.yandex.direct.grid.processing.service.promoextension.GdPromoExtensionConverter.toGdPromoExtension
import ru.yandex.direct.grid.processing.service.promoextension.PromoExtensionGraphQlServiceUtils.PROMO_EXTENSIONS_CONTAINER_VALIDATOR
import ru.yandex.direct.grid.processing.service.promoextension.PromoExtensionGraphQlServiceUtils.PROMO_EXTENSION_FILTER_PROCESSOR
import ru.yandex.direct.grid.processing.service.promoextension.PromoExtensionGraphQlServiceUtils.getComparator
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.grid.processing.util.ResponseConverter.getResults
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.validation.result.PathHelper

@GridGraphQLService
open class PromoExtensionsGraphQlService @Autowired constructor(
    private val promoExtensionsDataLoader: PromoExtensionsDataLoader,
    private val associatedCidsForPromoExtensionDataLoader: AssociatedCidsForPromoExtensionDataLoader,
    private val moderationRejectedReasonIdsDataLoader: PromoExtensionModerationRejectedReasonIdsDataLoader,
    private val associatedPromoExtensionCampaignsByCidDataLoader: AssociatedPromoExtensionCampaignsByCidDataLoader,
    private val promoExtensionService: PromoExtensionService,
    private val clientDataService: ClientDataService,
    private val gridValidationService: GridValidationService,
    private val primaryStatusPromoExtensionDataLoader: PrimaryStatusPromoExtensionDataLoader,
    private val gridShortenerService: GridShortenerService,
    private val gridCacheService: GridCacheService,
    private val availableActionsPromoExtensionDataLoader: AvailableActionsPromoExtensionDataLoader,
) {
    @PreAuthorizeRead
    @GraphQLQuery(name = "promoExtensions")
    open fun getPromoExtensions(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLContext client: GdClient,
        @GraphQLArgument(name = "input") input: GdPromoExtensionsContainer,
        @GridRequestedFields promoExtensionContextRequestedFields: GridRequestedFieldsHolder,
    ): CompletableFuture<GdPromoExtensionContext> {
        gridValidationService.applyValidator(PROMO_EXTENSIONS_CONTAINER_VALIDATOR, input, false)
        val clientId: ClientId = ClientId.fromLong(client.info.id)

        val preparedInput: PreparedPromoExtensionQueryInput = prepareInput(input, clientId)

        val recordInfo: PromoExtensionsCacheRecordInfo = preparedInput
            .toPromoExtensionsCacheRecordInfo(clientId, input.cacheKey)
        val res: Optional<GdPromoExtensionContext> = gridCacheService.getFromCache(recordInfo, preparedInput.range)
        if (res.isPresent) {
            return CompletableFuture.completedFuture(res.get())
        }

        return getUnfilteredUnsortedPromoExtensions(preparedInput.filter, clientId)
            .thenApply { promoExtensionsRowsetUnfiltered ->
                val promoExtensions = promoExtensionsRowsetUnfiltered
                    .filter { PROMO_EXTENSION_FILTER_PROCESSOR.test(preparedInput.filter, it) }
                    .sortedWith(getComparator(preparedInput.orderBy))
                val resultPromoExtensionContext = GdPromoExtensionContext(
                    totalCount = promoExtensions.size,
                    promoExtensionIds = promoExtensions.map { it.id },
                    filter = preparedInput.filter,
                    rowset = listOf(), //запишется ниже внутри getResultAndSaveToCacheIfRequested
                    cacheKey = null, //запишется ниже внутри getResultAndSaveToCacheIfRequested при необходимости
                )
                gridCacheService.getResultAndSaveToCacheIfRequested(recordInfo, resultPromoExtensionContext,
                    promoExtensions, preparedInput.range,
                    promoExtensionContextRequestedFields.isKPropertyPresent(GdPromoExtensionContext::cacheKey)
                )
            }
    }

    private fun prepareInput(
        userInput: GdPromoExtensionsContainer,
        clientId: ClientId
    ): PreparedPromoExtensionQueryInput {
        val range: LimitOffset = CacheUtils.normalizeLimitOffset(userInput.limitOffset)
        return if (userInput.filterKey != null) {
            val savedFilter: GdPromoExtensionFilter = gridShortenerService.getSavedFilter(userInput.filterKey, clientId,
                GdPromoExtensionFilter::class.java, { GdPromoExtensionFilter() })
            PreparedPromoExtensionQueryInput(savedFilter, userInput.orderBy.orEmpty(), range)
        } else {
            PreparedPromoExtensionQueryInput(requireNotNull(userInput.filter), userInput.orderBy.orEmpty(), range)
        }
    }

    /**
     * Получение промоакций по списку id
     * Если никаких идентификаторов передано, не было, возвращаем полный список акций клиента
     * Если передан пустой список id, вернём пустой список
     */
    private fun getUnfilteredUnsortedPromoExtensions(
        filter: GdPromoExtensionFilter,
        clientId: ClientId
    ): CompletableFuture<List<GdPromoExtension>> {
        return if (filter.promoExtensionIdIn == null) {
            CompletableFuture.completedFuture(promoExtensionsDataLoader.getAllClientPromoExtensions(clientId))
        } else {
            promoExtensionsDataLoader.getMassPromoExtensionsByIds(filter.promoExtensionIdIn)
        }
    }

    @PreAuthorizeRead
    @GraphQLQuery(name = "allClientPromoExtensions")
    open fun getAllClientPromoExtensions(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLContext client: GdClient,
    ): List<GdPromoExtension> {
        val clientId: ClientId = ClientId.fromLong(client.info.id)
        return promoExtensionsDataLoader.getAllClientPromoExtensions(clientId).sortedByDescending { it.id }
    }

    @GraphQLQuery(name = "status")
    open fun status(@GraphQLContext promoExtension: GdPromoExtension): GdPromoExtensionStatus {
        return when (promoExtension.statusModerate) {
            Ready, Sending, Sent -> GdPromoExtensionStatus.ON_MODERATION
            Yes -> GdPromoExtensionStatus.ACCEPTED
            No -> GdPromoExtensionStatus.DECLINED
        }
    }

    @GraphQLQuery(name = "primaryStatus")
    open fun primaryStatus(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLContext promoExtension: GdPromoExtension
    ): CompletableFuture<GdPromoExtensionPrimaryStatus> {
        return associatedCids(promoExtension)
            .thenCompose { primaryStatusPromoExtensionDataLoader.get().load(promoExtension, it) }
    }

    @GraphQLQuery(name = "moderationRejectedReasonIds")
    open fun modReasonIds(@GraphQLContext promoExtension: GdPromoExtension): CompletableFuture<List<Long>> {
        if (promoExtension.statusModerate == No) {
            return moderationRejectedReasonIdsDataLoader.get().load(promoExtension.id)
        } else {
            return CompletableFuture.completedFuture(listOf())
        }
    }

    @GraphQLQuery(name = "availableActions")
    open fun availableActions(@GraphQLContext promoExtension: GdPromoExtension): CompletableFuture<Set<GdPromoExtensionAction>> {
        return associatedCids(promoExtension)
            .thenCompose { availableActionsPromoExtensionDataLoader.get().load(promoExtension.id, it) }
    }

    @GraphQLQuery(name = "associatedCids")
    open fun associatedCids(@GraphQLContext promoExtension: GdPromoExtension): CompletableFuture<List<Long>> {
        return associatedCidsForPromoExtensionDataLoader.get().load(promoExtension.id)
    }

    @GraphQLQuery(name = "associatedCampaigns")
    open fun associatedCampaigns(
        @GraphQLContext promoExtension: GdPromoExtension,
    ): CompletableFuture<List<GdPromoExtensionAssociatedCampaign>> {
        return associatedCidsForPromoExtensionDataLoader.get().load(promoExtension.id)
            .thenCompose { associatedPromoExtensionCampaignsByCidDataLoader.get().loadMany(it) }
            .thenApply { it.filterNotNull() }
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addPromoExtensions")
    open fun addPromoExtensions(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdAddPromoExtensionsInput
    ): GdPromoExtensionPayload {
        context.queriedClient = getGdClientInfoForMutation(context)
        val promoExtensions = input.promoExtensions.map {
            PromoExtension(
                promoExtensionId = null,
                clientId = context.subjectUser!!.clientId,
                type = it.type.dbType,
                amount = it.amount,
                unit = it.unit?.coreValue,
                prefix = it.prefix?.dbPrefix,
                href = it.href,
                description = it.description,
                startDate = it.startDate,
                finishDate = it.finishDate,
                statusModerate = Ready,
            )
        }

        return convertAddOrUpdateOperationResult(
            promoExtensionService.addReturnResultWithPromos(context.subjectUser!!.clientId, promoExtensions),
            GdAddPromoExtensionsInput::promoExtensions.name,
            input.promoExtensions.size
        )
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updatePromoExtensions")
    open fun updatePromoExtensions(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdUpdatePromoExtensionsInput
    ): GdPromoExtensionPayload {
        context.queriedClient = getGdClientInfoForMutation(context)
        val promoExtensionChanges = input.promoExtensions.map {
            val ktModelChange = KtModelChanges<Long, PromoExtension>(it.id)
            ktModelChange.process(PromoExtension::type, it.type.dbType)
            ktModelChange.process(PromoExtension::amount, it.amount)
            ktModelChange.process(PromoExtension::unit, it.unit?.coreValue)
            ktModelChange.process(PromoExtension::prefix, it.prefix?.dbPrefix)
            ktModelChange.process(PromoExtension::href, it.href)
            ktModelChange.process(PromoExtension::description, it.description)
            ktModelChange.process(PromoExtension::startDate, it.startDate)
            ktModelChange.process(PromoExtension::finishDate, it.finishDate)
            ktModelChange
        }

        return convertAddOrUpdateOperationResult(
            promoExtensionService.updateReturnResultWithPromos(context.subjectUser!!.clientId, promoExtensionChanges),
            GdUpdatePromoExtensionsInput::promoExtensions.name,
            input.promoExtensions.size
        )
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "deletePromoExtensions")
    open fun deletePromoExtensions(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdDeletePromoExtensionsInput
    ): GdPromoExtensionDeletePayload {
        return convertDeleteOperationResult(promoExtensionService.delete(context.subjectUser!!.clientId, input.ids))
    }

    private fun convertAddOrUpdateOperationResult(
        massResult: MassResult<PromoExtension?>,
        inputFieldName: String,
        inputSize: Int,
    ): GdPromoExtensionPayload {
        val mutationResults = getResults(massResult,
            { promoExtension ->
                GdPromoExtensionPayloadItem(
                    promoExtension?.promoExtensionId,
                    promoExtension?.let { toGdPromoExtension(it) }
                )
            }
        ).map {
            it ?: GdPromoExtensionPayloadItem(null, null)
        }.ifEmpty {//пустой, если ошибка уровня операции. Сейчас только в случае превышения лимита на число акций
            List(inputSize) { GdPromoExtensionPayloadItem(null, null) }
        }
        val gdValidationResult = gridValidationService.getValidationResult(
            massResult, PathHelper.path(PathHelper.field(inputFieldName))
        )
        return GdPromoExtensionPayload()
            .withMutationResults(mutationResults)
            .withValidationResult(gdValidationResult)
    }

    private fun convertDeleteOperationResult(
        massResult: MassResult<Long?>,
    ): GdPromoExtensionDeletePayload {
        val mutationResults = getResults(massResult) { it }

        val gdValidationResult = gridValidationService.getValidationResult(
            massResult, PathHelper.path(PathHelper.field(GdDeletePromoExtensionsInput::ids.name))
        )
        return GdPromoExtensionDeletePayload()
            .withMutationResults(mutationResults)
            .withValidationResult(gdValidationResult)
    }

    private fun getGdClientInfoForMutation(context: GridGraphQLContext) =
        clientDataService.getClientInfo(context, setOf(context.subjectUser!!.clientId.asLong())).firstOrNull()
}
