package ru.yandex.direct.grid.processing.service.pub.freelancer;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.currency.Money;
import ru.yandex.direct.grid.processing.annotations.PublicGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdUserPublicInfo;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancer;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerBase;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFeedback;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFilter;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerRegion;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerSkill;
import ru.yandex.direct.grid.processing.service.freelancer.FreelancerDataService;
import ru.yandex.direct.grid.processing.service.freelancer.FreelancerSkillDataLoader;

/**
 * Публичный сервис про фирилансеров
 */
@PublicGraphQLService
@ParametersAreNonnullByDefault
public class FreelancerPublicGraphQlService {

    private final FreelancerDataService freelancerDataService;
    private final FreelancerSkillDataLoader freelancerSkillDataLoader;

    @Autowired
    public FreelancerPublicGraphQlService(
            FreelancerDataService freelancerDataService,
            FreelancerSkillDataLoader freelancerSkillDataLoader) {
        this.freelancerDataService = freelancerDataService;
        this.freelancerSkillDataLoader = freelancerSkillDataLoader;
    }

    /**
     * Получение списка фрилансеров с общедоступной информацией
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "freelancersList")
    public List<GdFreelancerBase> allFreelancers(@GraphQLArgument(name = "filter") GdFreelancerFilter filter) {
        return freelancerDataService.getFreelancers(filter, GdFreelancerBase::new, null);
    }

    @GraphQLQuery(name = "mainSkill")
    public GdFreelancerSkill getFreelancerMainSkill(@GraphQLContext GdFreelancer freelancer) {
        return freelancerDataService.getMainSkill(freelancer.getFreelancerId());
    }

    @GraphQLQuery(name = "feedbacks")
    public CompletableFuture<List<@GraphQLNonNull GdFreelancerFeedback>> getFreelancerFeedbacks(
            @GraphQLContext GdFreelancer freelancer) {
        return CompletableFuture
                .completedFuture(
                        freelancerDataService.getFreelancerFeedbacks(-1L, freelancer.getFreelancerId()));
    }

    /**
     * Регион фрилансера. Отдельным запросом подтягиваем данные о стране и названии региона фрилансера.
     */
    @GraphQLQuery(
            name = "region",
            description = "Данные о стране и названии региона фрилансера.")
    public CompletableFuture<GdFreelancerRegion> getFreelancerRegion(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancer freelancer,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        Long regionId = freelancer.getRegionId();
        return freelancerDataService.getRegionTranslationFuture(regionId, lang);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "skills")
    public CompletableFuture<List<@GraphQLNonNull GdFreelancerSkill>> getFreelancersSkills(
            @GraphQLContext GdFreelancer freelancer) {
        return freelancerSkillDataLoader.get().load(freelancer.getFreelancerId());
    }

    @GraphQLQuery(name = "authorInfo")
    public CompletableFuture<GdUserPublicInfo> getFreelancerFeedbacksAuthorInfo(
            @GraphQLContext GdFreelancerFeedback feedback) {
        return freelancerDataService.getFeedbackAuthorInfo(feedback);
    }

    /**
     * Добавляет {@link GdFreelancerSkill} свойство {@code priceMoney} типа {@link Money}
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "priceMoney")
    public Money getFreelancerSkillPriceMoney(@GraphQLContext GdFreelancerSkill skill) {
        return freelancerDataService.getSkillPriceMoney(skill);
    }
}
