package ru.yandex.direct.grid.processing.service.recommendation;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.gson.Gson;
import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.recommendation.model.KpiDailyBudget;
import ru.yandex.direct.core.entity.recommendation.model.KpiWeeklyBudget;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationQueueInfo;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.recommendation.GdExecuteRecommedations;
import ru.yandex.direct.grid.processing.model.recommendation.GdExecuteRecommedationsPayload;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKey;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiDailyBudget;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKpiWeeklyBudget;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytcore.entity.recommendation.service.RecommendationService;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isEmpty;

/**
 * Сервис, принимающий данные о принятых рекомендациях клиента и кладущий их в очередь
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class RecommendationMutationGraphQlService {
    private static final Gson GSON = new Gson();
    private static final Logger logger = LoggerFactory.getLogger(RecommendationMutationGraphQlService.class);

    private final GridRecommendationService gridRecommendationService;
    private final RecommendationService recommendationService;
    private final FeatureService featureService;

    @Autowired
    public RecommendationMutationGraphQlService(GridRecommendationService gridRecommendationService,
                                                RecommendationService recommendationService,
                                                FeatureService featureService) {
        this.gridRecommendationService = gridRecommendationService;
        this.recommendationService = recommendationService;
        this.featureService = featureService;
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "executeRecommendation")
    public GdExecuteRecommedationsPayload executeRecommendation(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdExecuteRecommedations input) {
        final Long clientId = context.getSubjectUser().getClientId().asLong();
        User operator = context.getOperator();

        List<RecommendationQueueInfo> recommendations = input.getItems().stream()
                .map(key -> convertToInternal(clientId, operator, key))
                .filter(Objects::nonNull)
                .collect(toList());

        logger.info("Executing recommendations: {}", JsonUtils.toJson(recommendations));

        Map<RecommendationQueueInfo, RecommendationStatus> recommendationStatuses
                = recommendationService.execute(clientId, operator.getUid(), recommendations);

        List<GdRecommendationKey> items = recommendationStatuses.entrySet().stream()
                .map(this::convertToExternal)
                .collect(toList());
        return new GdExecuteRecommedationsPayload().withItems(items);
    }

    @Nullable
    private RecommendationQueueInfo convertToInternal(Long clientId,
                                                      User operator, GdRecommendationKey key) {
        if ((GdiRecommendationType.dailyBudget.equals(key.getType()) && featureService
                .isEnabledForClientId(ClientId.fromLong(clientId),
                        FeatureName.DISABLE_OLD_DAILY_BUDGET_RECOMMENDATION)) ||
                (GdiRecommendationType.increaseStrategyWeeklyBudget.equals(key.getType()) && featureService
                        .isEnabledForClientId(ClientId.fromLong(clientId),
                                FeatureName.DISABLE_OLD_WEEKLY_BUDGET_RECOMMENDATION))
        ) {
            return new RecommendationQueueInfo()
                    .withClientId(clientId)
                    .withUid(operator.getUid())
                    .withType(key.getType().getId())
                    .withCampaignId(key.getCid())
                    .withAdGroupId(key.getPid())
                    .withBannerId(key.getBid())
                    .withUserKey1(key.getUserKey1())
                    .withUserKey2(key.getUserKey2())
                    .withUserKey3(key.getUserKey3())
                    .withTimestamp(key.getTimestamp())
                    .withSecTime(LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS))
                    .withQueueTime(LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS));
        }
        List<GdiRecommendation> list = gridRecommendationService.getAvailableRecommendations(clientId,
                operator, singleton(key.getType()),
                key.getCid() == null || key.getCid() == 0 ? null : singleton(key.getCid()),
                key.getPid() == null || key.getPid() == 0 ? null : singleton(key.getPid()),
                key.getBid() == null || key.getBid() == 0 ? null : singleton(key.getBid()),
                isEmpty(key.getUserKey1()) ? null : singleton(key.getUserKey1()),
                isEmpty(key.getUserKey2()) ? null : singleton(key.getUserKey2()),
                isEmpty(key.getUserKey3()) ? null : singleton(key.getUserKey3()));

        if (list.isEmpty()) {
            return null;
        }

        String json = list.get(0).getKpi();
        String convertedJson = null;
        if (json != null) {
            try {
                switch (key.getType()) {
                    case dailyBudget:
                        GdRecommendationKpiDailyBudget dataDailyBudget =
                                GSON.fromJson(json, GdRecommendationKpiDailyBudget.class);
                        convertedJson = GSON.toJson(new KpiDailyBudget()
                                .withCurrentDailyBudget(dataDailyBudget.getCurrentDailyBudget())
                                .withRecommendedDailyBudget(dataDailyBudget.getRecommendedDailyBudget()));
                        break;
                    case weeklyBudget:
                    case increaseStrategyWeeklyBudget:
                    case increaseStrategyTargetCPA:
                    case decreaseStrategyTargetROI:
                        GdRecommendationKpiWeeklyBudget dataWeeklyBudget =
                                GSON.fromJson(json, GdRecommendationKpiWeeklyBudget.class);
                        convertedJson = GSON.toJson(new KpiWeeklyBudget()
                                .withCurrentWeeklyBudget(dataWeeklyBudget.getCurrentWeeklyBudget())
                                .withRecommendedWeeklyBudget(dataWeeklyBudget.getRecommendedWeeklyBudget())
                                .withCurrentTargetCPA(dataWeeklyBudget.getCurrentTargetCPA())
                                .withRecommendedTargetCPA(dataWeeklyBudget.getRecommendedTargetCPA())
                                .withCurrentTargetROI(dataWeeklyBudget.getCurrentTargetROI())
                                .withRecommendedTargetROI(dataWeeklyBudget.getRecommendedTargetROI()));
                        break;
                }
            } catch (Exception e) {
                logger.warn("Error during converting json {} for recommendation {}", json, GSON.toJson(key), e);
                return null;
            }
        }

        return new RecommendationQueueInfo()
                .withClientId(clientId)
                .withUid(operator.getUid())
                .withType(key.getType().getId())
                .withCampaignId(key.getCid())
                .withAdGroupId(key.getPid())
                .withBannerId(key.getBid())
                .withUserKey1(key.getUserKey1())
                .withUserKey2(key.getUserKey2())
                .withUserKey3(key.getUserKey3())
                .withTimestamp(key.getTimestamp())
                .withSecTime(LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS))
                .withQueueTime(LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS))
                .withJsonData(convertedJson);
    }

    private GdRecommendationKey convertToExternal(Map.Entry<RecommendationQueueInfo, RecommendationStatus> e) {
        RecommendationQueueInfo key = e.getKey();
        return new GdRecommendationKey()
                .withType(GdiRecommendationType.fromId(key.getType()))
                .withCid((key.getCampaignId()))
                .withPid(key.getAdGroupId())
                .withBid(key.getBannerId())
                .withUserKey1(key.getUserKey1())
                .withUserKey2(key.getUserKey2())
                .withUserKey3(key.getUserKey3())
                .withTimestamp(key.getTimestamp());
    }
}
