package ru.yandex.direct.grid.processing.service.recommendation;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.core.entity.recommendation.service.GridRecommendationService;
import ru.yandex.direct.grid.model.entity.recommendation.GdiRecommendationType;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationKey;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationOrderBy;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationSummary;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendationSummaryContext;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;

/**
 * Сервис, возвращающий данные о рекомендациях клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class RecommendationSummaryGraphQlService {

    private final GridValidationService gridValidationService;
    private final GridRecommendationService gridRecommendationService;

    @Autowired
    public RecommendationSummaryGraphQlService(
            GridValidationService gridValidationService,
            GridRecommendationService gridRecommendationService) {

        this.gridValidationService = gridValidationService;
        this.gridRecommendationService = gridRecommendationService;
    }

    /**
     * GraphQL подзапрос. Получает информацию о рекоммендациях клиента, полученного из контекста выполнения запроса
     *
     * @param clientContext клиент, к которому относятся рекомендации
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "recommendationsSummary")
    public GdRecommendationSummaryContext getRecommendationsSummary(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @Nullable @GraphQLArgument(name = "filter") Set<GdiRecommendationType> filter,
            @Nullable @GraphQLArgument(name = "orderBy") List<@GraphQLNonNull GdRecommendationOrderBy> orderByList,
            @Nullable @GraphQLArgument(name = "limitOffset") GdLimitOffset limitOffset,
            @Nullable @GraphQLArgument(name = "cacheKey") String cacheKey) {
        gridValidationService.validateLimitOffset(limitOffset);
        GdClientInfo client = context.getQueriedClient();

        final Long clientId = client.getId();

        final List<GdRecommendationSummary> rowset =
                gridRecommendationService.getRecommendationSummary(clientId, context.getOperator(), filter);

        return new GdRecommendationSummaryContext()
                .withTotalCount(rowset.size())
                .withRowset(rowset);
    }

    /**
     * GraphQL подзапрос. Получает информацию о рекоммендациях клиента на гриде кампаний
     *
     * @param clientContext клиент, к которому относятся рекомендации
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "recommendationsSummaryCampaignGrid")
    public GdRecommendationSummaryContext getRecommendationsSummaryCampaignGrid(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @Nullable @GraphQLArgument(name = "filter") Set<GdiRecommendationType> filter,
            @Nullable @GraphQLArgument(name = "orderBy") List<@GraphQLNonNull GdRecommendationOrderBy> orderByList,
            @Nullable @GraphQLArgument(name = "limitOffset") GdLimitOffset limitOffset,
            @Nullable @GraphQLArgument(name = "cacheKey") String cacheKey) {
        gridValidationService.validateLimitOffset(limitOffset);
        GdClientInfo client = context.getQueriedClient();

        final Long clientId = client.getId();

        final List<GdRecommendationSummary> rowset =
                gridRecommendationService.getRecommendationSummaryCampaignGrid(clientId, filter);

        return new GdRecommendationSummaryContext()
                .withTotalCount(rowset.size())
                .withRowset(rowset);
    }


    /**
     * GraphQL подзапрос. Получает информацию о статусах рекоммендаций по ключам
     *
     * @param clientContext клиент, к которому относятся рекомендации
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "recommendationsStatuses")
    public List<@GraphQLNonNull GdiRecommendation> getRecommendationsStatuses(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @GraphQLArgument(name = "keys") Set<@GraphQLNonNull GdRecommendationKey> recommendationKeys) {
        Long clientId = context.getQueriedClient().getId();

        Set<RecommendationKey> internalKeys = recommendationKeys.stream()
                .map(key -> new RecommendationKey()
                        .withClientId(clientId)
                        .withType(key.getType().getId())
                        .withCampaignId(key.getCid())
                        .withAdGroupId(key.getPid())
                        .withBannerId(key.getBid())
                        .withUserKey1(key.getUserKey1())
                        .withUserKey2(key.getUserKey2())
                        .withUserKey3(key.getUserKey3())
                        .withTimestamp(key.getTimestamp())
                )
                .collect(Collectors.toSet());

        return gridRecommendationService.getRecommendationsStatuses(clientId, internalKeys);
    }
}
