package ru.yandex.direct.grid.processing.service.region;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.geo.model.GeoRegion;
import ru.yandex.direct.core.entity.geo.model.GeoRegionType;
import ru.yandex.direct.core.entity.geo.model.GeoRegionWithAdRegion;
import ru.yandex.direct.core.entity.geo.repository.GeoRegionRepository;
import ru.yandex.direct.geosearch.GeosearchClient;
import ru.yandex.direct.geosearch.model.GeoObject;
import ru.yandex.direct.regions.Region;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

@Service
@ParametersAreNonnullByDefault
public class RegionDataService {
    private static final Logger logger = LoggerFactory.getLogger(RegionDataService.class);
    private static final String ERROR_TEMPLATE = "Exception" +
            " occurred when trying to obtain nearest commercial region for %s";

    private final AdRegionDataService adRegionDataService;
    private final GeosearchClient geosearchClient;
    private final GeoRegionRepository geoRegionRepository;
    private final DirectConfig config;

    public RegionDataService(AdRegionDataService adRegionDataService,
                             GeosearchClient geosearchClient,
                             GeoRegionRepository geoRegionRepository,
                             DirectConfig config) {
        this.adRegionDataService = adRegionDataService;
        this.geosearchClient = geosearchClient;
        this.geoRegionRepository = geoRegionRepository;
        this.config = config;
    }

    public Collection<GeoRegionWithAdRegion> searchRegions(String text) {
        var geoObjects = geosearchClient
                .searchAddress(text)
                .stream()
                // Почему-то в результатах могут попасться регионы с одинаковым geoId (тогда неясно, что может быть
                // уникальным идентификатором региона, хорошо бы в этом разобраться), сильно на точночть не повлияет,
                // если отбросить один из регионов с повторяющимся regionId, так как регионы на практике соседние
                .collect(toMap(GeoObject::getGeoId, Function.identity(), (o1, o2) -> o1));

        var geoRegions = geoRegionRepository.getGeoRegionsByIds(geoObjects.keySet());

        return geoRegions
                .stream()
                .map(region -> convertToRegionWithAdRegion(geoObjects, region))
                .collect(toList());
    }

    private GeoRegionWithAdRegion convertToRegionWithAdRegion(Map<Long, GeoObject> geoObjects, GeoRegion region) {
        var longitude = geoObjects.get(region.getId()).getX().doubleValue();
        var latitude = geoObjects.get(region.getId()).getY().doubleValue();
        var geoRegionWithAdRegion = mapGeoRegionToGeoRegionWithAdRegion(region);
        try {
            return populateAdRegion(latitude, longitude, geoRegionWithAdRegion);
        } catch (InterruptedException e) {
            logger.warn(String.format(ERROR_TEMPLATE, region.getId()), e);
            Thread.currentThread().interrupt();
        } catch (ExecutionException | TimeoutException e) {
            logger.warn(String.format(ERROR_TEMPLATE, region.getId()), e);
        }
        return geoRegionWithAdRegion;
    }

    private GeoRegionWithAdRegion populateAdRegion(
            double latitude,
            double longitude,
            GeoRegionWithAdRegion geoRegionWithAdRegion
    ) throws InterruptedException, ExecutionException, TimeoutException {
        var lookupTimeout = config.getDuration("commerial-regions.lookup_timeout").toMillis();
        var maybeRegion = adRegionDataService
                .findCommercialRegion(latitude, longitude)
                .get(lookupTimeout, TimeUnit.MILLISECONDS);

        if (maybeRegion.isEmpty()) {
            return geoRegionWithAdRegion;
        }

        var commercialRegion = maybeRegion.get();
        GeoRegion adRegion = mapCommercialRegionToGeoRegion(commercialRegion);
        return geoRegionWithAdRegion.withAdRegion(adRegion);
    }

    private GeoRegionWithAdRegion mapGeoRegionToGeoRegionWithAdRegion(GeoRegion region) {
        return new GeoRegionWithAdRegion()
                .withId(region.getId())
                .withEname(region.getEname())
                .withName(region.getName())
                .withOfficeId(region.getOfficeId())
                .withParentId(region.getParentId())
                .withTrname(region.getTrname())
                .withUaname(region.getUaname())
                .withType(region.getType());
    }

    private GeoRegion mapCommercialRegionToGeoRegion(Region commercialRegion) {
        return new GeoRegion()
                .withId(commercialRegion.getId())
                .withEname(commercialRegion.getNameEn())
                .withName(commercialRegion.getNameRu())
                .withParentId(commercialRegion.getParent().getId())
                .withTrname(commercialRegion.getNameTr())
                .withUaname(commercialRegion.getNameUa())
                .withType(GeoRegionType.fromTypedValue((long) commercialRegion.getType()));
    }
}
