package ru.yandex.direct.grid.processing.service.region;

import java.util.Collection;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.geo.model.GeoRegionWithAdRegion;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.exception.GridRequestTimeoutException;
import ru.yandex.direct.grid.processing.exception.NoSuchObjectException;
import ru.yandex.direct.grid.processing.model.GdCoordinates;
import ru.yandex.direct.grid.processing.model.placement.GdRegion;
import ru.yandex.direct.grid.processing.model.placement.GdRegionDesc;
import ru.yandex.direct.grid.processing.model.placement.GdRegionWithAdRegion;
import ru.yandex.direct.regions.Region;

import static java.util.stream.Collectors.toList;

/**
 * Сервис, принимающий точку вида {latitude: X, longitude: Y} и возвращающий ближайший коммерческий регион.
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class RegionGraphQlService {

    private final AdRegionDataService adRegionDataService;
    private final RegionDataService regionDataService;
    private final DirectConfig config;

    public RegionGraphQlService(AdRegionDataService adRegionDataService,
                                RegionDataService regionDataService,
                                DirectConfig config) {
        this.adRegionDataService = adRegionDataService;
        this.regionDataService = regionDataService;
        this.config = config;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "region")
    public GdRegionDesc getGeoSearch(
            @GraphQLNonNull @GraphQLArgument(name = "coordinates") GdCoordinates coordinates
    ) throws InterruptedException, ExecutionException {
        var lookupTimeout = config.getDuration("commerial-regions.lookup_timeout").toMillis();

        try {
            return adRegionDataService
                    .findCommercialRegion(coordinates.getX().doubleValue(), coordinates.getY().doubleValue())
                    .get(lookupTimeout, TimeUnit.MILLISECONDS)
                    .map(this::mapToGdRegion)
                    // Такого не должно случиться, но тем не менее
                    .orElseThrow(() -> new NoSuchObjectException(
                            String.format(
                                    "Region with coordinates (%f, %f) not found",
                                    coordinates.getX().doubleValue(),
                                    coordinates.getY().doubleValue()
                            )
                    ));
        } catch (TimeoutException e) {
            throw new GridRequestTimeoutException();
        }
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "regionByText")
    public Collection<GdRegionWithAdRegion> getRegionByText(
            @GraphQLNonNull @GraphQLArgument(name = "text") String text
    ) {
        return regionDataService.searchRegions(text).stream().map(this::mapGdRegionWithAdRegion).collect(toList());
    }

    private GdRegionWithAdRegion mapGdRegionWithAdRegion(GeoRegionWithAdRegion region) {
        var adRegion = region.getAdRegion();
        var gdAdRegion = new GdRegion()
                .withId(adRegion.getId())
                .withName(adRegion.getName())
                .withNameEn(adRegion.getEname())
                .withNameRu(adRegion.getName())
                .withNameTr(adRegion.getTrname())
                .withNameUa(adRegion.getUaname())
                .withParent(adRegion.getParentId())
                .withType(adRegion.getType().getTypedValue().intValue());
        return new GdRegionWithAdRegion()
                .withAdRegion(gdAdRegion)
                .withId(region.getId())
                .withName(region.getName())
                .withNameRu(region.getName())
                .withNameEn(region.getEname())
                .withNameTr(region.getTrname())
                .withNameUa(region.getUaname())
                .withParent(region.getParentId())
                .withType(region.getType().getTypedValue().intValue());
    }

    @GraphQLNonNull
    private GdRegionDesc mapToGdRegion(Region region) {
        return new GdRegionDesc()
                .withId(region.getId())
                .withNameRu(region.getNameRu())
                .withNameEn(region.getNameEn())
                .withNameUa(region.getNameUa())
                .withNameTr(region.getNameTr());
    }

}
