package ru.yandex.direct.grid.processing.service.shortener;

import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.frontdb.repository.FilterShortcutRepository;
import ru.yandex.direct.grid.processing.model.shortener.GdShortFilter;
import ru.yandex.direct.grid.processing.model.shortener.GdShortFilterUnion;
import ru.yandex.direct.utils.JsonUtils;

@Service
@ParametersAreNonnullByDefault
public class GridShortenerService {

    private static final ObjectMapper FILTER_MAPPER = new ObjectMapper()
            .registerModule(new JavaTimeModule())
            .registerModule(new KotlinModule())
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    static {
        FILTER_MAPPER.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    private final FilterShortcutRepository filterShortcutRepository;

    @Autowired
    public GridShortenerService(FilterShortcutRepository filterShortcutRepository) {
        this.filterShortcutRepository = filterShortcutRepository;
    }

    public GdShortFilter saveFilter(ClientId clientId, GdShortFilterUnion gdShortFilterUnion, @Nullable String filterKey) {
        try {
            Object gdFilter = getNonNullFilterFromUnion(gdShortFilterUnion);
            String filterAsJson = FILTER_MAPPER.writeValueAsString(gdFilter);
            String key = filterShortcutRepository.saveFilter(clientId, filterAsJson, filterKey);
            return new GdShortFilter()
                    .withFilterKey(key);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    public <T> T getSavedFilter(String filterKey, ClientId clientId, Class<T> clazz, Supplier<T> stub) {
        String savedFilter = filterShortcutRepository.getFilter(clientId, filterKey);
        if (savedFilter != null) {
            return JsonUtils.fromJson(savedFilter, clazz);
        } else {
            return stub.get();
        }
    }

    private static Object getNonNullFilterFromUnion(GdShortFilterUnion gdShortFilterUnion) {
        return StreamEx.of(GdShortFilterUnion.allModelProperties())
                .map(x -> x.getRaw(gdShortFilterUnion))
                .nonNull()
                .findAny()
                .orElseThrow();
    }
}
