package ru.yandex.direct.grid.processing.service.showcondition;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.showcondition.GdValidateKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.GdValidateKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsOperators;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdBulkRefineKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdChangeKeywordsCase;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdDeleteKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdDeleteKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPreviewPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdMoveSearchQueriesToMinusKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdMoveToMinusKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdMoveToMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdRefineKeyword;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdRefineKeywordPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdRefineKeywordWithMinuses;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSuspendKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSuspendKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUnsuspendKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUnsuspendKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywordsPayload;
import ru.yandex.direct.grid.processing.processor.GraphQLRateLimit;
import ru.yandex.direct.grid.processing.service.showcondition.keywords.FindAndReplaceDataService;
import ru.yandex.direct.grid.processing.service.showcondition.keywords.KeywordsDataService;
import ru.yandex.direct.grid.processing.service.showcondition.keywords.RefineKeywordService;
import ru.yandex.direct.grid.processing.service.showcondition.searchQueries.SearchQueriesDataService;

import static ru.yandex.direct.grid.processing.util.UpdateDataHelper.toUidAndClientId;

/**
 * Сервис, для изменения ключевых фраз
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class KeywordsGraphQlService {

    public static final String SUSPEND_KEYWORDS_MUTATION = "suspendKeywords";

    private final KeywordsDataService keywordsDataService;
    private final FindAndReplaceDataService findAndReplaceDataService;
    private final RefineKeywordService refineKeywordService;
    private final SearchQueriesDataService searchQueriesDataService;

    @Autowired
    public KeywordsGraphQlService(KeywordsDataService keywordsDataService,
                                  FindAndReplaceDataService findAndReplaceDataService,
                                  RefineKeywordService refineKeywordService,
                                  SearchQueriesDataService searchQueriesDataService) {
        this.keywordsDataService = keywordsDataService;
        this.findAndReplaceDataService = findAndReplaceDataService;
        this.refineKeywordService = refineKeywordService;
        this.searchQueriesDataService = searchQueriesDataService;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "validateKeywords")
    public GdValidateKeywordsPayload validateKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdValidateKeywords input) {
        return keywordsDataService.validateKeywords(input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addKeywords")
    public GdAddKeywordsPayload addKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddKeywords input) {
        return keywordsDataService
                .addKeywords(input, context.getOperator().getUid(), toUidAndClientId(context.getSubjectUser()));
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "deleteKeywords")
    public GdDeleteKeywordsPayload deleteKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdDeleteKeywords input) {
        return keywordsDataService.deleteKeywords(input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "moveToMinusKeywords")
    public GdMoveToMinusKeywordsPayload moveToMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMoveToMinusKeywords input) {
        return keywordsDataService.moveToMinusKeywords(input);
    }

    /**
     * Принимает на вход поисковой запрос и пытается добавить его в минус-слова кампании. Сначала проверяет наличие
     * фразы с таким же текстом, если такая фраза существует, то она удаляется.
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "moveSearchQueryToMinusKeywords")
    public GdMoveToMinusKeywordsPayload moveSearchQueryToMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMoveSearchQueriesToMinusKeywords input) {
        return searchQueriesDataService.moveToMinusKeywords(context.getSubjectUser(), context.getOperator(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = SUSPEND_KEYWORDS_MUTATION)
    public GdSuspendKeywordsPayload suspendKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSuspendKeywords input) {
        return keywordsDataService.suspendKeywords(input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "unsuspendKeywords")
    public GdUnsuspendKeywordsPayload unsuspendKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUnsuspendKeywords input) {
        return keywordsDataService.unSuspendKeywords(input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateKeywords")
    public GdUpdateKeywordsPayload updateKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateKeywords input) {
        return keywordsDataService
                .updateKeywords(input, context.getOperator().getUid(), toUidAndClientId(context.getSubjectUser()));
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "getFindAndReplaceKeywordsPreview")
    public GdFindAndReplaceKeywordsPreviewPayload getFindAndReplaceKeywordsPreview(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdFindAndReplaceKeywords input) {
        return findAndReplaceDataService
                .getPreview(input, context.getOperator().getUid(), toUidAndClientId(context.getSubjectUser()));
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "findAndReplaceKeywords")
    public GdFindAndReplaceKeywordsPayload findAndReplaceKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdFindAndReplaceKeywords input) {
        return findAndReplaceDataService.findAndReplaceKeywords(input,
                context.getOperator().getUid(),
                toUidAndClientId(context.getSubjectUser())
        );
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "changeKeywordsCase")
    public GdFindAndReplaceKeywordsPayload changeKeywordsCase(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdChangeKeywordsCase input) {
        return findAndReplaceDataService.changeKeywordsCase(input,
                context.getOperator().getUid(),
                toUidAndClientId(context.getSubjectUser())
        );
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "refineKeyword")
    @GraphQLRateLimit(periodInSeconds = 3600, maxRequests = 50)
    public GdRefineKeywordPayload refineKeyword(
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRefineKeyword input) {
        return refineKeywordService.refine(input);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "bulkRefineKeywords")
    @GraphQLRateLimit(periodInSeconds = 3600, maxRequests = 50)
    public GdRefineKeywordPayload bulkRefineKeywords(
            @GraphQLNonNull @GraphQLArgument(name = "input") GdBulkRefineKeywords input) {
        return refineKeywordService.refine(input);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "refineKeywordWithMinuses")
    @GraphQLRateLimit(periodInSeconds = 3600, maxRequests = 50)
    public GdRefineKeywordPayload refineKeywordWithMinuses(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRefineKeywordWithMinuses input) {
        return refineKeywordService.refine(context.getSubjectUser().getClientId(), context.getSubjectUser().getUid(),
                input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addKeywordsOperators")
    public GdFindAndReplaceKeywordsPayload addKeywordsOperators(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddKeywordsOperators input) {
        return findAndReplaceDataService.addKeywordsOperators(input,
                context.getOperator().getUid(),
                toUidAndClientId(context.getSubjectUser())
        );
    }

}
