package ru.yandex.direct.grid.processing.service.showcondition;

import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.exception.GridValidationException;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.retargeting.GdCaRetargetingConditionContext;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargeting;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingCondition;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionContext;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionsAndAvailableShortcutsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingContext;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingWithTotals;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdDeleteRetargetingConditions;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdEstimateRetargetingCondition;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdEstimateRetargetingConditionContainer;
import ru.yandex.direct.grid.processing.service.bidmodifier.BidModifierDataService;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.grid.processing.service.showcondition.container.RetargetingsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.showcondition.retargeting.RetargetingDataService;
import ru.yandex.direct.grid.processing.service.showcondition.validation.RetargetingValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toCoreRetargetingCondition;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toGdRetargetingContext;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toRetargetingsCacheRecordInfo;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Сервис, работающий с ретаргетингами клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class RetargetingGraphQlService {
    private static final Logger logger = LoggerFactory.getLogger(RetargetingGraphQlService.class);

    public static final String RETARGETING_RESOLVER_NAME = "retargetings";
    public static final String SEARCH_RETARGETING_RESOLVER_NAME = "searchRetargetings";
    public static final String RETARGETING_CONDITIONS_RESOLVER_NAME = "retargetingConditions";
    public static final String RETARGETING_CONDITIONS_SAFE_RESOLVER_NAME = "retargetingConditionsSafe";
    public static final String RETARGETING_CONDITIONS_AND_AVAILABLE_SHORTCUTS_RESOLVER_NAME =
            "retargetingConditionsAndAvailableShortcuts";
    public static final String RETARGETING_CONDITIONS_AND_AVAILABLE_SHORTCUTS_SAFE_RESOLVER_NAME =
            "retargetingConditionsAndAvailableShortcutsSafe";

    private final GridValidationService gridValidationService;
    private final GridCacheService gridCacheService;
    private final GridShortenerService gridShortenerService;
    private final FeatureService featureService;

    private final RetargetingConditionService retargetingConditionService;
    private final RetargetingValidationService retargetingValidationService;
    private final RetargetingDataService retargetingDataService;
    private final BidModifierDataService bidModifierDataService;

    @Autowired
    public RetargetingGraphQlService(
            GridValidationService gridValidationService,
            GridCacheService gridCacheService,
            GridShortenerService gridShortenerService, RetargetingConditionService retargetingConditionService,
            RetargetingValidationService retargetingValidationService,
            FeatureService featureService,
            RetargetingDataService retargetingDataService,
            BidModifierDataService bidModifierDataService) {
        this.gridValidationService = gridValidationService;
        this.gridCacheService = gridCacheService;
        this.gridShortenerService = gridShortenerService;
        this.retargetingConditionService = retargetingConditionService;
        this.retargetingValidationService = retargetingValidationService;
        this.featureService = featureService;
        this.retargetingDataService = retargetingDataService;
        this.bidModifierDataService = bidModifierDataService;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = RETARGETING_RESOLVER_NAME)
    public GdRetargetingContext getRetargetings(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRetargetingsContainer input) {
        retargetingValidationService.validateGdRetargetingsContainer(input);
        GdClientInfo client = context.getQueriedClient();

        setFilterIfNull(input, client);

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());

        // пытаемся прочитать из кеша нужный диапазон строк
        RetargetingsCacheRecordInfo recordInfo = toRetargetingsCacheRecordInfo(client.getId(), input);
        Optional<GdRetargetingContext> res = gridCacheService.getFromCache(recordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        boolean cpcAndCpmOnOneGridEnabled =
                featureService.isEnabledForClientId(ClientId.fromLong(client.getId()), CPC_AND_CPM_ON_ONE_GRID_ENABLED);

        // в кеше данные не нашлись, читаем из mysql/YT
        GdRetargetingWithTotals retargetingWithTotals = retargetingDataService.getRetargetingsRowset(context, input);
        GdRetargetingContext retargetingContext =
                toGdRetargetingContext(retargetingWithTotals, input.getFilter(), cpcAndCpmOnOneGridEnabled);
        retargetingContext.setFilter(input.getFilter());

        // сохраняем в кеш, если надо, и возвращаем нужный диапазон строк в результате
        return gridCacheService.getResultAndSaveToCacheIfRequested(recordInfo, retargetingContext,
                retargetingWithTotals.getGdRetargetings(), range,
                context.getFetchedFieldsReslover().getRetargeting().getCacheKey());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = SEARCH_RETARGETING_RESOLVER_NAME)
    public GdRetargetingContext getSearchRetargetings(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRetargetingsContainer input) {
        retargetingValidationService.validateGdRetargetingsContainerForSearch(input);
        var client = context.getQueriedClient();
        setFilterIfNull(input, client);

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        // пытаемся прочитать из кеша нужный диапазон строк
        RetargetingsCacheRecordInfo recordInfo = toRetargetingsCacheRecordInfo(client.getId(), input);
        Optional<GdRetargetingContext> result = gridCacheService.getFromCache(recordInfo, range);
        if (result.isPresent()) {
            return result.get();
        }

        //получаем данные от retargeting_filter корректировок
        List<GdRetargeting> retargetingsFromBidModifiers = bidModifierDataService
                .getSearchRetargetingsFromRetargetingFilterModifiers(context, input);
        GdRetargetingContext retargetingContext = toGdRetargetingContext(retargetingsFromBidModifiers);

        // сохраняем в кеш, если надо, и возвращаем нужный диапазон строк в результате
        return gridCacheService.getResultAndSaveToCacheIfRequested(recordInfo, retargetingContext,
                retargetingsFromBidModifiers, range,
                context.getFetchedFieldsReslover().getRetargeting().getCacheKey());
    }

    private void setFilterIfNull(
            GdRetargetingsContainer input,
            GdClientInfo client) {
        if (input.getFilterKey() != null) {
            GdRetargetingFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    ClientId.fromLong(client.getId()),
                    GdRetargetingFilter.class,
                    () -> new GdRetargetingFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }
    }


    /**
     * Возвращение условий ретаргетинга клиента (в том числе сохраненных шорткатов)
     */
    @GraphQLNonNull
    @GraphQLQuery(name = RETARGETING_CONDITIONS_RESOLVER_NAME)
    public GdRetargetingConditionContext getRetargetingConditions(
            @SuppressWarnings("unused") @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRetargetingConditionsContainer input) {
        retargetingValidationService.validateGdRetargetingConditionsContainer(input);
        List<GdRetargetingCondition> rowset = retargetingDataService
                .getRetargetingConditionsRowset(client.getInfo(), input.getFilter());

        return new GdRetargetingConditionContext()
                .withIsMetrikaAvailable(true)
                .withRowset(rowset)
                .withTotalCount(rowset.size());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = RETARGETING_CONDITIONS_SAFE_RESOLVER_NAME)
    public GdRetargetingConditionContext getRetargetingConditionsSafe(
            @SuppressWarnings("unused") @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRetargetingConditionsContainer input) {
        retargetingValidationService.validateGdRetargetingConditionsContainer(input);
        try {
            List<GdRetargetingCondition> rowset = retargetingDataService
                    .getRetargetingConditionsRowset(client.getInfo(), input.getFilter());

            return new GdRetargetingConditionContext()
                    .withIsMetrikaAvailable(true)
                    .withRowset(rowset)
                    .withTotalCount(rowset.size());
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for retargeting conditions for clientId: " + client.getInfo().getId(), e);
            return makeFailedRetargetingConditionContext();
        }
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "caRetargetingConditions")
    public GdCaRetargetingConditionContext getCaRetargetingCondition(
            @SuppressWarnings("unused") @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRetargetingConditionsContainer input) {
        retargetingValidationService.validateGdRetargetingConditionsContainer(input);
        var rowset = retargetingDataService.getCaRetargetingConditions(client.getInfo(), input.getFilter());

        return new GdCaRetargetingConditionContext()
                .withRowset(rowset);
    }

    /**
     * Возвращение условий ретаргетинга клиента и доступных для создания шорткатов
     */
    @GraphQLNonNull
    @GraphQLQuery(name = RETARGETING_CONDITIONS_AND_AVAILABLE_SHORTCUTS_RESOLVER_NAME)
    public GdRetargetingConditionContext getRetargetingConditionsAndAvailableShortcuts(
            @SuppressWarnings("unused") @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRetargetingConditionsAndAvailableShortcutsContainer input) {
        retargetingValidationService.validateGdRetargetingConditionsAndAvailableShortcutsContainer(input);
        List<GdRetargetingCondition> rowset =
                retargetingDataService.getRetargetingConditionsAndAvailableShortcutsRowset(client.getInfo(),
                        input.getFilter(), input.getCampaignId());

        return new GdRetargetingConditionContext()
                .withIsMetrikaAvailable(true)
                .withRowset(rowset)
                .withTotalCount(rowset.size());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = RETARGETING_CONDITIONS_AND_AVAILABLE_SHORTCUTS_SAFE_RESOLVER_NAME)
    public GdRetargetingConditionContext getRetargetingConditionsAndAvailableShortcutsSafe(
            @SuppressWarnings("unused") @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRetargetingConditionsAndAvailableShortcutsContainer input) {
        retargetingValidationService.validateGdRetargetingConditionsAndAvailableShortcutsContainer(input);
        try {
            List<GdRetargetingCondition> rowset =
                    retargetingDataService.getRetargetingConditionsAndAvailableShortcutsRowset(client.getInfo(),
                            input.getFilter(), input.getCampaignId());

            return new GdRetargetingConditionContext()
                    .withIsMetrikaAvailable(true)
                    .withRowset(rowset)
                    .withTotalCount(rowset.size());
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for retargeting conditions and available shortcuts for " +
                    "clientId: " + client.getInfo().getId(), e);
            return makeFailedRetargetingConditionContext();
        }
    }


    /**
     * Оценка аудитории для условия ретаргетинга
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "estimateRetargetingCondition")
    public GdEstimateRetargetingCondition estimateRetargetingCondition(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdEstimateRetargetingConditionContainer input) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        retargetingValidationService.validateEstimateRetargetingCondition(input);

        ClientId clientId = context.getSubjectUser().getClientId();
        Result<Long> operationResult = retargetingConditionService
                .estimateRetargetingCondition(clientId,
                        toCoreRetargetingCondition(clientId, input.getConditionRules()));

        if (!operationResult.getErrors().isEmpty()) {
            GdValidationResult validationResult = gridValidationService.getValidationResult(operationResult,
                    path(field(GdDeleteRetargetingConditions.RETARGETING_CONDITION_IDS)));
            throw new GridValidationException(validationResult);
        }

        return new GdEstimateRetargetingCondition()
                .withEstimation(operationResult.getResult());
    }

    private static GdRetargetingConditionContext makeFailedRetargetingConditionContext() {
        return new GdRetargetingConditionContext()
                .withIsMetrikaAvailable(false)
                .withRowset(List.of())
                .withTotalCount(0);
    }
}
