package ru.yandex.direct.grid.processing.service.showcondition;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.service.KeywordUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.ShowConditionFetchedFieldsResolver;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.showcondition.GdAutotargetingStatInput;
import ru.yandex.direct.grid.processing.model.showcondition.GdAutotargetingStatisticsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.GdKeyword;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowCondition;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionCategoryStat;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionFilter;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionStatItem;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionTotals;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionsContainer;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionsContext;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.group.GroupDataService;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.grid.processing.service.showcondition.container.ShowConditionsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.showcondition.keywords.ShowConditionDataService;
import ru.yandex.direct.grid.processing.service.showcondition.validation.ShowConditionValidationService;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.intapi.client.model.response.statistics.CampaignStatisticsItem;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptySet;
import static java.util.Collections.reverseOrder;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignType;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils.CPC_CAMPAIGN_TYPES;
import static ru.yandex.direct.grid.processing.service.showcondition.ShowConditionTotals.totalsFromCampaignStatisticItem;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.AutotargetingStatConverter.getCategoryComparator;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.AutotargetingStatConverter.toGdRelevanceMatchCategory;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.AutotargetingStatConverter.toSearchQueriesRequest;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionConverter.toGdShowConditionsContext;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionConverter.toShowConditionsCacheRecordInfo;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;

/**
 * Сервис, возвращающий данные об условиях показа баннеров клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class ShowConditionGraphQlService {
    public static final String SHOW_CONDITIONS_RESOLVER_NAME = "showConditions";
    static final String ADS_IN_AD_GROUP_COUNT_RESOLVER_NAME = "adsInAdGroupCount";
    static final String AUTOTARGETING_STAT = "autotargetingStat";
    static final String AUTOTARGETING_STATISTICS = "autotargetingStatistics";

    private final GridCacheService gridCacheService;
    private final GridShortenerService gridShortenerService;
    private final GroupDataService groupDataService;
    private final ShowConditionDataService showConditionDataService;
    private final ShowConditionValidationService showConditionValidationService;
    private final FeatureService featureService;
    private final IntApiClient intApiClient;

    @Autowired
    public ShowConditionGraphQlService(GridCacheService gridCacheService,
                                       GridShortenerService gridShortenerService,
                                       GroupDataService groupDataService,
                                       ShowConditionDataService showConditionDataService,
                                       FeatureService featureService,
                                       ShowConditionValidationService showConditionValidationService,
                                       IntApiClient intApiClient) {
        this.gridCacheService = gridCacheService;
        this.gridShortenerService = gridShortenerService;
        this.groupDataService = groupDataService;
        this.showConditionDataService = showConditionDataService;
        this.featureService = featureService;
        this.showConditionValidationService = showConditionValidationService;
        this.intApiClient = intApiClient;
    }

    /**
     * GraphQL подзапрос. Получает информацию о статистике автотаргетинга по категориям.
     * TODO УДАЛИТЬ ПОСЛЕ ПЕРЕХОДА НА AUTOTARGETING_STATISTICS
     */
    @SuppressWarnings("WeakerAccess")
    @GraphQLNonNull
    @GraphQLQuery(name = AUTOTARGETING_STAT)
    public List<@GraphQLNonNull GdShowConditionCategoryStat> getAutotargetingStat(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAutotargetingStatInput input
    ) {
        final var request = toSearchQueriesRequest(input, context);
        final var searchQueriesStat = intApiClient.getCampaignStatistics(request);
        final var groupsByTargetingCategory = StreamEx.of(searchQueriesStat.getData())
                .filter(s -> {
                    //contextCond может приходить из intapi как phrase, a не как contextcond_orig
                    var contextCond = nullableNvl(s.getContextCondAsPhrase(), s.getContextCond());
                    return s.getTargetingCategory() != null &&
                            contextCond != null && contextCond.startsWith(KeywordUtils.AUTOTARGETING_KEYWORD);
                })
                .groupingBy(CampaignStatisticsItem::getTargetingCategory);

        return groupsByTargetingCategory.entrySet().stream()
                .map(e -> getShowConditionCategoryStat(e, true))
                .sorted(Comparator.comparing(i -> i.getTotals().getCost(), reverseOrder()))
                .collect(Collectors.toList());
    }

    /**
     * GraphQL подзапрос. Получает информацию о статистике автотаргетинга по категориям.
     */
    @SuppressWarnings("WeakerAccess")
    @GraphQLNonNull
    @GraphQLQuery(name = AUTOTARGETING_STATISTICS)
    public GdAutotargetingStatisticsPayload getAutotargetingStatistics(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAutotargetingStatInput input
    ) {
        final var request = toSearchQueriesRequest(input, context);
        final var searchQueriesStat = intApiClient.getCampaignStatistics(request);

        final var groupsByTargetingCategory = StreamEx.of(searchQueriesStat.getData())
                .groupingBy(CampaignStatisticsItem::getTargetingCategory);
        GdShowConditionTotals campaignTotals = totalsFromCampaignStatisticItem(searchQueriesStat.getTotals());

        // если в запросе была всего одна категория, то limit-offset уже применен на уровне запроса
        // (детали в методе AutotargetingStatConverter.toSearchQueriesRequest)
        var isOneCategoryAndLimitApplied = input.getRelevanceMatchCategory() != null
                && input.getRelevanceMatchCategory().size() == 1;

        var categoryComparator = getCategoryComparator(input.getOrderBy());
        var rowsByCategories = groupsByTargetingCategory.entrySet().stream()
                .map(e -> {
                    GdShowConditionCategoryStat result = getShowConditionCategoryStat(e, false);

                    if (isOneCategoryAndLimitApplied) {
                        result.setTotals(campaignTotals);
                    } else if (input.getLimitOffset() != null) {
                        int offset = input.getLimitOffset().getOffset();
                        int limit = input.getLimitOffset().getLimit();
                        var items =
                                StreamEx.of(result.getSearchQueries())
                                        .skip(offset)
                                        .limit(limit)
                                        .toList();
                        result.setSearchQueries(items);
                    }

                    return result;
                })
                .sorted(categoryComparator)
                .collect(Collectors.toList());

        return new GdAutotargetingStatisticsPayload()
                .withRowset(rowsByCategories)
                .withTotals(campaignTotals);
    }

    private static GdShowConditionCategoryStat getShowConditionCategoryStat(
            Map.Entry<String, List<CampaignStatisticsItem>> e,
            boolean sortQueriesByCost) {

        Stream<GdShowConditionStatItem> stream = e.getValue()
                .stream()
                .map(i -> new GdShowConditionStatItem()
                        .withTotals(totalsFromCampaignStatisticItem(i))
                        .withSearchQuery(i.getSearchQuery()));
        if (sortQueriesByCost) {
            stream = stream
                    .sorted(Comparator.comparing(i -> i.getTotals().getCost(), reverseOrder()));
        }
        var items = stream
                .collect(Collectors.toList());

        var category = toGdRelevanceMatchCategory(e.getKey());
        GdShowConditionTotals totals = ShowConditionTotals.getCategoryTotals(items);
        if (category == null) {
            // Статистика без категории чаще всего соответствует РСЯ.
            // На всякий случай скрываем то, что пришло как "поисковые фразы", чтобы не показывать мусор
            items = List.of();
        }
        return new GdShowConditionCategoryStat()
                .withTotals(totals)
                .withCategory(category)
                .withSearchQueries(items);
    }

    /**
     * GraphQL подзапрос. Получает информацию об условиях показа клиента, полученного из контекста выполнения
     * запроса
     */
    @SuppressWarnings("WeakerAccess")
    @GraphQLNonNull
    @GraphQLQuery(name = SHOW_CONDITIONS_RESOLVER_NAME)
    public GdShowConditionsContext getShowConditions(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdShowConditionsContainer input) {
        showConditionValidationService.validateGdShowConditionsContainer(input);
        GdStatRequirements statRequirements =
                normalizeStatRequirements(input.getStatRequirements(), context.getInstant(), null);
        input.setStatRequirements(statRequirements);

        GdClientInfo client = context.getQueriedClient();

        if (input.getFilterKey() != null) {
            GdShowConditionFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    ClientId.fromLong(client.getId()),
                    GdShowConditionFilter.class,
                    () -> new GdShowConditionFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        // пытаемся прочитать из кеша нужный диапазон строк
        ShowConditionsCacheRecordInfo recordInfo = toShowConditionsCacheRecordInfo(client.getId(), input);
        Optional<GdShowConditionsContext> res = gridCacheService.getFromCache(recordInfo, range);

        boolean cpcAndCpmOnOneGridEnabled =
                featureService.isEnabledForClientId(ClientId.fromLong(client.getId()), CPC_AND_CPM_ON_ONE_GRID_ENABLED);

        ShowConditionFetchedFieldsResolver showConditionFetchedFieldsResolver =
                context.getFetchedFieldsReslover().getShowCondition();

        GdShowConditionsContext showConditionsContext;
        if (res.isPresent()) {
            showConditionsContext = res.get();

            //Добавляем данные торгов, только если они были запрошены
            if (showConditionFetchedFieldsResolver.getAuctionData()) {
                addKeywordsAuctionData(showConditionsContext, client, cpcAndCpmOnOneGridEnabled);
            }
        } else {
            // В кеше данные не нашлись, читаем из mysql/YT
            var gdShowConditionWithTotals = showConditionDataService
                    .getShowConditions(client, input, context, range, cpcAndCpmOnOneGridEnabled);
            showConditionsContext =
                    toGdShowConditionsContext(gdShowConditionWithTotals, input.getFilter(), cpcAndCpmOnOneGridEnabled);

            // Сохраняем в кеш, если надо, и возвращаем нужный диапазон строк в результате
            gridCacheService.getResultAndSaveToCacheIfRequested(
                    recordInfo,
                    showConditionsContext,
                    gdShowConditionWithTotals.getGdShowConditions(),
                    range,
                    showConditionFetchedFieldsResolver.getCacheKey()
            );
        }
        // Дозапрашиваем mainAd, т.к. они не хранятся в кеше и подгружаются к обрезанному результату
        showConditionDataService.setMainAds(client, showConditionsContext.getRowset());

        return showConditionsContext;
    }

    /**
     * GraphQL подзапрос. Получает кол-во объявлений для группы фразы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = ADS_IN_AD_GROUP_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getAdsInAdGroupCount(@GraphQLContext GdShowCondition gdShowCondition) {
        return groupDataService.getAdsCount(gdShowCondition.getAdGroupId());
    }

    private void addKeywordsAuctionData(GdShowConditionsContext gdShowConditionsContext,
                                        GdClientInfo client,
                                        boolean cpcAndCpmOnOneGridEnabled) {
        //Смотрим, какие фразы взятые из кеша требуют дополучения данных торгов
        List<Long> keywordIdsToAddAuctionData = StreamEx.of(gdShowConditionsContext.getRowset())
                .select(GdKeyword.class)
                .filter(keyword -> keyword.getAuctionData() == null)
                .filter(keyword -> !cpcAndCpmOnOneGridEnabled
                        || CPC_CAMPAIGN_TYPES.contains(toCampaignType(keyword.getAdGroup().getCampaign().getType())))
                .map(GdKeyword::getId)
                .toList();

        //Если надо, добавляем данных в отдаваемый чанк
        if (!keywordIdsToAddAuctionData.isEmpty()) {
            List<GdShowCondition> populatedRowset =
                    showConditionDataService.populateKeywordsWithExternalPriceData(
                            client, keywordIdsToAddAuctionData, gdShowConditionsContext.getRowset()
                    );
            gdShowConditionsContext.setRowset(populatedRowset);
        }
    }
}
