package ru.yandex.direct.grid.processing.service.showcondition.bids;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetAutoBids;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetAutoBidsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetBids;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetBidsPayload;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.grid.processing.service.showcondition.converter.BidsDataRequestConverter;
import ru.yandex.direct.grid.processing.service.showcondition.validation.ShowConditionValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.result.MassResult;

import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.bidChangeNotAllowedForBsRarelyLoadedAdGroup;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getSuccessfullyUpdatedIds;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getUpdatedIdsByDefectInWarns;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class BidsDataService {

    public static final int MAX_IDS_PER_REQUEST = 10000;

    private final BidService bidService;
    private final GridValidationService gridValidationService;
    private final RetargetingService retargetingService;
    private final ShowConditionValidationService showConditionValidationService;
    private final RelevanceMatchService relevanceMatchService;
    private final GridContextProvider gridContextProvider;


    @Autowired
    public BidsDataService(BidService bidService, GridValidationService gridValidationService,
                           RetargetingService retargetingService,
                           ShowConditionValidationService showConditionValidationService,
                           RelevanceMatchService relevanceMatchService,
                           GridContextProvider gridContextProvider) {
        this.bidService = bidService;
        this.gridValidationService = gridValidationService;
        this.retargetingService = retargetingService;
        this.showConditionValidationService = showConditionValidationService;
        this.relevanceMatchService = relevanceMatchService;
        this.gridContextProvider = gridContextProvider;
    }

    /**
     * Обновить ставки для фраз с использованием мастера ставок
     *
     * @param input запрос со списком идентификаторов условий показа, и правилами выставления ставок
     */
    GdSetAutoBidsPayload setAutoBids(GdSetAutoBids input) {
        showConditionValidationService.validateSetAutoBidsRequest(input);
        GridGraphQLContext context = gridContextProvider.getGridContext();
        List<SetAutoBidItem> setAutoBidItems = BidsDataRequestConverter.convertRequestToSetAutoBidItems(input);
        MassResult<SetAutoBidItem> result = bidService.setAutoBids(context.getSubjectUser().getClientId(),
                context.getOperator().getUid(), setAutoBidItems, false);

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdSetAutoBids.KEYWORD_IDS)));

        List<Long> successfullyUpdatedIds = getSuccessfullyUpdatedIds(result);

        // Костыль для случая, когда из интерфейса пытаются поменять ставку ключевой фразы для группы со статусом
        // "Мало показов". В этом случае изменение не происходит, но валидация возвращает для кф warning и
        // соответственно здесь getSuccessfullyUpdatedIds относит кф к успешно измененным.
        // Поэтому ищем дополнительно id фраз с соответствующим предупреждением и отфильтровываем их из успешных.
        List<Long> updatedIdsWithBsRarelyLoadedWarn = getUpdatedIdsByDefectInWarns(result,
                bidChangeNotAllowedForBsRarelyLoadedAdGroup());
        List<Long> successfullyUpdatedIdsWithoutBsRarelyLoadedWarning =
                filterList(successfullyUpdatedIds, id -> !updatedIdsWithBsRarelyLoadedWarn.contains(id));

        return new GdSetAutoBidsPayload()
                .withKeywordIds(successfullyUpdatedIdsWithoutBsRarelyLoadedWarning)
                .withValidationResult(validationResult);
    }

    /**
     * Выставить единые ставки для фраз
     *
     * @param input запрос со списком идентификаторов условий показа, и правилами выставления ставок
     */
    public GdSetBidsPayload setBids(GdSetBids input) {
        showConditionValidationService.validateSetBidsRequest(input);

        GridGraphQLContext context = gridContextProvider.getGridContext();
        ClientId clientId = context.getSubjectUser().getClientId();

        Set<Long> relevanceMatchIds = relevanceMatchService.getRelevanceMatchIds(clientId, input.getShowConditionIds());

        List<SetBidItem> setBidItems = BidsDataRequestConverter.convertRequestToSetBidItems(input, id ->
                relevanceMatchIds.contains(id) ? ShowConditionType.RELEVANCE_MATCH : ShowConditionType.KEYWORD);

        MassResult<SetBidItem> result = bidService.setBids(clientId, context.getOperator().getUid(), setBidItems);

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdSetBids.SHOW_CONDITION_IDS)));

        return new GdSetBidsPayload()
                .withShowConditionIds(getSuccessfullyUpdatedIds(result))
                .withValidationResult(validationResult);
    }

    /**
     * Выставить единые ставки для ретаргетингов
     *
     * @param input запрос со списком идентификаторов условий показа, и правилами выставления ставок
     */
    public GdSetBidsPayload setBidsRetargeting(GdSetBids input) {
        showConditionValidationService.validateSetBidsRequest(input);
        GridGraphQLContext context = gridContextProvider.getGridContext();

        List<SetBidItem> setBidItems = BidsDataRequestConverter.convertRequestToSetBidItems(input,
                id -> ShowConditionType.AUDIENCE_TARGET);

        MassResult<SetBidItem> result = retargetingService.setBids(setBidItems, context.getSubjectUser().getClientId(),
                context.getOperator().getUid());

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdSetBids.SHOW_CONDITION_IDS)));
        return new GdSetBidsPayload()
                .withShowConditionIds(getSuccessfullyUpdatedIds(result))
                .withValidationResult(validationResult);
    }

}
