package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.model.AutoBudgetPriority;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsItem;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasAutotargetingPrefix;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.phraseWithoutAutotargetingPrefix;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class AddKeywordsRequestConverter {
    private static final Integer MEDIUM_PRIORITY = AutoBudgetPriority.MEDIUM.getTypedValue();

    private final ClientService clientService;
    private final AdGroupService adGroupService;

    @Autowired
    public AddKeywordsRequestConverter(ClientService clientService,
                                       AdGroupService adGroupService) {
        this.clientService = clientService;
        this.adGroupService = adGroupService;
    }

    private static List<Keyword> convertRequest(GdAddKeywords request,
                                                Map<Long, DbStrategy> adGroupToCampaignStrategy,
                                                BigDecimal keywordsDefaultPrice) {
        return mapList(request.getAddItems(), item -> {
            Long adGroupId = item.getAdGroupId();
            DbStrategy strategy = adGroupToCampaignStrategy.get(adGroupId);
            return convertKeywordItem(item, strategy, keywordsDefaultPrice);
        });
    }

    private static Keyword convertKeywordItem(GdAddKeywordsItem item, @Nullable DbStrategy strategy,
                                              BigDecimal minPrice) {
        Keyword keyword = new Keyword()
                .withAdGroupId(item.getAdGroupId())
                .withPhrase(phraseWithoutAutotargetingPrefix(item.getKeyword()))
                .withIsAutotargeting(hasAutotargetingPrefix(item.getKeyword()))
                .withPrice(item.getPrice())
                .withPriceContext(item.getPriceContext())
                .withAutobudgetPriority(item.getAutobudgetPriority());

        if (keyword.getPrice() == null && strategy != null && !strategy.isAutoBudget() && !strategy.isSearchStop()) {
            keyword.setPrice(minPrice);
        }
        if (keyword.getPriceContext() == null && strategy != null && strategy.isDifferentPlaces()
                && !strategy.isAutoBudget() && !strategy.isNetStop()) {
            keyword.setPriceContext(minPrice);
        }
        if (keyword.getAutobudgetPriority() == null) {
            keyword.setAutobudgetPriority(MEDIUM_PRIORITY);
        }
        return keyword;
    }

    private static Set<Long> getReferencedAdGroupIds(GdAddKeywords request) {
        return listToSet(request.getAddItems(), GdAddKeywordsItem::getAdGroupId);
    }

    public List<Keyword> convertRequest(GdAddKeywords request, ClientId clientId) {
        Client client = checkNotNull(clientService.getClient(clientId));
        Currency clientCurrency = client.getWorkCurrency().getCurrency();
        BigDecimal keywordsDefaultPrice = clientCurrency.getMinPrice();

        Set<Long> referencedAdGroupIds = getReferencedAdGroupIds(request);

        Map<Long, DbStrategy> adGroupToCampaignStrategy =
                adGroupService.getStrategiesByAdGroupIds(clientId, referencedAdGroupIds);

        return convertRequest(request, adGroupToCampaignStrategy, keywordsDefaultPrice);
    }

}
