package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.MapUtils;

import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRelevanceMatchCategory;
import ru.yandex.direct.grid.processing.model.showcondition.GdAutotargetingStatInput;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionCategoryStat;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsOrder;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsOrderBy;
import ru.yandex.direct.grid.processing.model.statistics.GdCampaignStatisticsOrderByField;
import ru.yandex.direct.intapi.client.model.request.statistics.CampaignStatisticsRequest;
import ru.yandex.direct.intapi.client.model.request.statistics.ReportOptions;
import ru.yandex.direct.intapi.client.model.request.statistics.ReportType;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionColumn;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionConditionOperator;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionFilterColumn;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupBy;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;

import static java.util.Collections.reverseOrder;
import static java.util.Collections.singletonMap;
import static ru.yandex.direct.grid.processing.service.statistics.utils.StatisticsUtils.normalizePeriod;
import static ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionConditionOperator.EQ;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public final class AutotargetingStatConverter {

    private static final String RELEVANCE_MATCH = "relevance-match";

    private static final Set<ReportOptionColumn> COLUMNS = Set.of(
            ReportOptionColumn.SEARCH_QUERY,
            ReportOptionColumn.SHOWS,
            ReportOptionColumn.CLICKS,
            ReportOptionColumn.CONVERSIONS,
            ReportOptionColumn.COST
    );

    private static final Set<ReportOptionGroupBy> GROUP_BY = Set.of(
            ReportOptionGroupBy.CONTEXT_COND,
            ReportOptionGroupBy.TARGETING_CATEGORY,
            ReportOptionGroupBy.SEARCH_QUERY
    );

    private static final Map<ReportOptionFilterColumn, Map<ReportOptionConditionOperator, ?>> FILTERS = Map.of(
            ReportOptionFilterColumn.CONTEXT_TYPE, singletonMap(EQ, RELEVANCE_MATCH)
    );

    private static final Map<GdRelevanceMatchCategory, TargetingCategory> RELEVANCE_MATCH_TO_TARGETING = Map.of(
            GdRelevanceMatchCategory.EXACT_MARK, TargetingCategory.EXACT,
            GdRelevanceMatchCategory.ALTERNATIVE_MARK, TargetingCategory.ALTERNATIVE,
            GdRelevanceMatchCategory.COMPETITOR_MARK, TargetingCategory.COMPETITOR,
            GdRelevanceMatchCategory.BROADER_MARK, TargetingCategory.BROADER,
            GdRelevanceMatchCategory.ACCESSORY_MARK, TargetingCategory.ACCESSORY
    );

    private static final Map<TargetingCategory, GdRelevanceMatchCategory> RELEVANCE_MATCH_FROM_TARGETING =
            MapUtils.invertMap(RELEVANCE_MATCH_TO_TARGETING);

    public static final GdCampaignStatisticsOrderByField DEFAULT_ORDER_BY_FIELD = GdCampaignStatisticsOrderByField.COST;
    public static final GdCampaignStatisticsOrder DEFAULT_ORDER = GdCampaignStatisticsOrder.DESC;
    /**
     * "{@code "goalId": 13}" трактуется МОЛ как особое значение для получения статистики по целям из кампании.
     * Имеются в виду как цели из настроек стратегии, так и ключевые.
     * <p/>
     * <a href="https://st.yandex-team.ru/BSDEV-82509#620f7c495cbfbd6d098385cc">Подробнее в Трекере</a>
     */
    public static final long MASTER_REPORT_ANY_CAMPAIGN_GOAL_ID = 13L;

    public static CampaignStatisticsRequest toSearchQueriesRequest(
            GdAutotargetingStatInput input,
            GridGraphQLContext context
    ) {
        Long clientUid = context.getSubjectUser().getUid();
        Long operatorUid = context.getOperator().getUid();

        normalizePeriod(input.getPeriod(), context.getInstant());

        var reportOptions = new ReportOptions()
                .withReportType(ReportType.MOC)
                .withCampaignId(input.getCampaignId())
                .withDateFrom(input.getPeriod().getFrom())
                .withDateTo(input.getPeriod().getTo())
                .withColumns(COLUMNS)
                .withGroupBy(GROUP_BY)
                .withGroupByDate(ReportOptionGroupByDate.NONE);

        Map<ReportOptionConditionOperator, List<String>> targetingCategoryFilter;
        if (isEmpty(input.getRelevanceMatchCategory())) {
            targetingCategoryFilter = Map.of();
        } else {
            var targetingCategories = mapList(input.getRelevanceMatchCategory(), RELEVANCE_MATCH_TO_TARGETING::get);
            var categoriesStringList = targetingCategories.stream()
                    .map(TargetingCategory::getStringValue)
                    .collect(Collectors.toList());

            targetingCategoryFilter = singletonMap(EQ, categoriesStringList);
        }
        var reportFilters = new HashMap<>(FILTERS);
        reportFilters.put(ReportOptionFilterColumn.TARGETING_CATEGORY, targetingCategoryFilter);
        // Для получения конверсий только по целям из кампании, передаём goalId=13
        //   https://st.yandex-team.ru/BSDEV-82509#620f7c495cbfbd6d098385cc
        reportFilters.put(ReportOptionFilterColumn.GOAL_IDS, Map.of(EQ, List.of(MASTER_REPORT_ANY_CAMPAIGN_GOAL_ID)));
        reportOptions
                .withFilters(reportFilters);

        if (input.getLimitOffset() != null) {
            // Применяем лимит в запросе только в случае, когда запрашивается одна категория.
            //   Иначе мы не сможем регулировать лимит по каждой из категорий
            boolean singleCategoryRequested =
                    input.getRelevanceMatchCategory() != null &&
                            input.getRelevanceMatchCategory().size() == 1;
            if (singleCategoryRequested) {
                reportOptions
                        .withLimit(input.getLimitOffset().getLimit())
                        .withOffset(input.getLimitOffset().getOffset());
            }
        }

        if (input.getOrderBy() != null) {
            reportOptions
                    .withOrderByField(GdCampaignStatisticsOrderByField.toSource(input.getOrderBy().getField()))
                    .withOrder(GdCampaignStatisticsOrder.toSource(input.getOrderBy().getOrder()));
        } else {
            reportOptions
                    .withOrderByField(GdCampaignStatisticsOrderByField.toSource(DEFAULT_ORDER_BY_FIELD))
                    .withOrder(GdCampaignStatisticsOrder.toSource(DEFAULT_ORDER));
        }

        return new CampaignStatisticsRequest()
                .withReportOptions(reportOptions)
                .withOperatorUid(operatorUid)
                .withUid(clientUid);
    }

    public static Comparator<GdShowConditionCategoryStat> getCategoryComparator(
            @Nullable GdCampaignStatisticsOrderBy orderBy) {
        if (orderBy != null) {
            Function<GdShowConditionCategoryStat, BigDecimal> keyExtractor;
            var field = nvl(orderBy.getField(), DEFAULT_ORDER_BY_FIELD);
            switch (field) {
                case SHOWS:
                    keyExtractor = category -> category.getTotals().getShows();
                    break;
                case CLICKS:
                    keyExtractor = category -> category.getTotals().getClicks();
                    break;
                case CONVERSIONS:
                    keyExtractor = category -> category.getTotals().getConversions();
                    break;
                case COST_PER_CONVERSION:
                    keyExtractor = category -> category.getTotals().getCostPerConversion();
                    break;
                case COST:
                default:
                    keyExtractor = category -> category.getTotals().getCost();
                    break;
            }

            var order = nvl(orderBy.getOrder(), DEFAULT_ORDER);
            if (order == GdCampaignStatisticsOrder.DESC) {
                return Comparator.comparing(keyExtractor, reverseOrder());
            } else {
                return Comparator.comparing(keyExtractor);
            }
        }

        // дефолтное поведение оригинального метода
        return Comparator.comparing(category -> category.getTotals().getCost(), reverseOrder());
    }

    @Nullable
    public static GdRelevanceMatchCategory toGdRelevanceMatchCategory(String targetingCategoryStr) {
        TargetingCategory targetingCategory = TargetingCategory.fromStringValue(targetingCategoryStr);
        if (targetingCategory == TargetingCategory.UNDEFINED) {
            // Это не настоящая категория, возвращаем как null
            return null;
        }
        return RELEVANCE_MATCH_FROM_TARGETING.get(targetingCategory);
    }
}
