package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.bids.container.BidTargetType;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetAutoNetworkByCoverage;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByPosition;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByTrafficVolume;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetAutoBids;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetBids;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class BidsDataRequestConverter {
    public static List<SetAutoBidItem> convertRequestToSetAutoBidItems(GdSetAutoBids input) {
        if (input.getKeywordIds() == null || input.getKeywordIds().isEmpty()) {
            return Collections.emptyList();
        }
        SetAutoNetworkByCoverage setAutoNetworkByCoverage =
                ifNotNull(input.getNetworkByCoverage(),
                        networkByCoverage -> new SetAutoNetworkByCoverage()
                                .withMaxBid(networkByCoverage.getMaxBid())
                                .withContextCoverage(networkByCoverage.getContextCoverage())
                                .withIncreasePercent(networkByCoverage.getIncreasePercent())
                );

        SetAutoSearchByTrafficVolume setAutoSearchByTrafficVolume =
                ifNotNull(input.getSearchByTrafficVolume(),
                        searchByTrafficVolume -> new SetAutoSearchByTrafficVolume()
                                .withMaxBid(searchByTrafficVolume.getMaxBid())
                                .withTargetTrafficVolume(searchByTrafficVolume.getTargetTrafficVolume())
                                .withIncreasePercent(searchByTrafficVolume.getIncreasePercentage())
                                .withSetMaximum(searchByTrafficVolume.getSetMaximum())
                );

        List<SetAutoBidItem> result = new ArrayList<>(input.getKeywordIds().size());
        result.addAll(mapList(input.getKeywordIds(),
                id -> new SetAutoBidItem()
                        .withId(id)
                        .withSearchByPosition(new SetAutoSearchByPosition())
                        .withNetworkByCoverage(nvl(setAutoNetworkByCoverage, new SetAutoNetworkByCoverage()))
                        .withSearchByTrafficVolume(
                                nvl(setAutoSearchByTrafficVolume, new SetAutoSearchByTrafficVolume()))
                        .withScope((EnumSet<BidTargetType>) getScope(setAutoNetworkByCoverage,
                                setAutoSearchByTrafficVolume)))
        );

        return result;
    }

    public static List<SetBidItem> convertRequestToSetBidItems(GdSetBids input, Function<Long, ShowConditionType> showConditionTypeChooser) {
        return mapList(input.getShowConditionIds(), id -> new SetBidItem()
                .withId(id)
                .withAutobudgetPriority(input.getAutobudgetPriority())
                .withPriceContext(input.getExactPriceContext())
                .withPriceSearch(input.getExactPriceSearch())
                .withShowConditionType(showConditionTypeChooser.apply(id))
        );
    }


    private static Set<BidTargetType> getScope(@Nullable SetAutoNetworkByCoverage setAutoNetworkByCoverage,
                                               @Nullable SetAutoSearchByTrafficVolume setAutoSearchByTrafficVolume) {
        if (setAutoNetworkByCoverage != null && setAutoSearchByTrafficVolume != null) {
            return EnumSet.of(BidTargetType.CONTEXT, BidTargetType.SEARCH_BY_TRAFFIC_VOLUME);
        }
        if (setAutoSearchByTrafficVolume != null) {
            return EnumSet.of(BidTargetType.SEARCH_BY_TRAFFIC_VOLUME);
        }
        if (setAutoNetworkByCoverage != null) {
            return EnumSet.of(BidTargetType.CONTEXT);
        }
        throw new IllegalStateException("Should be at least one of fields: [searchByTrafficVolume, networkByCoverage]");
    }
}
