package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.FindAndReplaceKeywordsCacheFilterData;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPreviewPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPreviewPayloadItem;
import ru.yandex.direct.grid.processing.service.showcondition.container.FindAndReplaceKeywordsCacheRecordInfo;
import ru.yandex.direct.utils.FunctionalUtils;

import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasAutotargetingPrefix;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class FindAndReplaceDataConverter {

    @Nullable
    public static List<String> convertChangedMinusKeywordsToList(@Nullable String changedMinusKeywords) {
        if (changedMinusKeywords == null) {
            return null;
        }

        if (changedMinusKeywords.trim().isEmpty()) {
            return Collections.emptyList();
        }

        return StreamEx.split(changedMinusKeywords, "\\s+")
                .filter(StringUtils::isNotEmpty)
                .map(String::trim)
                .toList();
    }

    public static FindAndReplaceKeywordsCacheRecordInfo getCacheRecordInfo(ClientId clientId,
                                                                           GdFindAndReplaceKeywords input) {
        return new FindAndReplaceKeywordsCacheRecordInfo(clientId.asLong(), input.getCacheKey(),
                new FindAndReplaceKeywordsCacheFilterData()
                        .withKeywordIds(ImmutableSet.copyOf(input.getKeywordIds()))
                        .withSearchText(input.getSearchText())
                        .withChangeText(input.getChangeText())
                        .withChangeMode(input.getChangeMode())
                        .withFields(ImmutableSet.copyOf(input.getFields()))
                        .withSearchOptions(input.getSearchOptions().copy())
        );
    }

    public static GdFindAndReplaceKeywordsPreviewPayload getEmptyPreviewPayload() {
        return new GdFindAndReplaceKeywordsPreviewPayload()
                .withCacheKey("")
                .withTotalCount(0)
                .withKeywordIds(Collections.emptyList())
                .withRowset(Collections.emptyList());
    }

    public static GdFindAndReplaceKeywordsPayload getEmptyPayload() {
        return new GdFindAndReplaceKeywordsPayload()
                .withCacheKey("")
                .withUpdatedTotalCount(0)
                .withUpdatedKeywordIds(Collections.emptyList())
                .withRowset(Collections.emptyList())
                .withAffectedKeywords(Collections.emptyList());
    }

    public static List<Keyword> toCoreKeywords(List<Long> keywordIds,
                                               List<GdFindAndReplaceKeywordsPreviewPayloadItem> updateKeywords) {
        Map<Long, GdFindAndReplaceKeywordsPreviewPayloadItem> updateKeywordsById =
                FunctionalUtils.listToMap(updateKeywords, GdFindAndReplaceKeywordsPreviewPayloadItem::getId);

        return mapList(keywordIds, id -> new Keyword()
                .withId(id)
                .withPhrase(ifNotNull(updateKeywordsById.get(id), FindAndReplaceDataConverter::toPhrase))
                .withIsAutotargeting(ifNotNull(updateKeywordsById.get(id),
                        FindAndReplaceDataConverter::isAutotargeting)));
    }

    public static String toPhrase(GdFindAndReplaceKeywordsPreviewPayloadItem item) {
        String keyword = nvl(item.getChangedKeyword(), item.getKeyword());
        List<String> minusKeywords = nvl(item.getChangedMinusKeywords(), item.getMinusKeywords());
        return KeywordUtils.mergePhraseWithMinusPhrases(keyword, minusKeywords);
    }

    public static boolean isAutotargeting(GdFindAndReplaceKeywordsPreviewPayloadItem item) {
        String keyword = nvl(item.getChangedKeyword(), item.getKeyword());
        return hasAutotargetingPrefix(keyword);
    }
}
