package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.advq.query.ast.Word;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdChangeKeywordsCaseMode;
import ru.yandex.direct.libs.keywordutils.model.AnyKeyword;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.OrderedKeyword;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class KeywordCaseConverter {

    public KeywordWithMinuses changeCase(KeywordWithMinuses keywordWithMinuses, GdChangeKeywordsCaseMode replaceMode,
                                         boolean replaceKeyword, boolean replaceMinuses) {
        KeywordCopyAndModifyCase keywordCopyAndModifyCase = new KeywordCopyAndModifyCase(replaceMode, replaceKeyword,
                replaceMinuses);

        return keywordCopyAndModifyCase.copyAndModify(keywordWithMinuses);
    }

    /**
     * Принцип работы: на вход получаем древовидную структуру KeywordWithMinuses
     * <p>
     * Создаем структурную копию, а когда доходим до листа, где хранится непосредственно текст слова, применяем к нему
     * модификацию регистра
     * <p>
     * В результате получаем такое же представление ключевой фразы с точки зрения операторов !"[]+ и минус-слов, но с
     * нужным регистром
     * <p>
     *
     * Пример:
     * "!купить -слона" при преобразовании в верний регистр "!КУПИТЬ -СЛОНА"
     */
    private static class KeywordCopyAndModifyCase {
        private final GdChangeKeywordsCaseMode replaceMode;
        private final boolean replaceKeyword;
        private final boolean replaceMinuses;

        private KeywordCopyAndModifyCase(GdChangeKeywordsCaseMode replaceMode, boolean replaceKeyword,
                                         boolean replaceMinuses) {
            this.replaceMode = replaceMode;
            this.replaceKeyword = replaceKeyword;
            this.replaceMinuses = replaceMinuses;
        }

        private KeywordWithMinuses copyAndModify(KeywordWithMinuses keywordWithMinuses) {
            Keyword newKeyword = replaceKeyword ? copyKeyword(keywordWithMinuses.getKeyword()) :
                    keywordWithMinuses.getKeyword();

            List<Keyword> newMinusKeywords = replaceMinuses ? mapList(keywordWithMinuses.getMinusKeywords(),
                    this::copyKeyword) : keywordWithMinuses.getMinusKeywords();

            return new KeywordWithMinuses(newKeyword, newMinusKeywords);
        }

        private Keyword copyKeyword(Keyword keyword) {
            List<AnyKeyword> newAllKeywords = EntryStream.of(keyword.getAllKeywords())
                    .mapKeyValue(this::copyAnyKeyword)
                    .toList();

            return new Keyword(keyword.isQuoted(), newAllKeywords);
        }

        private AnyKeyword copyAnyKeyword(int indexOfKeyword, AnyKeyword anyKeyword) {
            boolean isFirstBlockInEntireKeyword = indexOfKeyword == 0;
            if (anyKeyword instanceof SingleKeyword) {
                return copySingleKeyword((SingleKeyword) anyKeyword, isFirstBlockInEntireKeyword);
            } else if (anyKeyword instanceof OrderedKeyword) {
                return copyOrderedKeyword((OrderedKeyword) anyKeyword, isFirstBlockInEntireKeyword);
            } else {
                throw new UnsupportedOperationException("Unexpected type for anyKeyword: " + anyKeyword.toString());
            }
        }

        private OrderedKeyword copyOrderedKeyword(OrderedKeyword orderedKeyword, boolean isFirstBlock) {
            List<SingleKeyword> newSingleKeywords = EntryStream.of(orderedKeyword.getSingleKeywords())
                    .mapKeyValue((index, value) -> copySingleKeyword(value, isFirstBlock && index == 0))
                    .toList();

            return new OrderedKeyword(newSingleKeywords);
        }

        private SingleKeyword copySingleKeyword(SingleKeyword singleKeyword, boolean firstWordInEntireKeyword) {
            Word newWord = copyWord(singleKeyword.getWord(), firstWordInEntireKeyword);

            return new SingleKeyword(newWord);
        }

        private Word copyWord(Word word, boolean firstWordInEntireKeyword) {
            String newText = modifyText(word.getText(), firstWordInEntireKeyword);

            return new Word(word.getKind(), newText);
        }

        private String modifyText(String text, boolean firstWordInEntireKeyword) {
            switch (replaceMode) {
                case UPPERCASE:
                    return text.toUpperCase();
                case LOWERCASE:
                    return text.toLowerCase();
                case CAPITALIZE_ALL_WORDS:
                    return capitalize(text.toLowerCase());
                case CAPITALIZE_FIRST_WORD:
                    return firstWordInEntireKeyword ?
                            capitalize(text.toLowerCase()) :
                            text.toLowerCase();
                default:
                    throw new UnsupportedOperationException("Unsupported replaceMode: " + replaceMode.toString());
            }
        }
    }
}
