package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowCondition;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionType;
import ru.yandex.direct.grid.processing.model.showcondition.GdAuctionDataItem;
import ru.yandex.direct.grid.processing.model.showcondition.GdPokazometerCostsAndClicksItem;
import ru.yandex.direct.grid.processing.model.showcondition.GdPokazometerData;
import ru.yandex.direct.grid.processing.model.showcondition.GdPokazometerPriceForCoverageItem;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.pokazometer.PhraseResponse;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasNoAutotargetingPrefix;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;

@ParametersAreNonnullByDefault
public class KeywordConverter {

    private static final List<Long> TRAFFIC_VOLUME_PRESET = Arrays.asList(5L, 15L, 75L, 100L);

    /**
     * Используется для получения данных аукциона из Торгов.
     * @return список id только ключевых слов
     */
    public static List<Long> getKeywordsIds(LimitOffset limitOffset, List<GdiShowCondition> conditions) {
        List<Long> keywordsIds = filterAndMapList(conditions,
                c -> c.getType() == GdiShowConditionType.KEYWORD &&
                        (hasNoAutotargetingPrefix(c.getIsAutotargeting())),
                GdiShowCondition::getId);

        return limitOffset.apply(keywordsIds);
    }

    public static GdPokazometerData toGdPokazometerData(KeywordBidPokazometerData keywordBidPokazometerData) {
        BiFunction<PhraseResponse.Coverage, Money, GdPokazometerPriceForCoverageItem>
                toGdPokazometerPriceForCoverageItem =
                (coverage, price) -> new GdPokazometerPriceForCoverageItem()
                        .withCoverage(coverage.getPercentage())
                        .withPrice(price.bigDecimalValue());

        BiFunction<Money, Integer, GdPokazometerCostsAndClicksItem> toGdPokazometerDataItemWithClicksAndPrice =
                (price, clicks) -> new GdPokazometerCostsAndClicksItem()
                        .withClicks(clicks)
                        .withPrice(price.bigDecimalValue());

        List<GdPokazometerPriceForCoverageItem> gdPokazometerPriceForCoverageItems =
                EntryStream.of(keywordBidPokazometerData.getCoverageWithPrices())
                        .mapKeyValue(toGdPokazometerPriceForCoverageItem)
                        .sortedBy(GdPokazometerPriceForCoverageItem::getCoverage)
                        .toList();

        List<GdPokazometerCostsAndClicksItem> gdPokazometerCostsAndClicksItems =
                EntryStream.of(keywordBidPokazometerData.getAllCostsAndClicks())
                        .mapKeyValue(toGdPokazometerDataItemWithClicksAndPrice)
                        .sortedBy(GdPokazometerCostsAndClicksItem::getClicks)
                        .toList();

        return new GdPokazometerData()
                .withPriceForCoverage(gdPokazometerPriceForCoverageItems)
                .withAllCostsAndClicks(gdPokazometerCostsAndClicksItems)
                .withMaxShows(keywordBidPokazometerData.getAllCostsAndClicks().isEmpty() ? null : Collections
                        .max(keywordBidPokazometerData.getAllCostsAndClicks().values()));
    }

    public static List<GdAuctionDataItem> getAuctionsDataItems(KeywordTrafaretData data) {
        Map<Long, GdAuctionDataItem> gdAuctionDataItemMap = data.getBidItems().stream()
                .map(item -> new GdAuctionDataItem()
                        .withTrafficVolume(
                                BigDecimal.valueOf(item.getPositionCtrCorrection()).scaleByPowerOfTen(-4).longValue())
                        .withBid(item.getBid().bigDecimalValue())
                        .withAmnestyPrice(item.getPrice().bigDecimalValue()))
                .collect(Collectors.toMap(GdAuctionDataItem::getTrafficVolume, identity(),
                        (auctionDataItem1, auctionDataItem2) -> auctionDataItem1));

        setShowInTableForItems(gdAuctionDataItemMap);
        return new ArrayList<>(gdAuctionDataItemMap.values());
    }

    private static void setShowInTableForItems(Map<Long, GdAuctionDataItem> items) {
        LongSummaryStatistics trafficVolumesStat = items.values().stream()
                .mapToLong(GdAuctionDataItem::getTrafficVolume)
                .summaryStatistics();

        Set<Long> showInTable = new HashSet<>(TRAFFIC_VOLUME_PRESET);
        showInTable.add(trafficVolumesStat.getMin());
        showInTable.add(trafficVolumesStat.getMax());

        showInTable.stream()
                .filter(x -> x <= trafficVolumesStat.getMax() && x >= trafficVolumesStat.getMin())
                .forEach(tv -> items.compute(tv, (k, v) -> v.withShowInTable(true)));
    }

}
