package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.util.List;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.UpdatedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.service.KeywordsUpdateOperation;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridKeywordsParser;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsPayloadItem;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywordsPayloadItem;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class KeywordsResponseConverter {

    public static List<GdAddKeywordsPayloadItem> getSuccessfullyAddedKeywords(MassResult<AddedKeywordInfo> result) {
        return StreamEx.of(result.getResult())
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .map(keywordInfo -> new GdAddKeywordsPayloadItem()
                        .withAdGroupId(keywordInfo.getAdGroupId())
                        .withKeywordId(keywordInfo.getId()))
                .toList();
    }

    public static List<GdUpdateKeywordsPayloadItem> getSuccessfullyUpdatedKeywords(
            GridKeywordsParser gridKeywordsParser,
            MassResult<UpdatedKeywordInfo> result) {
        return StreamEx.of(result.getResult())
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .map(updatedKeywordInfo -> {
                    KeywordWithMinuses keywordWithMinuses =
                            gridKeywordsParser.parseKeyword(updatedKeywordInfo.getResultPhrase());

                    return new GdUpdateKeywordsPayloadItem()
                            .withId(updatedKeywordInfo.getId())
                            .withDuplicate(updatedKeywordInfo.isDeleted())
                            .withIsSuspended(updatedKeywordInfo.getIsSuspended())
                            .withKeyword(keywordWithMinuses.getKeyword().toString())
                            .withMinusKeywords(convertMinuses(keywordWithMinuses.getMinusKeywords()));
                })
                .toList();
    }

    public static List<GdUpdateKeywordsPayloadItem> getAffectedKeywords(
            GridKeywordsParser gridKeywordsParser,
            KeywordsUpdateOperation keywordsUpdateOperation) {
        List<GdUpdateKeywordsPayloadItem> affectedKeywords =
                getSuccessfullyUpdatedOtherExistingKeywords(gridKeywordsParser, keywordsUpdateOperation);

        if (keywordsUpdateOperation.getPotentiallyDeletedKeywordIds() != null) {
            List<GdUpdateKeywordsPayloadItem> deletedKeywords =
                    mapList(keywordsUpdateOperation.getPotentiallyDeletedKeywordIds(),
                            id -> new GdUpdateKeywordsPayloadItem()
                                    .withId(id)
                                    .withDuplicate(true)
                    );
            affectedKeywords.addAll(deletedKeywords);
        }

        return affectedKeywords;
    }

    public static List<GdUpdateKeywordsPayloadItem> getSuccessfullyUpdatedOtherExistingKeywords(
            GridKeywordsParser gridKeywordsParser,
            KeywordsUpdateOperation keywordsUpdateOperation) {
        return mapList(keywordsUpdateOperation.getAffectedKeywordInfoList(), affectedKeywordInfo -> {
            KeywordWithMinuses keywordWithMinuses =
                    gridKeywordsParser.parseKeyword(affectedKeywordInfo.getSourcePhrase());
            List<String> minuses = convertMinuses(keywordWithMinuses.getMinusKeywords());
            minuses.addAll(affectedKeywordInfo.getAddedMinuses());

            return new GdUpdateKeywordsPayloadItem()
                    .withId(affectedKeywordInfo.getId())
                    .withDuplicate(false)
                    .withIsSuspended(affectedKeywordInfo.getIsSuspended())
                    .withKeyword(keywordWithMinuses.getKeyword().toString())
                    .withMinusKeywords(minuses);
        });
    }

    public static List<String> convertMinuses(List<Keyword> minusKeywords) {
        return mapList(minusKeywords, Keyword::toString);
    }
}
