package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingCondition;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionFilter;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionRuleItem;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionType;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingFilter;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiBidsRetargeting;
import ru.yandex.direct.grid.core.entity.showcondition.service.GridShowConditionConstants;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.processing.model.goal.GdGoalTruncated;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargeting;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingAccess;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingBaseStatus;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingCondition;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionGoalsInfo;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleItem;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleItemReq;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleType;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionTruncated;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionType;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingContext;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingFeatures;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingOrderBy;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingOrderByField;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingWithTotals;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditionItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateRetargetingConditionItem;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionAutobudgetPriority;
import ru.yandex.direct.grid.processing.service.goal.GoalDataConverter;
import ru.yandex.direct.grid.processing.service.showcondition.container.RetargetingsCacheFilterData;
import ru.yandex.direct.grid.processing.service.showcondition.container.RetargetingsCacheRecordInfo;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.web.core.model.retargeting.CryptaInterestTypeWeb;

import static ru.yandex.direct.grid.core.entity.showcondition.service.GridRetargetingConditionService.DEFAULT_AUDIENCE_DOMAIN;
import static ru.yandex.direct.grid.core.entity.showcondition.service.GridRetargetingConditionService.GEO_SEGMENT_SUBTYPES;
import static ru.yandex.direct.grid.core.util.stats.GridStatNew.addZeros;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getCountOfTrueBooleanValues;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.recalcTotalStatsForUnitedGrid;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

public class RetargetingConverter {

    public static RetargetingsCacheRecordInfo toRetargetingsCacheRecordInfo(
            long clientId,
            GdRetargetingsContainer input) {
        return new RetargetingsCacheRecordInfo(clientId, input.getCacheKey(),
                new RetargetingsCacheFilterData()
                        .withFilter(input.getFilter())
                        .withOrderBy(input.getOrderBy())
                        .withStatRequirements(input.getStatRequirements()));
    }

    public static GdRetargetingContext toGdRetargetingContext(GdRetargetingWithTotals retargetingWithTotals,
                                                              GdRetargetingFilter inputFilter,
                                                              boolean cpcAndCpmOnOneGridEnabled) {
        var rowsetFull = retargetingWithTotals.getGdRetargetings();
        var totalStats = nvl(retargetingWithTotals.getTotalStats(),
                calcTotalStats(mapList(rowsetFull, GdRetargeting::getStats)));

        if (cpcAndCpmOnOneGridEnabled) {
            Map<GdCampaignType, List<GdEntityStats>> campaignTypeToStats = StreamEx.of(rowsetFull)
                    .mapToEntry(retarget -> retarget.getAdGroup().getCampaign().getType(), GdRetargeting::getStats)
                    .grouping();
            recalcTotalStatsForUnitedGrid(totalStats, campaignTypeToStats);
        }

        // показываем предупреждение если есть тоталы из БД и был фильтр по коду (либо мог быть)
        var totalStatsWithoutFiltersWarn = retargetingWithTotals.getTotalStats() != null
                && (rowsetFull.size() < GridShowConditionConstants.getMaxConditionRows() || hasAnyCodeFilter(inputFilter));
        return new GdRetargetingContext()
                .withTotalCount(rowsetFull.size())
                .withHasObjectsOverLimit(rowsetFull.size() >= GridShowConditionConstants.getMaxConditionRows())
                .withRetargetingIds(listToSet(rowsetFull, GdRetargeting::getRetargetingId))
                .withFeatures(toGdRetargetingFeatures(rowsetFull))
                .withTotalStatsWithoutFiltersWarn(totalStatsWithoutFiltersWarn)
                .withTotalStats(totalStats);
    }

    public static GdRetargetingContext toGdRetargetingContext(List<GdRetargeting> rowset) {
        return new GdRetargetingContext()
                .withTotalCount(rowset.size())
                .withHasObjectsOverLimit(rowset.size() >= GridShowConditionConstants.getMaxConditionRows())
                .withRetargetingIds(listToSet(rowset, GdRetargeting::getRetargetingId))
                .withFeatures(toGdRetargetingFeatures(rowset));
    }

    /**
     * Присутствуют ли в фильтре поля, которые фильтруют условия в коде, а не в запросе к БД
     */
    private static boolean hasAnyCodeFilter(GdRetargetingFilter inputFilter) {
        return !CollectionUtils.isEmpty(inputFilter.getStatusIn())
                || !CollectionUtils.isEmpty(inputFilter.getReasonsContainSome());
    }

    static GdRetargetingFeatures toGdRetargetingFeatures(List<GdRetargeting> rowsetFull) {
        List<GdRetargetingAccess> gdRetargetingsAccesses = mapList(rowsetFull, GdRetargeting::getAccess);
        int canBeEditCount =
                getCountOfTrueBooleanValues(gdRetargetingsAccesses, GdRetargetingAccess::getCanEdit);

        return new GdRetargetingFeatures()
                .withCanEditCount(canBeEditCount);
    }

    public static GdRetargeting toGdRetargeting(GdiBidsRetargeting gdiBidsRetargeting,
                                                GdCampaignTruncated campaign,
                                                GdAdGroupTruncated adGroup,
                                                boolean cpcAndCpmOnOneGridEnabled) {
        boolean canEdit = !campaign.getStatus().getReadOnly()
                && !adGroup.getStatus().getArchived()
                && campaign.getType() != GdCampaignType.CPM_PRICE;

        GdiEntityStats internalStats = gdiBidsRetargeting.getStat() != null
                ? gdiBidsRetargeting.getStat()
                : addZeros(new GdiEntityStats());
        return new GdRetargeting()
                .withRetargetingId(gdiBidsRetargeting.getRetargetingId())
                .withRetargetingConditionId(gdiBidsRetargeting.getRetargetingConditionId())
                .withAdGroupId(gdiBidsRetargeting.getAdGroupId())
                .withCampaignId(gdiBidsRetargeting.getCampaignId())
                .withPriceContext(gdiBidsRetargeting.getPriceContext())
                .withAutoBudgetPriority(
                        GdShowConditionAutobudgetPriority.fromSource(gdiBidsRetargeting.getAutoBudgetPriority()))
                .withReach(gdiBidsRetargeting.getReach())
                .withIsSuspended(gdiBidsRetargeting.getIsSuspended())
                .withAggregatedStatus(gdiBidsRetargeting.getAggregatedStatus())
                .withAggregatedStatusInfo(gdiBidsRetargeting.getAggregatedStatusInfo())
                .withRetargetingCondition(ifNotNull(gdiBidsRetargeting.getRetargetingCondition(),
                        RetargetingConverter::toGdRetargetingConditionTruncated))
                .withAdGroup(adGroup)
                .withStats(StatHelper
                        .internalStatsToOuter(internalStats, campaign.getType(), cpcAndCpmOnOneGridEnabled))
                .withAccess(new GdRetargetingAccess()
                        .withCanEdit(canEdit));
    }

    public static void addAdGroupAndAccessInfo(GdRetargeting gdRetargeting,
                                               GdCampaignTruncated campaign, GdAdGroupTruncated adGroup) {
        gdRetargeting.setAdGroup(adGroup);
        gdRetargeting.setAccess(new GdRetargetingAccess()
                .withCanEdit(!campaign.getStatus().getReadOnly()
                        && !adGroup.getStatus().getArchived()
                        && campaign.getType() != GdCampaignType.CPM_PRICE));
    }

    public static List<GdiRetargetingOrderBy> toInternalRetargetingOrderBy(List<GdRetargetingOrderBy> orderBy) {
        return mapList(orderBy, elem -> new GdiRetargetingOrderBy()
                .withField(GdRetargetingOrderByField.toSource(elem.getField()))
                .withOrder(elem.getOrder()));
    }

    public static GdiRetargetingFilter toInternalRetargetingFilter(GdRetargetingFilter filter) {
        return new GdiRetargetingFilter()
                .withRetargetingIdIn(filter.getRetargetingIdIn())
                .withRetargetingIdNotIn(filter.getRetargetingIdNotIn())
                .withRetargetingConditionIdIn(filter.getRetargetingConditionIdIn())
                .withRetargetingConditionIdNotIn(filter.getRetargetingConditionIdNotIn())
                .withCampaignIdIn(filter.getCampaignIdIn())
                .withAdGroupIdIn(filter.getAdGroupIdIn())
                .withNameContains(filter.getNameContains())
                .withTypeIn(mapSet(filter.getTypeIn(), RetargetingConverter::convertToGdiRetargetingConditionType))
                .withTypeNotIn(mapSet(filter.getTypeNotIn(),
                        RetargetingConverter::convertToGdiRetargetingConditionType))
                .withNameNotContains(filter.getNameNotContains())
                .withStatusIn(mapSet(filter.getStatusIn(), GdRetargetingBaseStatus::toSource))
                .withMinPriceContext(filter.getMinPriceContext())
                .withMaxPriceContext(filter.getMaxPriceContext())
                .withInterest(filter.getInterest())
                .withStats(StatHelper.toInternalStatsFilter(filter.getStats()))
                .withReasonsContainSome(filter.getReasonsContainSome());
    }

    public static GdRetargetingCondition toGdRetargetingCondition(GdiRetargetingCondition internal,
                                                                  Map<Long, GdCampaignTruncated> campaignsInfo,
                                                                  Map<Long, List<Long>> campaignIdsWithRetargetingBidModifierByConditionId) {
        Set<Long> campaignIds = ImmutableSet.<Long>builder()
                .addAll(internal.getCampaignIds())
                .addAll(campaignIdsWithRetargetingBidModifierByConditionId
                        .getOrDefault(internal.getRetargetingConditionId(), Collections.emptyList()))
                .build();

        List<GdCampaignTruncated> campaigns =
                filterAndMapList(campaignIds, campaignsInfo::containsKey, campaignsInfo::get);

        return toGdRetargetingConditionBase(internal)
                .withHasUnavailableGoals(internal.getHasUnavailableGoals())
                .withAvailableForRetargeting(!internal.getNegative())
                .withGoalsInfo(getRetargetingConditionGoalsInfo(internal))
                .withAdGroupIds(internal.getAdGroupIds())
                .withCampaigns(campaigns);
    }

    private static GdRetargetingConditionTruncated toGdRetargetingConditionTruncated(GdiRetargetingCondition internal) {
        return toGdRetargetingConditionBase(internal);
    }

    private static GdRetargetingCondition toGdRetargetingConditionBase(GdiRetargetingCondition internal) {
        return new GdRetargetingCondition()
                .withRetargetingConditionId(internal.getRetargetingConditionId())
                .withType(convertToGdRetargetingConditionType(internal.getType()))
                .withName(internal.getName())
                .withInterest(internal.getInterest())
                .withDescription(internal.getDescription())
                .withConditionRules(
                        mapList(internal.getConditionRules(), RetargetingConverter::toGdRetargetingConditionRuleItem));
    }

    private static GdRetargetingConditionGoalsInfo getRetargetingConditionGoalsInfo(GdiRetargetingCondition internal) {
        return new GdRetargetingConditionGoalsInfo()
                .withHasGeoSegments(internal.getHasGeoSegments())
                .withGoalDomains(nvl(internal.getGoalDomains(), Collections.emptySet()))
                .withGoalIds(nvl(internal.getGoalIds(), Collections.emptySet()))
                .withHasUnavailableGoals(internal.getHasUnavailableGoals());
    }

    public static GdRetargetingConditionRuleItem toGdRetargetingConditionRuleItem(
            GdiRetargetingConditionRuleItem internal) {
        return new GdRetargetingConditionRuleItem()
                .withGoals(mapList(internal.getGoals(), GoalDataConverter::toGdGoalTruncated))
                .withType(GdRetargetingConditionRuleType.fromSource(internal.getType()))
                .withInterestType(internal.getInterestType())
                .withSectionId(internal.getSectionId());
    }

    public static GdiRetargetingConditionRuleItem toGdiRetargetingConditionRuleItem(Rule coreRule) {
        return ifNotNull(JsonUtils.toJsonNullable(coreRule),
                t -> JsonUtils.fromJson(t, GdiRetargetingConditionRuleItem.class));
    }

    /**
     * Подходит только для маппинга доступных (несохраненных) шорткатов
     */
    public static GdRetargetingCondition toGdRetargetingCondition(RetargetingCondition internal) {
        return new GdRetargetingCondition()
                .withRetargetingConditionId(internal.getId())
                .withType(toGdRetargetingConditionType(internal.getType()))
                .withName(internal.getName())
                .withInterest(internal.getInterest())
                .withDescription(internal.getDescription())
                .withConditionRules(toGdRetargetingConditionRules(internal.getRules()))
                .withHasUnavailableGoals(internal.getAvailable() == null || !internal.getAvailable())
                .withAvailableForRetargeting(!internal.getNegative())
                .withGoalsInfo(getGdRetargetingConditionGoalsInfo(internal))
                .withAdGroupIds(Collections.emptySet())
                .withCampaigns(Collections.emptyList());
    }

    private static GdRetargetingConditionGoalsInfo getGdRetargetingConditionGoalsInfo(RetargetingCondition internal) {
        List<Goal> goals = internal.collectGoals();

        goals.stream()
                .filter(goal -> goal.getType() == GoalType.AUDIENCE)
                .forEach(x -> x.setDomain(DEFAULT_AUDIENCE_DOMAIN));
        var goalIds = listToSet(goals, Goal::getId);
        var goalDomains = StreamEx.of(goals)
                .map(Goal::getDomain)
                .nonNull()
                .remove(String::isEmpty)
                .toSet();
        var hasGeoSegments = goals.stream()
                .map(Goal::getSubtype)
                .anyMatch(GEO_SEGMENT_SUBTYPES::contains);

        return new GdRetargetingConditionGoalsInfo()
                .withHasGeoSegments(hasGeoSegments)
                .withGoalDomains(goalDomains)
                .withGoalIds(goalIds)
                .withHasUnavailableGoals(internal.getAvailable() == null || !internal.getAvailable());
    }

    public static GdiRetargetingConditionFilter toInternalRetargetingConditionFilter(Long clientId,
                                                                                     GdRetargetingConditionFilter retargetingFilter) {
        return new GdiRetargetingConditionFilter()
                .withClientId(clientId)
                .withRetargetingConditionIdIn(retargetingFilter.getRetargetingConditionIdIn())
                .withRetargetingConditionIdNotIn(retargetingFilter.getRetargetingConditionIdNotIn())
                .withRetargetingConditionTypeIn(toInternalTypeSet(retargetingFilter.getRetargetingConditionTypeIn()))
                .withNameIn(retargetingFilter.getNameIn())
                .withNameNotIn(retargetingFilter.getNameNotIn())
                .withIsInterest(retargetingFilter.getInterest())
                .withIsNegative(retargetingFilter.getNegative())
                .withIsAutoRetargeting(retargetingFilter.getAutoRetargeting());
    }

    public static Set<GdiRetargetingConditionType> toInternalTypeSet(Set<GdRetargetingConditionType> typeSet) {
        if (CollectionUtils.isEmpty(typeSet)) {
            //В случае, если фильтр по типу не задан, явно указываем все доступные "внешние типы"
            return StreamEx.of(GdRetargetingConditionType.values())
                    .map(RetargetingConverter::convertToGdiRetargetingConditionType)
                    .toSet();
        }

        return mapSet(typeSet, RetargetingConverter::convertToGdiRetargetingConditionType);
    }

    private static GdRetargetingConditionType convertToGdRetargetingConditionType(GdiRetargetingConditionType type) {
        return GdRetargetingConditionType.valueOf(type.name());
    }

    private static GdiRetargetingConditionType convertToGdiRetargetingConditionType(GdRetargetingConditionType type) {
        return GdiRetargetingConditionType.valueOf(type.name());
    }

    public static RetargetingCondition toCoreRetargetingCondition(ClientId clientId,
                                                                  GdCreateRetargetingConditionItem retargetingConditionItem) {
        RetargetingCondition retargetingCondition = new RetargetingCondition();
        retargetingCondition
                .withName(retargetingConditionItem.getName())
                .withDescription(retargetingConditionItem.getDescription())
                .withType(toConditionType(retargetingConditionItem.getType()))
                .withClientId(clientId.asLong())
                .withInterest(retargetingConditionItem.getIsInterest())
                .withRules(mapList(retargetingConditionItem.getConditionRules(),
                        RetargetingConverter::toCoreRetargetingConditionRule));
        return retargetingCondition;
    }

    public static RetargetingCondition toCoreRetargetingCondition(ClientId clientId,
                                                                  GdUpdateRetargetingConditionItem retargetingConditionItem) {
        RetargetingCondition retargetingCondition = new RetargetingCondition();
        retargetingCondition
                .withId(retargetingConditionItem.getRetargetingConditionId())
                .withName(retargetingConditionItem.getName())
                .withDescription(retargetingConditionItem.getDescription())
                .withType(toConditionType(retargetingConditionItem.getType()))
                .withClientId(clientId.asLong())
                .withRules(mapList(retargetingConditionItem.getConditionRules(),
                        RetargetingConverter::toCoreRetargetingConditionRule));
        return retargetingCondition;
    }

    private static ConditionType toConditionType(GdRetargetingConditionType gdRetargetingConditionType) {
        return ConditionType.valueOf(gdRetargetingConditionType.name().toLowerCase());
    }

    private static GdRetargetingConditionType toGdRetargetingConditionType(ConditionType conditionType) {
        return GdRetargetingConditionType.valueOf(conditionType.name().toUpperCase());
    }

    public static RetargetingCondition toCoreRetargetingCondition(ClientId clientId,
                                                                  List<GdRetargetingConditionRuleItemReq> retargetingConditionRules) {
        RetargetingCondition retargetingCondition = new RetargetingCondition();
        List<Rule> rules = mapList(retargetingConditionRules, RetargetingConverter::toCoreRetargetingConditionRule);
        retargetingCondition
                .withClientId(clientId.asLong())
                .withRules(rules);
        return retargetingCondition;
    }

    public static Rule toCoreRetargetingConditionRule(GdRetargetingConditionRuleItemReq gridRule) {
        List<Goal> goals = mapList(gridRule.getGoals(), GoalDataConverter::toCoreGoal);
        return new Rule()
                .withType(GdRetargetingConditionRuleType.toSource(gridRule.getType()))
                .withInterestType(ifNotNull(gridRule.getInterestType(), CryptaInterestTypeWeb::toCoreType))
                .withGoals(goals);
    }

    public static Rule toCaCoreRetargetingConditionRule(List<Goal> goals) {
        return new Rule()
                .withType(RuleType.OR)
                .withInterestType(CryptaInterestType.all)
                .withGoals(goals);
    }

    private static List<Rule> toCoreRetargetingConditionRules(
            List<GdRetargetingConditionRuleItemReq> gdRetargetingConditionRules) {
        return mapList(gdRetargetingConditionRules, RetargetingConverter::toCoreRetargetingConditionRule);
    }

    private static GdRetargetingConditionRuleItem toGdRetargetingConditionRule(Rule rule) {
        List<GdGoalTruncated> goals = mapList(rule.getGoals(), GoalDataConverter::toGdGoalTruncated);
        return new GdRetargetingConditionRuleItem()
                .withType(GdRetargetingConditionRuleType.fromSource(rule.getType()))
                .withInterestType(rule.getInterestType())
                .withGoals(goals);
    }

    private static List<GdRetargetingConditionRuleItem> toGdRetargetingConditionRules(List<Rule> rules) {
        return mapList(rules, RetargetingConverter::toGdRetargetingConditionRule);
    }

    public static ModelChanges<RetargetingCondition> toRetargetingConditionModelChanges(
            GdUpdateRetargetingConditionItem retargetingCondition) {
        ModelChanges<RetargetingCondition> modelChanges =
                new ModelChanges<>(retargetingCondition.getRetargetingConditionId(),
                        RetargetingCondition.class);

        modelChanges.processNotNull(retargetingCondition.getName(), RetargetingCondition.NAME);
        modelChanges.processNotNull(retargetingCondition.getDescription(), RetargetingCondition.DESCRIPTION);
        modelChanges.processNotNull(toCoreRetargetingConditionRules(retargetingCondition.getConditionRules()),
                RetargetingCondition.RULES);

        return modelChanges;
    }
}
