package ru.yandex.direct.grid.processing.service.showcondition.converter;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowCondition;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionFilter;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionPrimaryStatus;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionStatusModerate;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionType;
import ru.yandex.direct.grid.core.entity.showcondition.service.GridShowConditionConstants;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.campaign.GdCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlatform;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyManual;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorFacade;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.showcondition.GdAuctionData;
import ru.yandex.direct.grid.processing.model.showcondition.GdKeyword;
import ru.yandex.direct.grid.processing.model.showcondition.GdPokazometerData;
import ru.yandex.direct.grid.processing.model.showcondition.GdRelevanceMatch;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowCondition;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionAccess;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionAction;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionAutobudgetPriority;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionFilter;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionOrderBy;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionOrderByField;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionPrimaryStatus;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionStatus;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionStatusModerate;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionType;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionWithTotals;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionsContainer;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionsContext;
import ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils;
import ru.yandex.direct.grid.processing.service.showcondition.container.ShowConditionsCacheFilterData;
import ru.yandex.direct.grid.processing.service.showcondition.container.ShowConditionsCacheRecordInfo;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.pokazometer.GroupRequest;
import ru.yandex.direct.pokazometer.PhraseRequest;
import ru.yandex.direct.utils.CollectionUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.AUTOTARGETING_PREFIX;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasNoAutotargetingPrefix;
import static ru.yandex.direct.grid.model.campaign.GdCampaignType.CONTENT_PROMOTION;
import static ru.yandex.direct.grid.model.campaign.GdCampaignType.CPM_BANNER;
import static ru.yandex.direct.grid.model.campaign.GdCampaignType.CPM_DEALS;
import static ru.yandex.direct.grid.model.campaign.GdCampaignType.MCBANNER;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignType;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignPlatform;
import static ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorHelper.STRATEGIES_EXTRACTORS_BY_TYPES;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionFeatureCalculator.FEATURE_CALCULATOR;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalGoalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.recalcTotalStatsForUnitedGrid;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

public class ShowConditionConverter {

    private static final Set<GdCampaignType> CAMPAIGN_TYPES_WITHOUT_TRAFARET_AUCTION =
            ImmutableSet.<GdCampaignType>builder()
                    .add(MCBANNER)
                    .add(CONTENT_PROMOTION)
                    .build();

    private static final GdStrategyExtractorFacade GD_STRATEGY_EXTRACTOR_FACADE =
            new GdStrategyExtractorFacade(STRATEGIES_EXTRACTORS_BY_TYPES);

    private static final BiMap<GdiShowConditionType, GdShowConditionType> TO_GD_SHOW_CONDITION_TYPE_MAP =
            ImmutableBiMap.<GdiShowConditionType, GdShowConditionType>builder()
                    .put(GdiShowConditionType.KEYWORD, GdShowConditionType.KEYWORD)
                    .put(GdiShowConditionType.RELEVANCE_MATCH, GdShowConditionType.RELEVANCE_MATCH)
                    .build();
    private static final BiMap<GdShowConditionType, GdiShowConditionType> GD_SHOW_CONDITION_TYPE_TO_INTERNAL_MAP =
            TO_GD_SHOW_CONDITION_TYPE_MAP.inverse();

    static GdShowConditionType toGdShowConditionType(GdiShowConditionType gdiShowCondition) {
        return TO_GD_SHOW_CONDITION_TYPE_MAP.get(gdiShowCondition);
    }

    @Nullable
    static GdiShowConditionType toInternalShowConditionType(GdShowConditionType gdShowConditionType) {
        return GD_SHOW_CONDITION_TYPE_TO_INTERNAL_MAP.get(gdShowConditionType);
    }

    /**
     * Преобразовывает в {@link Set} внутренних типов с фильтрацией null'ов (типы, которых нет в
     * {@link GdiShowConditionType})
     */
    @Nullable
    static Set<GdiShowConditionType> toInternalShowConditionTypes(
            @Nullable Set<GdShowConditionType> gdShowConditionTypeSet) {
        if (gdShowConditionTypeSet == null) {
            return null;
        }

        return StreamEx.of(gdShowConditionTypeSet)
                .map(ShowConditionConverter::toInternalShowConditionType)
                .nonNull()
                .toSet();
    }

    public static GdShowCondition toOuter(
            int index,
            GdiShowCondition condition,
            GdCampaignTruncated campaign,
            GdAdGroupTruncated group,
            GdAuctionData keywordTrafaretData,
            GdPokazometerData keywordPokazometerData,
            boolean cpcAndCpmOnOneGridEnabled
    ) {
        boolean hideTrafaretAndPokazometerData = cpcAndCpmOnOneGridEnabled &&
                CampaignServiceUtils.CPM_CAMPAIGN_TYPES.contains(toCampaignType(campaign.getType()));

        GdShowConditionStatus status = calcStatus(condition, campaign, group);
        GdShowConditionAccess access = calcAccess(condition, campaign, status);
        return getShowConditionImplementation(condition, campaign,
                hideTrafaretAndPokazometerData ? null : keywordTrafaretData,
                hideTrafaretAndPokazometerData ? null : keywordPokazometerData)
                .withIndex(index)
                .withId(condition.getId())
                .withAdGroupId(condition.getGroupId())
                .withAdGroup(group)
                .withCampaignId(condition.getCampaignId())
                .withType(toGdShowConditionType(condition.getType()))
                .withAccess(access)
                .withStatus(status)
                .withStats(StatHelper.internalStatsToOuter(
                        condition.getStat(), campaign.getType(), cpcAndCpmOnOneGridEnabled))
                .withGoalStats(ifNotNull(condition.getGoalStats(),
                        goalStats -> mapList(goalStats, StatHelper::internalGoalStatToOuter)));
    }

    private static GdShowConditionStatus calcStatus(GdiShowCondition internalCondition, GdCampaignTruncated campaign,
                                                    GdAdGroupTruncated group) {
        return new GdShowConditionStatus()
                .withReadOnly(campaign.getStatus().getReadOnly() || group.getStatus().getArchived())
                .withDeleted(internalCondition.getDeleted())
                .withSuspended(internalCondition.getSuspended())
                .withStatusModerate(toGdShowConditionStatusModerate(internalCondition))
                .withPrimaryStatus(calcPrimaryStatus(internalCondition));
    }

    private static GdShowConditionAccess calcAccess(GdiShowCondition condition, GdCampaignTruncated campaign,
                                                    GdShowConditionStatus status) {
        Set<GdShowConditionAction> actions = new HashSet<>();
        if (notEquals(campaign.getStatus().getArchived(), Boolean.TRUE)
                && campaign.getAccess().getActions().contains(GdCampaignAction.EDIT_CAMP)) {
            actions.add(GdShowConditionAction.DELETE);
            if (Objects.equals(condition.getSuspended(), Boolean.TRUE)) {
                actions.add(GdShowConditionAction.UNSUSPEND);
            } else {
                actions.add(GdShowConditionAction.SUSPEND);
            }
        }
        return new GdShowConditionAccess()
                .withCanEdit(!status.getReadOnly())
                .withActions(actions);
    }

    private static GdShowConditionStatusModerate toGdShowConditionStatusModerate(GdiShowCondition internalCondition) {
        GdiShowConditionStatusModerate internalStatusModerate =
                internalCondition.getType() == GdiShowConditionType.RELEVANCE_MATCH
                        ? GdiShowConditionStatusModerate.YES
                        : internalCondition.getStatusModerate();
        return GdShowConditionStatusModerate.fromSource(internalStatusModerate);
    }

    private static GdShowCondition getShowConditionImplementation(GdiShowCondition internalCondition,
                                                                  GdCampaignTruncated campaign,
                                                                  GdAuctionData auctionData,
                                                                  GdPokazometerData pokazometerData) {
        switch (internalCondition.getType()) {
            case KEYWORD:
                return getKeyword(internalCondition, campaign, auctionData, pokazometerData);
            case RELEVANCE_MATCH:
                return getRelevanceMatch(internalCondition, campaign);
            default:
                throw new IllegalArgumentException("Unknown showCondition type: " + internalCondition.getType());
        }
    }

    private static GdRelevanceMatch getRelevanceMatch(GdiShowCondition internalCondition,
                                                      GdCampaignTruncated campaign) {
        return new GdRelevanceMatch()
                .withPrice(calcPrice(internalCondition, campaign))
                .withPriceContext(calcPriceContext(internalCondition, campaign))
                .withAutobudgetPriority(calcAutoBudgetPriority(internalCondition, campaign));
    }

    private static GdShowConditionAutobudgetPriority calcAutoBudgetPriority(GdiShowCondition internalCondition,
                                                                            GdCampaignTruncated campaign) {
        if (!campaign.getFlatStrategy().getIsAutoBudget()) {
            return null;
        }
        return GdShowConditionAutobudgetPriority.fromSource(internalCondition.getAutobudgetPriority());
    }

    private static Boolean isPhraseHasTrafaretAuction(GdCampaignTruncated campaign,
                                                      @Nullable GdAuctionData auctionData) {
        return !campaign.getFlatStrategy().getIsAutoBudget() &&
                (campaign.getFlatStrategy().getPlatform() != null &&
                        !(campaign.getFlatStrategy().getPlatform() == GdCampaignPlatform.CONTEXT)) &&
                !(CAMPAIGN_TYPES_WITHOUT_TRAFARET_AUCTION.contains(campaign.getType())) &&
                auctionData != null &&
                auctionData.getAuctionDataItems() != null &&
                !auctionData.getAuctionDataItems().isEmpty();
    }

    private static GdKeyword getKeyword(GdiShowCondition internalCondition, GdCampaignTruncated campaign,
                                        GdAuctionData auctionData, GdPokazometerData pokazometerData) {
        GdKeyword keyword = new GdKeyword()
                .withKeyword(getKeywordFromGdi(internalCondition.getIsAutotargeting(), internalCondition.getKeyword()))
                .withMinusKeywords(nvl(internalCondition.getMinusKeywords(), Collections.emptyList()))
                .withHasPhraseTrafaretAuction(isPhraseHasTrafaretAuction(campaign,
                        auctionData)) //todo(pashkus) удалить, когда фронт перейдет на новое поле
                .withUsePhraseTrafaretAuction(isPhraseHasTrafaretAuction(campaign, auctionData))
                .withUseContextPricePrediction(hasCampaignContextPricesToShow(campaign))
                .withPrice(calcPrice(internalCondition, campaign))
                .withPriceContext(calcPriceContext(internalCondition, campaign))
                .withPokazometerData(pokazometerData)
                .withShowsForecast(internalCondition.getShowsForecast())
                .withAutobudgetPriority(calcAutoBudgetPriority(internalCondition, campaign))
                .withAuctionData(auctionData)
                .withWeightedCtr(BigDecimal.ZERO);

        addAuctionDataToShowCondition(keyword, campaign, auctionData);
        return keyword;
    }

    //Пока работаем только с фразами
    public static void addAuctionDataToShowCondition(GdShowCondition showCondition, GdCampaignTruncated campaign,
                                                     GdAuctionData auctionData) {
        checkArgument(showCondition instanceof GdKeyword,
                String.format("Only keywords are supported. ShowCondition with ID: %s not supported",
                        showCondition.getId()));
        ((GdKeyword) showCondition)
                .withUsePhraseTrafaretAuction(isPhraseHasTrafaretAuction(campaign, auctionData))
                .withAuctionData(auctionData)
                .withWeightedCtr(BigDecimal.ZERO);
    }

    public static String getKeywordFromGdi(Boolean isAutotargeting, String keyword) {
        return hasNoAutotargetingPrefix(isAutotargeting)
                ? keyword
                : AUTOTARGETING_PREFIX + keyword;
    }

    @Nullable
    private static BigDecimal calcPriceContext(GdiShowCondition internalCondition, GdCampaignTruncated campaign) {
        return !campaign.getStrategy().getIsAutoBudget() && hasCampaignContextPricesToShow(campaign) ?
                internalCondition.getPriceContext() : null;
    }

    @Nullable
    private static BigDecimal calcPrice(GdiShowCondition internalCondition, GdCampaignTruncated campaign) {
        return !campaign.getStrategy().getIsAutoBudget() ? internalCondition.getPrice() : null;
    }

    public static boolean hasContextPriceToShow(GdShowCondition showCondition) {
        return showCondition instanceof GdKeyword && ((GdKeyword) showCondition).getPriceContext() != null;
    }

    /**
     * Нужно ли показывать информацию о ставках в сети для этой кампании.
     *
     * @param campaign - кампания, для которой проверяем настройку стратегий
     */
    public static boolean hasCampaignContextPricesToShow(GdCampaignTruncated campaign) {

        if (hasOnlyPriceContext(campaign)) {
            return true;
        }

        boolean hasSeparateBiddingSupport =
                GD_STRATEGY_EXTRACTOR_FACADE.hasSeparateBiddingSupport(campaign.getFlatStrategy(), campaign.getType());

        return hasSeparateBiddingSupport &&
                (((GdCampaignStrategyManual) campaign.getFlatStrategy()).getSeparateBidding() ||
                        campaign.getFlatStrategy().getPlatform() == toGdCampaignPlatform(CampaignsPlatform.CONTEXT));
    }

    private static boolean hasOnlyPriceContext(GdCampaignTruncated campaign) {
        return campaign.getType() == CPM_BANNER || campaign.getType() == CPM_DEALS;
    }

    private static GdShowConditionPrimaryStatus calcPrimaryStatus(GdiShowCondition internalCondition) {
        if (internalCondition.getArchived()) {
            return GdShowConditionPrimaryStatus.ARCHIVED;
        } else if (internalCondition.getStatusModerate() == GdiShowConditionStatusModerate.NO) {
            return GdShowConditionPrimaryStatus.MODERATION_REJECTED;
        } else if (internalCondition.getStatusModerate() == GdiShowConditionStatusModerate.NEW) {
            return GdShowConditionPrimaryStatus.DRAFT;
        } else if (internalCondition.getSuspended()) {
            return GdShowConditionPrimaryStatus.STOPPED;
        }
        return GdShowConditionPrimaryStatus.ACTIVE;
    }

    public static GdiShowConditionFilter toInternalFilter(GdShowConditionFilter filter) {
        //TODO-buhter: выпилить после полного перехода на новый фильтр(DIRECT-99668)
        Set<GdiShowConditionPrimaryStatus> showConditionPrimaryStatusSet;
        if (filter.getShowConditionStatusIn() != null) {
            showConditionPrimaryStatusSet = mapSet(filter.getShowConditionStatusIn(),
                    GdShowConditionPrimaryStatus::toSource);
        } else if (filter.getStatusIn() != null) {
            showConditionPrimaryStatusSet = mapSet(filter.getStatusIn(), gdShowConditionBaseStatus -> {
                switch (gdShowConditionBaseStatus) {
                    case ACTIVE:
                        return GdiShowConditionPrimaryStatus.ACTIVE;
                    case SUSPENDED:
                        return GdiShowConditionPrimaryStatus.STOPPED;
                    case DRAFT:
                        return GdiShowConditionPrimaryStatus.DRAFT;
                    default:
                        throw new IllegalArgumentException("Unknown GdShowConditionBaseStatus: " + gdShowConditionBaseStatus.name());
                }
            });
        } else {
            showConditionPrimaryStatusSet = emptySet();
        }

        return new GdiShowConditionFilter()
                .withShowConditionIdIn((filter.getShowConditionIdIn()))
                .withShowConditionIdNotIn(filter.getShowConditionIdNotIn())
                .withCampaignIdIn(filter.getCampaignIdIn())
                .withAdGroupIdIn(filter.getAdGroupIdIn())

                .withKeywordIn(filter.getKeywordIn())
                .withKeywordNotIn(filter.getKeywordNotIn())
                .withKeywordContains(filter.getKeywordContains())
                .withKeywordNotContains(filter.getKeywordNotContains())
                .withKeywordWithoutMinusWordsContains(filter.getKeywordWithoutMinusWordsContains())
                .withMinusWordsContains(filter.getMinusWordsContains())

                .withMinPrice(filter.getMinPrice())
                .withMaxPrice(filter.getMaxPrice())
                .withMinPriceContext(filter.getMinPriceContext())
                .withMaxPriceContext(filter.getMaxPriceContext())

                .withAutobudgetPriorityIn(
                        mapSet(filter.getAutobudgetPriorityIn(), GdShowConditionAutobudgetPriority::toSource))
                .withShowConditionStatusIn(showConditionPrimaryStatusSet)
                .withTypeIn(toInternalShowConditionTypes(filter.getTypeIn()))
                .withStats(StatHelper.toInternalStatsFilter(filter.getStats()))
                .withGoalStats(mapList(filter.getGoalStats(), StatHelper::toInternalGoalStatsFilter))
                .withReasonsContainSome(filter.getReasonsContainSome());
    }

    public static ShowConditionsCacheRecordInfo toShowConditionsCacheRecordInfo(long clientId,
                                                                                GdShowConditionsContainer input) {
        return new ShowConditionsCacheRecordInfo(clientId, input.getCacheKey(),
                new ShowConditionsCacheFilterData()
                        .withFilter(input.getFilter())
                        .withOrderBy(input.getOrderBy())
                        .withStatRequirements(input.getStatRequirements()));
    }

    public static GdShowConditionsContext toGdShowConditionsContext(GdShowConditionWithTotals showConditionWithTotals,
                                                                    GdShowConditionFilter inputFilter,
                                                                    boolean cpcAndCpmOnOneGridEnabled) {
        var rowsetFull = showConditionWithTotals.getGdShowConditions();
        var totalStats = nvl(showConditionWithTotals.getTotalStats(),
                calcTotalStats(mapList(rowsetFull, GdShowCondition::getStats)));

        if (cpcAndCpmOnOneGridEnabled) {
            Map<GdCampaignType, List<GdEntityStats>> campaignTypeToStats = StreamEx.of(rowsetFull)
                    .mapToEntry(gd -> gd.getAdGroup().getCampaign().getType(), GdShowCondition::getStats)
                    .grouping();
            recalcTotalStatsForUnitedGrid(totalStats, campaignTypeToStats);
        }

        // показываем предупреждение если есть тоталы из БД и был фильтр по коду (либо мог быть)
        var totalStatsWithoutFiltersWarn = showConditionWithTotals.getTotalStats() != null
                && (rowsetFull.size() < GridShowConditionConstants.getMaxConditionRows() || hasAnyCodeFilter(inputFilter));
        return new GdShowConditionsContext()
                .withTotalCount(rowsetFull.size())
                .withHasObjectsOverLimit(rowsetFull.size() >= GridShowConditionConstants.getMaxConditionRows())
                .withShowConditionIds(listToSet(rowsetFull, GdShowCondition::getId))
                .withFeatures(FEATURE_CALCULATOR.apply(rowsetFull))
                .withTotalStats(totalStats)
                .withTotalStatsWithoutFiltersWarn(totalStatsWithoutFiltersWarn)
                .withTotalGoalStats(calcTotalGoalStats(totalStats, mapList(rowsetFull, GdShowCondition::getGoalStats)))
                .withFilter(inputFilter);
    }

    /**
     * Присутствуют ли в фильтре поля, которые фильтруют условия в коде, а не в запросе к БД
     */
    private static boolean hasAnyCodeFilter(GdShowConditionFilter inputFilter) {
        return inputFilter.getMinPriceContext() != null
                || !CollectionUtils.isEmpty(inputFilter.getStatusIn())
                || !CollectionUtils.isEmpty(inputFilter.getShowConditionStatusIn())
                || !CollectionUtils.isEmpty(inputFilter.getReasonsContainSome());
    }

    public static GdiShowConditionOrderBy toInternalOrderBy(GdShowConditionOrderBy orderBy) {
        return new GdiShowConditionOrderBy()
                .withOrder(orderBy.getOrder())
                .withField(GdShowConditionOrderByField.toSource(orderBy.getField()))
                .withGoalId(orderBy.getParams() == null ? null : orderBy.getParams().getGoalId());
    }

    public static GroupRequest toPokazometerGroupRequest(ImmutableList<Integer> regionIds,
                                                         List<GdiShowCondition> showConditions) {
        List<Long> geo = mapList(regionIds, Integer::longValue);
        List<PhraseRequest> phraseRequests = mapList(showConditions, ShowConditionConverter::toPhraseRequest);
        return new GroupRequest(phraseRequests, geo);
    }

    private static PhraseRequest toPhraseRequest(GdiShowCondition showCondition) {
        return new PhraseRequest(showCondition.getKeyword(),
                showCondition.getPriceContext().longValue(), showCondition.getId());
    }

}
