package ru.yandex.direct.grid.processing.service.showcondition.keywords;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.converter.KeywordUpdateConverter;
import ru.yandex.direct.core.entity.keyword.container.UpdatedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.FindAndReplaceKeyword;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.keyword.service.KeywordsUpdateOperation;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridKeywordsParser;
import ru.yandex.direct.grid.model.findandreplace.ReplaceRule;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsOperators;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdChangeKeywordsCase;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdChangeKeywordsCaseField;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordField;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPreviewPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywordsPreviewPayloadItem;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywordsPayloadItem;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.showcondition.container.FindAndReplaceKeywordsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordCaseConverter;
import ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordOperatorsConverter;
import ru.yandex.direct.grid.processing.service.showcondition.validation.ShowConditionValidationService;
import ru.yandex.direct.grid.processing.util.findandreplace.ReplaceRuleHelper;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.FindAndReplaceDataConverter.convertChangedMinusKeywordsToList;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.FindAndReplaceDataConverter.getCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.FindAndReplaceDataConverter.getEmptyPayload;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.FindAndReplaceDataConverter.getEmptyPreviewPayload;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.FindAndReplaceDataConverter.toCoreKeywords;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordsResponseConverter.convertMinuses;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordsResponseConverter.getAffectedKeywords;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordsResponseConverter.getSuccessfullyUpdatedKeywords;
import static ru.yandex.direct.operation.tree.ItemSubOperationExecutor.extractSubList;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class FindAndReplaceDataService {

    private static final String MINUS_KEYWORD_DELIMITER = " ";

    private final KeywordService keywordService;
    private final KeywordOperationFactory keywordOperationFactory;
    private final GridKeywordsParser gridKeywordsParser;
    private final ShowConditionValidationService showConditionValidationService;
    private final GridCacheService gridCacheService;

    @Autowired
    public FindAndReplaceDataService(KeywordService keywordService,
                                     KeywordOperationFactory keywordOperationFactory,
                                     GridKeywordsParser gridKeywordsParser,
                                     ShowConditionValidationService showConditionValidationService,
                                     GridCacheService gridCacheService) {
        this.keywordService = keywordService;
        this.keywordOperationFactory = keywordOperationFactory;
        this.gridKeywordsParser = gridKeywordsParser;
        this.showConditionValidationService = showConditionValidationService;
        this.gridCacheService = gridCacheService;
    }

    /**
     * Получить превью для Поиска и замены фраз
     */
    public GdFindAndReplaceKeywordsPreviewPayload getPreview(GdFindAndReplaceKeywords input, Long operatorUid,
                                                             UidAndClientId uidAndClientId) {
        showConditionValidationService.validateFindAndReplaceKeywordsRequest(input);

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        FindAndReplaceKeywordsCacheRecordInfo recordInfo = getCacheRecordInfo(uidAndClientId.getClientId(), input);
        Optional<GdFindAndReplaceKeywordsPreviewPayload> res = gridCacheService.getFromCache(recordInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        List<GdFindAndReplaceKeywordsPreviewPayloadItem> rowsetFull =
                getFindAndReplaceKeywords(input, uidAndClientId.getClientId());
        if (rowsetFull.isEmpty()) {
            return getEmptyPreviewPayload();
        }

        Map<Integer, Integer> updateIndexMap = new HashMap<>();
        List<Keyword> keywords = toCoreKeywords(input.getKeywordIds(), rowsetFull);
        KeywordsUpdateOperation keywordsUpdateOperation =
                getKeywordsUpdateOperation(keywords, operatorUid, uidAndClientId, updateIndexMap);
        MassResult<UpdatedKeywordInfo> result = keywordsUpdateOperation.prepare()
                .orElseGet(keywordsUpdateOperation::cancel);

        GdValidationResult validationResult =
                getGdValidationResult(keywords, updateIndexMap, result.getValidationResult());
        GdFindAndReplaceKeywordsPreviewPayload payload = new GdFindAndReplaceKeywordsPreviewPayload()
                .withTotalCount(rowsetFull.size())
                .withKeywordIds(mapList(rowsetFull, GdFindAndReplaceKeywordsPreviewPayloadItem::getId))
                .withValidationResult(validationResult);

        return gridCacheService.getResultAndSaveToCache(recordInfo, payload, rowsetFull, range);
    }

    /**
     * Поиск и замена фраз
     */
    public GdFindAndReplaceKeywordsPayload findAndReplaceKeywords(GdFindAndReplaceKeywords input, Long operatorUid,
                                                                  UidAndClientId uidAndClientId) {
        showConditionValidationService.validateFindAndReplaceKeywordsRequest(input);

        List<GdFindAndReplaceKeywordsPreviewPayloadItem> rowsetFull =
                getFindAndReplaceKeywords(input, uidAndClientId.getClientId());
        if (rowsetFull.isEmpty()) {
            return getEmptyPayload();
        }

        Map<Integer, Integer> updateIndexMap = new HashMap<>();
        List<Keyword> keywords = toCoreKeywords(input.getKeywordIds(), rowsetFull);
        KeywordsUpdateOperation keywordsUpdateOperation =
                getKeywordsUpdateOperation(keywords, operatorUid, uidAndClientId, updateIndexMap);
        MassResult<UpdatedKeywordInfo> result = keywordsUpdateOperation.prepareAndApply();

        List<GdUpdateKeywordsPayloadItem> successUpdatedKeywords =
                getSuccessfullyUpdatedKeywords(gridKeywordsParser, result);
        return new GdFindAndReplaceKeywordsPayload()
                .withUpdatedTotalCount(successUpdatedKeywords.size())
                .withUpdatedKeywordIds(mapList(successUpdatedKeywords, GdUpdateKeywordsPayloadItem::getId))
                .withRowset(successUpdatedKeywords)
                .withValidationResult(getGdValidationResult(keywords, updateIndexMap, result.getValidationResult()))
                .withAffectedKeywords(getAffectedKeywords(gridKeywordsParser, keywordsUpdateOperation))
                //TODO пока не поддерживаем кеширование. Добавим, когда фронт начнет использовать данные
                .withCacheKey("");
    }

    /**
     * Изменение регистра фраз
     */
    public GdFindAndReplaceKeywordsPayload changeKeywordsCase(@GraphQLNonNull GdChangeKeywordsCase input,
                                                              Long operatorUid,
                                                              UidAndClientId uidAndClientId) {
        showConditionValidationService.validateChangeKeywordsCaseRequest(input);

        List<GdFindAndReplaceKeywordsPreviewPayloadItem> rowsetFull =
                getChangeKeywordsCaseItems(input, uidAndClientId.getClientId());

        if (rowsetFull.isEmpty()) {
            return getEmptyPayload();
        }

        Map<Integer, Integer> updateIndexMap = new HashMap<>();
        List<Keyword> keywords = toCoreKeywords(input.getKeywordIds(), rowsetFull);
        KeywordsUpdateOperation keywordsUpdateOperation =
                getKeywordsUpdateOperation(keywords, operatorUid, uidAndClientId, updateIndexMap);
        MassResult<UpdatedKeywordInfo> result = keywordsUpdateOperation.prepareAndApply();

        List<GdUpdateKeywordsPayloadItem> successUpdatedKeywords =
                getSuccessfullyUpdatedKeywords(gridKeywordsParser, result);

        return new GdFindAndReplaceKeywordsPayload()
                .withUpdatedTotalCount(successUpdatedKeywords.size())
                .withUpdatedKeywordIds(mapList(successUpdatedKeywords, GdUpdateKeywordsPayloadItem::getId))
                .withRowset(successUpdatedKeywords)
                .withValidationResult(getGdValidationResult(keywords, updateIndexMap, result.getValidationResult()))
                .withAffectedKeywords(getAffectedKeywords(gridKeywordsParser, keywordsUpdateOperation));
    }

    private KeywordsUpdateOperation getKeywordsUpdateOperation(List<Keyword> keywords, Long operatorUid,
                                                               UidAndClientId uidAndClientId,
                                                               Map<Integer, Integer> updateIndexMap) {
        List<ModelChanges<Keyword>> keywordChangesToUpdate = extractSubList(updateIndexMap, keywords,
                k -> k.getPhrase() != null, KeywordUpdateConverter::keywordToModelChanges);

        return keywordOperationFactory.createKeywordsUpdateOperation(
                Applicability.PARTIAL, keywordChangesToUpdate,
                operatorUid, uidAndClientId.getClientId(), uidAndClientId.getUid(), true, false);
    }

    /**
     * Возвращает результат валидации с соответствием индексов запроса
     */
    @Nullable
    private GdValidationResult getGdValidationResult(List<Keyword> keywords, Map<Integer, Integer> updateIndexMap,
                                                     @Nullable ValidationResult<?, Defect> validationResult) {
        if (validationResult == null) {
            return null;
        }

        ValidationResult<List<Keyword>, Defect> destValidationResult = new ValidationResult<>(keywords);
        //noinspection unchecked
        mergeSubListValidationResults(destValidationResult,
                (ValidationResult<List<Keyword>, Defect>) validationResult, updateIndexMap);

        return showConditionValidationService.getValidationResultForRequestWithKeywordIds(destValidationResult);
    }

    private List<GdFindAndReplaceKeywordsPreviewPayloadItem> getFindAndReplaceKeywords(GdFindAndReplaceKeywords input,
                                                                                       ClientId clientId) {
        List<FindAndReplaceKeyword> keywords = keywordService.getFindAndReplaceKeyword(clientId, input.getKeywordIds());

        final ReplaceRule replaceRule = ReplaceRuleHelper.getReplaceRule(input);
        final boolean needReplaceKeywords = input.getFields().contains(GdFindAndReplaceKeywordField.KEYWORD);
        final boolean needReplaceMinusKeywords =
                input.getFields().contains(GdFindAndReplaceKeywordField.MINUS_KEYWORDS);

        List<GdFindAndReplaceKeywordsPreviewPayloadItem> previewItems = new ArrayList<>();
        keywords.forEach(keyword -> {
            KeywordWithMinuses keywordWithMinuses = gridKeywordsParser.parseKeyword(keyword.getPhrase());
            List<String> minusKeywords = convertMinuses(keywordWithMinuses.getMinusKeywords());
            String changedKeyword = null;
            String changedMinusKeywords = null;

            if (needReplaceKeywords) {
                changedKeyword = replaceRule.apply(keywordWithMinuses.getKeyword().toString());
            }

            if (needReplaceMinusKeywords) {
                String minusKeywordsAsString = String.join(MINUS_KEYWORD_DELIMITER, minusKeywords);
                changedMinusKeywords = replaceRule.apply(minusKeywordsAsString);
            }

            if (changedKeyword != null || changedMinusKeywords != null) {
                previewItems.add(new GdFindAndReplaceKeywordsPreviewPayloadItem()
                        .withId(keyword.getId())
                        .withKeyword(keywordWithMinuses.getKeyword().toString())
                        .withMinusKeywords(minusKeywords)
                        .withChangedKeyword(changedKeyword)
                        .withChangedMinusKeywords(convertChangedMinusKeywordsToList(changedMinusKeywords))
                );
            }
        });

        return previewItems;
    }

    private List<GdFindAndReplaceKeywordsPreviewPayloadItem> getChangeKeywordsCaseItems(@GraphQLNonNull GdChangeKeywordsCase input,
                                                                                        ClientId clientId) {
        List<FindAndReplaceKeyword> keywords = keywordService.getFindAndReplaceKeyword(clientId, input.getKeywordIds());

        KeywordCaseConverter keywordCaseConverter = new KeywordCaseConverter();

        final boolean needReplaceKeywords = input.getFields().contains(GdChangeKeywordsCaseField.KEYWORD);
        final boolean needReplaceMinusKeywords =
                input.getFields().contains(GdChangeKeywordsCaseField.MINUS_KEYWORDS);

        List<GdFindAndReplaceKeywordsPreviewPayloadItem> previewItems = new ArrayList<>();
        keywords.forEach(keyword -> {
            KeywordWithMinuses keywordWithMinuses = gridKeywordsParser.parseKeyword(keyword.getPhrase());
            KeywordWithMinuses newKeywordWithMinuses = keywordCaseConverter.changeCase(keywordWithMinuses,
                    input.getCaseMode(), needReplaceKeywords, needReplaceMinusKeywords);

            String oldKeyword = keywordWithMinuses.getKeyword().toString();
            String newKeyword = newKeywordWithMinuses.getKeyword().toString();
            List<String> oldMinusKeywords = convertMinuses(keywordWithMinuses.getMinusKeywords());
            List<String> newMinusKeywords = convertMinuses(newKeywordWithMinuses.getMinusKeywords());

            if (!keywordWithMinuses.equals(newKeywordWithMinuses)) {
                previewItems.add(new GdFindAndReplaceKeywordsPreviewPayloadItem()
                        .withId(keyword.getId())
                        .withKeyword(oldKeyword)
                        .withMinusKeywords(oldMinusKeywords)
                        .withChangedKeyword(newKeyword)
                        .withChangedMinusKeywords(newMinusKeywords)
                );
            }
        });

        return previewItems;
    }

    /**
     * добавление операторов
     */
    public GdFindAndReplaceKeywordsPayload addKeywordsOperators(@GraphQLNonNull GdAddKeywordsOperators input,
                                                                Long operatorUid,
                                                                UidAndClientId uidAndClientId) {
        showConditionValidationService.validateAddKeywordsOperators(input);
        List<GdFindAndReplaceKeywordsPreviewPayloadItem> rowsetFull =
                getAddKeywordsOperatorsItems(input, uidAndClientId.getClientId());
        if (rowsetFull.isEmpty()) {
            return getEmptyPayload();
        }
        Map<Integer, Integer> updateIndexMap = new HashMap<>();
        List<Keyword> keywords = toCoreKeywords(input.getKeywordIds(), rowsetFull);
        KeywordsUpdateOperation keywordsUpdateOperation =
                getKeywordsUpdateOperation(keywords, operatorUid, uidAndClientId, updateIndexMap);
        MassResult<UpdatedKeywordInfo> result = keywordsUpdateOperation.prepareAndApply();
        List<GdUpdateKeywordsPayloadItem> successUpdatedKeywords =
                getSuccessfullyUpdatedKeywords(gridKeywordsParser, result);
        return new GdFindAndReplaceKeywordsPayload()
                .withUpdatedTotalCount(successUpdatedKeywords.size())
                .withUpdatedKeywordIds(mapList(successUpdatedKeywords, GdUpdateKeywordsPayloadItem::getId))
                .withRowset(successUpdatedKeywords)
                .withValidationResult(getGdValidationResult(keywords, updateIndexMap, result.getValidationResult()))
                .withAffectedKeywords(getAffectedKeywords(gridKeywordsParser, keywordsUpdateOperation));
    }

    private List<GdFindAndReplaceKeywordsPreviewPayloadItem> getAddKeywordsOperatorsItems(@GraphQLNonNull GdAddKeywordsOperators input,
                                                                                          ClientId clientId) {
        List<FindAndReplaceKeyword> keywords = keywordService.getFindAndReplaceKeyword(clientId, input.getKeywordIds());
        KeywordOperatorsConverter operatorsConverter = new KeywordOperatorsConverter();
        List<GdFindAndReplaceKeywordsPreviewPayloadItem> previewItems = new ArrayList<>();
        keywords.forEach(keyword -> {
            KeywordWithMinuses keywordWithMinuses = gridKeywordsParser.parseKeyword(keyword.getPhrase());
            var changedKeyword = operatorsConverter.addOperators(keywordWithMinuses.getKeyword(),
                    input.getOperatorsMode());
            if (changedKeyword != null) {
                previewItems.add(new GdFindAndReplaceKeywordsPreviewPayloadItem()
                        .withId(keyword.getId())
                        .withKeyword(keywordWithMinuses.getKeyword().toString())
                        .withMinusKeywords(convertMinuses(keywordWithMinuses.getMinusKeywords()))
                        .withChangedKeyword(changedKeyword.toString())
                        .withChangedMinusKeywords(null)
                );
            }
        });
        return previewItems;
    }
}
