package ru.yandex.direct.grid.processing.service.showcondition.keywords;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.container.CampaignNewMinusKeywords;
import ru.yandex.direct.core.entity.campaign.service.AppendCampaignMinusKeywordsOperation;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.UpdatedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordDeleteOperation;
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.keyword.service.KeywordsAddOperation;
import ru.yandex.direct.core.entity.keyword.service.KeywordsUpdateOperation;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchDeleteOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchUpdateOperation;
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider;
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridKeywordsParser;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.showcondition.GdValidateKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.GdValidateKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdDeleteKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdDeleteKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdMoveToMinusKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdMoveToMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSuspendKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSuspendKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUnsuspendKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUnsuspendKeywordsPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywordsItem;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywordsPayload;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.grid.processing.service.showcondition.converter.AddKeywordsRequestConverter;
import ru.yandex.direct.grid.processing.service.showcondition.validation.ShowConditionValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.ConvertResultOperation;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.ResultConverters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator.keywordSyntaxValidator;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordsResponseConverter.getAffectedKeywords;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordsResponseConverter.getSuccessfullyAddedKeywords;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordsResponseConverter.getSuccessfullyUpdatedKeywords;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.getMassResultWithAggregatedValidationResult;
import static ru.yandex.direct.grid.processing.util.GdValidationResultUtils.concatValidationResults;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getSuccessfullyUpdatedIds;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.converter.Converters.nullSafeConverter;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class KeywordsDataService {

    private final KeywordOperationFactory keywordOperationFactory;
    private final AddKeywordsRequestConverter addKeywordsRequestConverter;
    private final GridValidationService gridValidationService;
    private final RelevanceMatchService relevanceMatchService;
    private final ShowConditionValidationService showConditionValidationService;
    private final GridKeywordsParser gridKeywordsParser;
    private final ClientService clientService;
    private final GridContextProvider gridContextProvider;
    private final KeywordService keywordService;
    private final CampaignService campaignService;
    private final GrutUacCampaignService grutUacCampaignService;
    private final GrutTransactionProvider grutTransactionProvider;
    private final int grutRetries;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public KeywordsDataService(KeywordOperationFactory keywordOperationFactory,
                               AddKeywordsRequestConverter addKeywordsRequestConverter,
                               GridValidationService gridValidationService,
                               RelevanceMatchService relevanceMatchService,
                               ShowConditionValidationService showConditionValidationService,
                               GridKeywordsParser gridKeywordsParser,
                               ClientService clientService,
                               GridContextProvider gridContextProvider,
                               KeywordService keywordService,
                               CampaignService campaignService,
                               GrutUacCampaignService grutUacCampaignService,
                               GrutTransactionProvider grutTransactionProvider,
                               @Value("${object_api.retries}") int grutRetries) {
        this.keywordOperationFactory = keywordOperationFactory;
        this.addKeywordsRequestConverter = addKeywordsRequestConverter;
        this.gridValidationService = gridValidationService;
        this.relevanceMatchService = relevanceMatchService;
        this.showConditionValidationService = showConditionValidationService;
        this.gridKeywordsParser = gridKeywordsParser;
        this.clientService = clientService;
        this.gridContextProvider = gridContextProvider;
        this.keywordService = keywordService;
        this.campaignService = campaignService;
        this.grutUacCampaignService = grutUacCampaignService;
        this.grutTransactionProvider = grutTransactionProvider;
        this.grutRetries = grutRetries;
    }

    public GdValidateKeywordsPayload validateKeywords(GdValidateKeywords input) {
        ItemValidationBuilder<GdValidateKeywords, Defect> ivb = ItemValidationBuilder.of(input);

        var validationResult = ivb.list(input.getKeywords(), GdValidateKeywords.KEYWORDS.name())
                .checkEachBy(keywordSyntaxValidator())
                .getResult();

        GdValidationResult gdValidationResult =
                gridValidationService.toGdValidationResult(validationResult, path(field(GdValidateKeywords.KEYWORDS)));

        return new GdValidateKeywordsPayload()
                .withValidationResult(gdValidationResult);
    }

    /**
     * Добавить ключевые фразы
     *
     * @param request        запрос со списком параметров ключевых фраз, которые нужно добавить
     * @param operatorUid    uid оператора, производящего действие
     * @param uidAndClientId uid и id клиента, над которым производится действие
     */
    public GdAddKeywordsPayload addKeywords(GdAddKeywords request, Long operatorUid, UidAndClientId uidAndClientId) {
        if (isEmpty(request.getAddItems())) {
            return new GdAddKeywordsPayload()
                    .withAddedItems(emptyList())
                    .withValidationResult(null);
        }
        showConditionValidationService.validateAddKeywordsRequest(request);

        List<Keyword> keywordsToAdd = addKeywordsRequestConverter.convertRequest(request, uidAndClientId.getClientId());
        KeywordsAddOperation addOperation = getKeywordsAddOperation(keywordsToAdd, operatorUid, uidAndClientId);
        MassResult<AddedKeywordInfo> result = addOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdAddKeywords.ADD_ITEMS)));
        return new GdAddKeywordsPayload()
                .withAddedItems(getSuccessfullyAddedKeywords(result))
                .withValidationResult(validationResult);
    }

    private KeywordsAddOperation getKeywordsAddOperation(List<Keyword> keywordsToAdd, Long operatorUid,
                                                         UidAndClientId uidAndClientId) {
        return keywordOperationFactory
                .createKeywordsAddOperation(Applicability.PARTIAL, keywordsToAdd, operatorUid,
                        uidAndClientId.getClientId(), uidAndClientId.getUid());
    }

    /**
     * Удалить ключевые фразы/автотатргетинги
     *
     * @param request запрос со списком идентификаторов ключевых фраз/автотатргетингов, которые нужно удалить
     */
    public GdDeleteKeywordsPayload deleteKeywords(GdDeleteKeywords request) {
        if (isEmpty(request.getKeywordIds())) {
            return new GdDeleteKeywordsPayload()
                    .withDeletedKeywordIds(emptyList())
                    .withValidationResult(null);
        }
        showConditionValidationService.validateRequestWithKeywordIds(request);

        User subjectUser = gridContextProvider.getGridContext().getSubjectUser();
        User operator = gridContextProvider.getGridContext().getOperator();

        List<Long> relevanceMatchIds = getRelevanceMatchIds(request.getKeywordIds(), subjectUser.getClientId());
        List<Long> showConditionIds = request.getKeywordIds();

        OperationCreator<Long, Operation<Long>> deleteKeywordCoreOperationCreator =
                inputItems -> createKeywordDeleteOperation(operator, subjectUser, inputItems);
        OperationCreator<Long, Operation<Long>> deleteRelevanceMatchCoreOperationCreator =
                inputItems -> createRelevanceMatchDeleteOperation(operator, subjectUser, inputItems);

        SplitAndMergeOperationAggregator<Long, Long> deleteOperation =
                SplitAndMergeOperationAggregator.builderForPartialOperations()
                        .addSubOperation(
                                Predicate.not(relevanceMatchIds::contains),
                                deleteKeywordCoreOperationCreator)
                        .addSubOperation(
                                relevanceMatchIds::contains,
                                deleteRelevanceMatchCoreOperationCreator)
                        .build();

        MassResult<Long> aggregateOperationResult = deleteOperation.execute(showConditionIds);
        aggregateOperationResult = getMassResultWithAggregatedValidationResult(aggregateOperationResult, showConditionIds);

        GdValidationResult validationResult =
                showConditionValidationService.getValidationResultForRequestWithKeywordIds(aggregateOperationResult);
        return new GdDeleteKeywordsPayload()
                .withDeletedKeywordIds(getSuccessfullyUpdatedIds(aggregateOperationResult, identity()))
                .withValidationResult(validationResult);
    }

    /**
     * Перенос фраз в минус-фразы на кампанию.
     * Происходит в два этапа: удаление фраз и вставка минус-фраз на кампанию.
     *
     * @param request запрос со списком идентификаторов ключевых фраз
     */
    public GdMoveToMinusKeywordsPayload moveToMinusKeywords(GdMoveToMinusKeywords request) {
        if (isEmpty(request.getKeywordIds())) {
            return new GdMoveToMinusKeywordsPayload()
                    .withMovedKeywordIds(emptyList())
                    .withValidationResult(null);
        }
        showConditionValidationService.validateRequestWithKeywordIds(request);

        User subjectUser = gridContextProvider.getGridContext().getSubjectUser();
        User operator = gridContextProvider.getGridContext().getOperator();

        List<Keyword> keywordsToMove =
                keywordService.getKeywords(subjectUser.getClientId(), request.getKeywordIds());
        GdDeleteKeywordsPayload deleteKeywordsPayload =
                deleteKeywords(new GdDeleteKeywords().withKeywordIds(request.getKeywordIds()));
        if (deleteKeywordsPayload.getDeletedKeywordIds().isEmpty()) {
            return new GdMoveToMinusKeywordsPayload()
                    .withMovedKeywordIds(emptyList())
                    .withValidationResult(deleteKeywordsPayload.getValidationResult());
        }

        Set<Long> deletedKeywordIds = new HashSet<>(deleteKeywordsPayload.getDeletedKeywordIds());
        Map<Long, List<String>> phrasesByCampaignsId = StreamEx.of(keywordsToMove)
                .filter(keyword -> deletedKeywordIds.contains(keyword.getId()))
                .groupingBy(Keyword::getCampaignId, Collectors.mapping(Keyword::getPhrase, Collectors.toList()));
        grutTransactionProvider.runInRetryableTransaction(grutRetries, null, () -> {
            grutUacCampaignService.deleteCampaignKeywordsAndAppendMinusKeywords(
                    phrasesByCampaignsId, phrasesByCampaignsId);
            return null;
        });

        List<CampaignNewMinusKeywords> campaignNewMinusKeywords =
                EntryStream.of(phrasesByCampaignsId).mapKeyValue(CampaignNewMinusKeywords::new).toList();

        AppendCampaignMinusKeywordsOperation appendMinusKeywordsOperation =
                campaignService.createAppendMinusKeywordsOperation(campaignNewMinusKeywords, operator.getUid(),
                        subjectUser.getClientId());
        MassResult<Long> appendMinusKeywordsResult = appendMinusKeywordsOperation.prepareAndApply();

        GdValidationResult appendMinusKeywordsVr =
                showConditionValidationService.getValidationResultForRequestWithKeywordIds(appendMinusKeywordsResult);
        return new GdMoveToMinusKeywordsPayload()
                .withMovedKeywordIds(deleteKeywordsPayload.getDeletedKeywordIds())
                .withValidationResult(concatValidationResults(appendMinusKeywordsVr,
                        deleteKeywordsPayload.getValidationResult()));
    }

    /**
     * Остановить показы по ключевым фразам//автотатргетингам
     *
     * @param request запрос со списком идентификаторов ключевых фраз/автотатргетингов, которые нужно остановить
     */
    public GdSuspendKeywordsPayload suspendKeywords(GdSuspendKeywords request) {
        showConditionValidationService.validateRequestWithKeywordIds(request);

        User subjectUser = gridContextProvider.getGridContext().getSubjectUser();
        User operator = gridContextProvider.getGridContext().getOperator();

        MassResult<Long> aggregateOperationResult = executeSuspendOperation(subjectUser, operator, request.getKeywordIds(), true);

        GdValidationResult validationResult =
                showConditionValidationService.getValidationResultForRequestWithKeywordIds(aggregateOperationResult);
        return new GdSuspendKeywordsPayload()
                .withSuspendedKeywordIds(getSuccessfullyUpdatedIds(aggregateOperationResult, identity()))
                .withValidationResult(validationResult);
    }

    /**
     * Возобновить показы по ключевым фразам/автотатргетингам
     *
     * @param request запрос со списком идентификаторов ключевых фраз/автотатргетингов, которые нужно возобновить
     */
    public GdUnsuspendKeywordsPayload unSuspendKeywords(GdUnsuspendKeywords request) {
        showConditionValidationService.validateRequestWithKeywordIds(request);

        User subjectUser = gridContextProvider.getGridContext().getSubjectUser();
        User operator = gridContextProvider.getGridContext().getOperator();

        MassResult<Long> aggregateOperationResult = executeSuspendOperation(subjectUser, operator, request.getKeywordIds(), false);

        GdValidationResult validationResult =
                showConditionValidationService.getValidationResultForRequestWithKeywordIds(aggregateOperationResult);
        return new GdUnsuspendKeywordsPayload()
                .withUnsuspendedKeywordIds(getSuccessfullyUpdatedIds(aggregateOperationResult, identity()))
                .withValidationResult(validationResult);
    }

    /**
     * Обновить параметры ключевых фраз
     *
     * @param request        запрос со списком идентификаторов ключевых фраз, которые нужно обновить
     * @param operatorUid    uid оператора, производящего действие
     * @param uidAndClientId uid и id клиента, над которым производится действие
     */
    public GdUpdateKeywordsPayload updateKeywords(
            GdUpdateKeywords request,
            Long operatorUid,
            UidAndClientId uidAndClientId) {
        if (isEmpty(request.getKeywordUpdateItems())) {
            return new GdUpdateKeywordsPayload()
                    .withUpdatedKeywords(emptyList())
                    .withAffectedKeywords(emptyList())
                    .withValidationResult(null);
        }
        showConditionValidationService.validateUpdateKeywordsRequest(request);

        List<ModelChanges<Keyword>> updateList = getKeywordUpdateChanges(request.getKeywordUpdateItems());

        KeywordsUpdateOperation keywordsUpdateOperation = keywordOperationFactory
                .createKeywordsUpdateOperation(Applicability.PARTIAL, updateList, operatorUid,
                        uidAndClientId.getClientId(), uidAndClientId.getUid(), true, false);
        MassResult<UpdatedKeywordInfo> result = keywordsUpdateOperation.prepareAndApply();

        GdValidationResult validationResult =
                gridValidationService.getValidationResult(result, path(field(GdUpdateKeywords.KEYWORD_UPDATE_ITEMS)));
        return new GdUpdateKeywordsPayload()
                .withUpdatedKeywords(getSuccessfullyUpdatedKeywords(gridKeywordsParser, result))
                .withAffectedKeywords(getAffectedKeywords(gridKeywordsParser, keywordsUpdateOperation))
                .withValidationResult(validationResult);
    }

    private static List<ModelChanges<Keyword>> getKeywordUpdateChanges(List<GdUpdateKeywordsItem> keywords) {
        return mapList(keywords,
                item -> new ModelChanges<>(item.getId(), Keyword.class)
                        .processNotNull(item.getKeyword(), Keyword.PHRASE)
                        .processNotNull(item.getPrice(), Keyword.PRICE)
                        .processNotNull(item.getPriceContext(), Keyword.PRICE_CONTEXT)
                        .processNotNull(item.getAutobudgetPriority(), Keyword.AUTOBUDGET_PRIORITY));
    }

    private static List<ModelChanges<Keyword>> getKeywordSuspendChanges(Collection<Long> showConditionIds, Collection<Long> relevanceMatchIds,
                                                                        boolean isSuspended) {
        return filterAndMapList(showConditionIds, Predicate.not(relevanceMatchIds::contains),
                id -> new ModelChanges<>(id, Keyword.class).process(isSuspended, Keyword.IS_SUSPENDED));
    }

    private static List<ModelChanges<RelevanceMatch>> getRelevanceMatchSuspendChanges(Collection<Long> relevanceMatchIds, boolean isSuspended) {
        return mapList(relevanceMatchIds, id -> new ModelChanges<>(id, RelevanceMatch.class).process(isSuspended, RelevanceMatch.IS_SUSPENDED));
    }

    private List<Long> getRelevanceMatchIds(Collection<Long> ids, ClientId clientId) {
        return List.copyOf(relevanceMatchService.getRelevanceMatchIds(clientId, ids));
    }

    private KeywordsUpdateOperation createKeywordUpdateOperation(User operator, User targetUser, List<ModelChanges<?>> keywordChanges) {
        @SuppressWarnings("unchecked")
        List<ModelChanges<Keyword>> castModelChanges = mapList(keywordChanges, mc -> (ModelChanges<Keyword>) mc);

        return keywordOperationFactory.createKeywordsUpdateOperation(Applicability.PARTIAL, castModelChanges, operator.getUid(),
                targetUser.getClientId(), targetUser.getUid(), false, true);
    }

    private RelevanceMatchUpdateOperation createRelevanceMatchUpdateOperation(User operator, User targetUser, List<ModelChanges<?>> relevanceMatch) {
        @SuppressWarnings("unchecked")
        List<ModelChanges<RelevanceMatch>> castModelChanges = mapList(relevanceMatch, mc -> (ModelChanges<RelevanceMatch>) mc);

        Client client = Preconditions.checkNotNull(clientService.getClient(targetUser.getClientId()));
        return relevanceMatchService.createPartialUpdateOperation(client.getWorkCurrency().getCurrency(), targetUser.getClientId(),
                targetUser.getUid(), operator.getUid(), castModelChanges);
    }

    private static ModelChanges<Keyword> toKeywordSuspendModelChanges(Long id, boolean isSuspended) {
        return new ModelChanges<>(id, Keyword.class).process(isSuspended, Keyword.IS_SUSPENDED);
    }

    private static ModelChanges<RelevanceMatch> toRelevanceMatchSuspendModelChanges(Long id, boolean isSuspended) {
        return new ModelChanges<>(id, RelevanceMatch.class).process(isSuspended, RelevanceMatch.IS_SUSPENDED);
    }

    private static List<ModelChanges<?>> getShowConditionsSuspendChanges(Collection<Long> showConditionIds, Collection<Long> relevanceMatchIds,
                                                                         boolean isSuspended) {
        return mapList(showConditionIds, id -> relevanceMatchIds.contains(id)
                ? toRelevanceMatchSuspendModelChanges(id, isSuspended)
                : toKeywordSuspendModelChanges(id, isSuspended));
    }

    private KeywordDeleteOperation createKeywordDeleteOperation(User operator, User targetUser, List<Long> keywordIds) {
        return keywordOperationFactory.createKeywordsDeleteOperation(Applicability.PARTIAL, keywordIds, operator.getUid(), targetUser.getClientId());
    }

    private RelevanceMatchDeleteOperation createRelevanceMatchDeleteOperation(User operator, User targetUser, List<Long> relevanceMatchIds) {
        return relevanceMatchService.createDeleteOperation(targetUser.getClientId(), operator.getUid(), relevanceMatchIds, Applicability.PARTIAL);
    }

    private MassResult<Long> executeSuspendOperation(User subjectUser, User operator, List<Long> showConditionIds, Boolean isSuspended) {
        List<Long> relevanceMatchIds = getRelevanceMatchIds(showConditionIds, subjectUser.getClientId());
        List<ModelChanges<?>> showConditionUpdateList = getShowConditionsSuspendChanges(showConditionIds, relevanceMatchIds, isSuspended);

        OperationCreator<ModelChanges<?>, Operation<Long>> updateKeywordCoreOperationCreator =
                inputItems -> new ConvertResultOperation<>(
                        createKeywordUpdateOperation(operator, subjectUser, inputItems),
                        ResultConverters.massResultValueConverter(nullSafeConverter(UpdatedKeywordInfo::getId))
                );

        OperationCreator<ModelChanges<?>, Operation<Long>> updateRelevanceMatchCoreOperationCreator =
                inputItems -> createRelevanceMatchUpdateOperation(operator, subjectUser, inputItems);

        SplitAndMergeOperationAggregator<ModelChanges<?>, Long> updateOperation =
                SplitAndMergeOperationAggregator.builderForPartialOperations()
                        .addSubOperation(
                                showConditionModelChanges -> showConditionModelChanges.getModelType().equals(Keyword.class),
                                updateKeywordCoreOperationCreator)
                        .addSubOperation(
                                showConditionModelChanges -> showConditionModelChanges.getModelType().equals(RelevanceMatch.class),
                                updateRelevanceMatchCoreOperationCreator)
                        .build();

        MassResult<Long> massResult = updateOperation.execute(showConditionUpdateList);

        return getMassResultWithAggregatedValidationResult(massResult, showConditionIds);
    }

}
