package ru.yandex.direct.grid.processing.service.showcondition.keywords;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.advq.AdvqClient;
import ru.yandex.direct.advq.AdvqSearchOptions;
import ru.yandex.direct.advq.SearchKeywordResult;
import ru.yandex.direct.advq.SearchRequest;
import ru.yandex.direct.advq.search.AdvqRequestKeyword;
import ru.yandex.direct.advq.search.ParserType;
import ru.yandex.direct.advq.search.Statistics;
import ru.yandex.direct.advq.search.StatisticsPhraseItem;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMinusKeywords;
import ru.yandex.direct.core.entity.campaign.service.BaseCampaignService;
import ru.yandex.direct.core.entity.keyword.model.KeywordWithMinuses;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdBulkRefineKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdBulkRefineKeywordsWithMinuses;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdRefineKeyword;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdRefineKeywordPayload;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdRefineKeywordWithMinuses;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.RefinedWord;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.RefinedWordAcc;
import ru.yandex.direct.grid.processing.service.showcondition.converter.RefineWordConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseBeforeNormalizationValidator.minusKeywordsAreValidBeforeNormalization;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION;
import static ru.yandex.direct.grid.processing.model.showcondition.mutation.RefinedWordAcc.copyRefinedWordAcc;
import static ru.yandex.direct.grid.processing.model.showcondition.mutation.RefinedWordAcc.newExactRefinedWordAcc;
import static ru.yandex.direct.grid.processing.model.showcondition.mutation.RefinedWordAcc.newRefinedWordAcc;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class RefineKeywordService {
    /**
     * При каждой последующей загрузке минус-слов, к набору минус-слов frontend'ом добавляются
     * минус-слова предыдущих страниц.
     * Это максимальное количество слов, которые могут быть добавлены к запросу
     * из предыдущих страниц. За одну страницу добавляется {@link #REFINED_WORDS_COUNT_PER_PAGE}.
     */
    static final int MAX_HUMAN_SCROLL_WORD_COUNT = 1000;
    /**
     * Сравнение уточняющих слов по убыванию количества показов.
     * При равенстве, сравнение слов лексикографически.
     */
    static final Comparator<RefinedWord> REFINED_WORD_COMPARATOR =
            Comparator.comparing(RefinedWord::getCount)
                    .reversed()
                    .thenComparing(RefinedWord::getWord);
    /**
     * Заглушка для пустого ответа.
     */
    static final GdRefineKeywordPayload EMPTY_PAYLOAD = new GdRefineKeywordPayload()
            .withHasMore(false)
            .withWords(emptyList());
    /**
     * Количество слов, которые отображаются пользователю при загрузке одной страниы.
     */
    private static final int REFINED_WORDS_COUNT_PER_PAGE = 10;
    /**
     * Таймаут запроса в Advq.
     */
    private static final Duration ADVQ_TIMEOUT = Duration.ofSeconds(120);
    /**
     * Параметры запроса в Advq.
     */
    private static final AdvqSearchOptions ADVQ_SEARCH_OPTIONS = new AdvqSearchOptions()
            .withParserType(ParserType.DIRECT)
            .withCalcTotalHits(Boolean.FALSE)
            .withFastMode(Boolean.TRUE)
            .withPhPage(0)
            .withPhPageSize(100);

    private static final Pattern SQUARE_QUOTES_WORD_PATTERN = Pattern.compile("\\[([^\\s]+)]");
    private static final Pattern SQUARE_QUOTES_MINUS_WORD_PATTERN = Pattern.compile("^[-\\s]*\\[([^\\s]+)]\\s*$");
    private static final Pattern QUOTE_STARTING_PATTERN = Pattern.compile("^\\s*\"");
    private static final Pattern SPACE_PATTERN = Pattern.compile("\\s+");

    private final KeywordNormalizer keywordNormalizer;
    private final KeywordWithLemmasFactory keywordWithLemmasFactory;
    private final BaseCampaignService baseCampaignService;
    private final StopWordService stopWordService;
    private final AdvqClient advqClient;
    private final ShardHelper shardHelper;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final AdGroupRepository adGroupRepository;
    private final GridValidationService gridValidationService;

    @Autowired
    public RefineKeywordService(KeywordNormalizer keywordNormalizer,
                                KeywordWithLemmasFactory keywordWithLemmasFactory,
                                BaseCampaignService baseCampaignService, StopWordService stopWordService,
                                AdvqClient advqClient, ShardHelper shardHelper,
                                MinusKeywordsPackRepository minusKeywordsPackRepository,
                                AdGroupRepository adGroupRepository, GridValidationService gridValidationService) {
        this.keywordNormalizer = keywordNormalizer;
        this.keywordWithLemmasFactory = keywordWithLemmasFactory;
        this.baseCampaignService = baseCampaignService;
        this.stopWordService = stopWordService;
        this.advqClient = advqClient;
        this.shardHelper = shardHelper;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.adGroupRepository = adGroupRepository;
        this.gridValidationService = gridValidationService;
    }

    /**
     * Уточнить ключевую фразу.
     *
     * @param request запрос на уточнение ключевой фразы
     * @return топ минус-слов с примерами использования, отсортированный по количеству показов
     */
    public GdRefineKeywordPayload refine(GdRefineKeyword request) {
        GdBulkRefineKeywords bulkRequest = RefineWordConverter.convertToBulk(request);
        return refine(bulkRequest);
    }

    /**
     * Уточнить набор ключевых фраз.
     *
     * @param request запрос на уточнение ключевых фраз
     * @return топ минус-слов с примерами использования, отсортированный по количеству показов.
     * Предлагаются слова, которыми можно уточнить хотя бы одну ключевую фразу (не обязательно все)
     */
    public GdRefineKeywordPayload refine(GdBulkRefineKeywords request) {
        int minusWordsCount = request.getMinusWords() == null ? 0 : request.getMinusWords().size();
        if (minusWordsCount > MAX_HUMAN_SCROLL_WORD_COUNT) {
            return EMPTY_PAYLOAD;
        }
        List<String> unquotedKeywords = StreamEx.of(request.getKeywords())
                .remove(k -> QUOTE_STARTING_PATTERN.matcher(k).find())
                .toList();
        if (unquotedKeywords.isEmpty()) {
            return EMPTY_PAYLOAD;
        }
        List<KeywordWithMinuses> advqKeywords = StreamEx.of(unquotedKeywords)
                .map(k -> buildKeywordToAdvq(k, request.getMinusWords()))
                .toList();
        Map<KeywordWithMinuses, Statistics> advqStats = getAdvqStatistics(advqKeywords, request.getGeo(),
                request.getAdGroupType());
        if (advqStats.isEmpty()) {
            return EMPTY_PAYLOAD;
        }
        Set<String> minusWords = request.getMinusWords() == null || request.getMinusWords().isEmpty()
                ? emptySet()
                : new HashSet<>(request.getMinusWords());
        List<RefinedWord> wordsStat = computeWordsStat(advqStats, minusWords);
        List<RefinedWord> subTopWords = wordsStat.size() > REFINED_WORDS_COUNT_PER_PAGE
                ? wordsStat.subList(0, REFINED_WORDS_COUNT_PER_PAGE)
                : wordsStat;
        boolean hasMore = wordsStat.size() > REFINED_WORDS_COUNT_PER_PAGE ||
                (wordsStat.size() == REFINED_WORDS_COUNT_PER_PAGE && anyHasNextPage(advqStats.values()));
        return new GdRefineKeywordPayload()
                .withWords(subTopWords)
                .withHasMore(hasMore);
    }

    /**
     * Уточнить ключевую фразу.
     *
     * @param clientId    идентификатор клиента
     * @param operatorUid id оператора
     * @param request     запрос на уточнение ключевой фразы
     * @return топ минус-слов с примерами использования, отсортированный по количеству показов
     */
    public GdRefineKeywordPayload refine(ClientId clientId, Long operatorUid, GdRefineKeywordWithMinuses request) {
        GdBulkRefineKeywordsWithMinuses bulkRequest = RefineWordConverter.convertToBulk(request);
        return refine(clientId, operatorUid, bulkRequest);
    }

    /**
     * Уточнить набор ключевых фраз.
     *
     * @param clientId    идентификатор клиента
     * @param operatorUid id оператора
     * @param request     запрос на уточнение ключевых фраз
     * @return топ минус-слов с примерами использования, отсортированный по количеству показов.
     * Предлагаются слова, которыми можно уточнить хотя бы одну ключевую фразу (не обязательно все)
     */
    public GdRefineKeywordPayload refine(ClientId clientId, Long operatorUid, GdBulkRefineKeywordsWithMinuses request) {
        int minusWordsCount = request.getMinusWords() == null ? 0 : request.getMinusWords().size();
        if (minusWordsCount > MAX_HUMAN_SCROLL_WORD_COUNT) {
            return EMPTY_PAYLOAD;
        }

        ValidationResult<GdBulkRefineKeywordsWithMinuses, Defect> vr = validateRequest(request);
        if (vr.hasAnyErrors()) {
            var gridVr = gridValidationService.toGdValidationResult(vr, path());
            return new GdRefineKeywordPayload().withValidationResult(gridVr).withHasMore(false);
        }

        Set<String> minusWords = collectMinusWords(clientId, operatorUid, request);

        List<String> unquotedKeywords = StreamEx.of(request.getKeywords())
                .remove(k -> QUOTE_STARTING_PATTERN.matcher(k).find())
                .toList();
        if (unquotedKeywords.isEmpty()) {
            return EMPTY_PAYLOAD;
        }
        List<KeywordWithMinuses> advqKeywords = StreamEx.of(unquotedKeywords)
                .map(k -> buildKeywordToAdvq(k, minusWords))
                .toList();
        Map<KeywordWithMinuses, Statistics> advqStats = getAdvqStatistics(advqKeywords, request.getGeo(),
                request.getAdGroupType());
        if (advqStats.isEmpty()) {
            return EMPTY_PAYLOAD;
        }

        List<RefinedWord> wordsStat = computeWordsStat(advqStats, minusWords);
        List<RefinedWord> subTopWords = wordsStat.size() > REFINED_WORDS_COUNT_PER_PAGE
                ? wordsStat.subList(0, REFINED_WORDS_COUNT_PER_PAGE)
                : wordsStat;
        boolean hasMore = wordsStat.size() > REFINED_WORDS_COUNT_PER_PAGE ||
                (wordsStat.size() == REFINED_WORDS_COUNT_PER_PAGE && anyHasNextPage(advqStats.values()));
        return new GdRefineKeywordPayload()
                .withWords(subTopWords)
                .withHasMore(hasMore);
    }

    private Set<String> collectMinusWords(ClientId clientId, Long operatorUid,
                                          GdBulkRefineKeywordsWithMinuses request) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Set<String> minusWords = new HashSet<>();
        if (request.getMinusWords() != null) {
            minusWords.addAll(request.getMinusWords());
        }

        List<Long> mwIds = new ArrayList<>();
        // по id группы
        if (request.getAdGroupId() != null) {
            List<Long> adGroupIds = List.of(request.getAdGroupId());
            // минус-слова на группе
            var adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);
            minusWords.addAll(StreamEx.of(adGroups).flatMap(ag -> ag.getMinusKeywords().stream()).toSet());

            // библиотечные наборы
            var adGroupIdToMinusWordPackIds = minusKeywordsPackRepository.getAdGroupsLibraryMinusKeywordsPacks(shard,
                    adGroupIds);
            mwIds.addAll(EntryStream.of(adGroupIdToMinusWordPackIds).values().flatMap(Collection::stream).toSet());

        }

        // по id наборов минус-фраз на группе
        if (request.getAdGroupMinusKeywordPackIds() != null) {
            mwIds.addAll(request.getAdGroupMinusKeywordPackIds());
        }

        // по id кампании
        if (request.getCampaignId() != null) {
            List<Long> campaignIds = List.of(request.getCampaignId());
            // минус-слова на кампании
            var campaigns = baseCampaignService.get(clientId, operatorUid, campaignIds);
            Set<String> campaignMinusWords = StreamEx.of(campaigns).select(CampaignWithMinusKeywords.class)
                    .flatMap(c -> c.getMinusKeywords().stream())
                    .toSet();
            minusWords.addAll(campaignMinusWords);

            // библиотечные наборы
            var campaignIdToMinusWordPackIds = minusKeywordsPackRepository.getCampaignsLibraryMinusKeywordsPacks(shard,
                    campaignIds);
            mwIds.addAll(EntryStream.of(campaignIdToMinusWordPackIds).values().flatMap(Collection::stream).toSet());
        }

        // по id наборов минус-фраз на кампании
        if (request.getCampaignMinusKeywordPackIds() != null) {
            mwIds.addAll(request.getCampaignMinusKeywordPackIds());
        }

        List<MinusKeywordsPack> minusKeywordsPacks = minusKeywordsPackRepository.get(shard, clientId, mwIds);
        List<String> packMinusWords = StreamEx.of(minusKeywordsPacks)
                .flatMap(p -> p.getMinusKeywords().stream())
                .toList();
        minusWords.addAll(packMinusWords);
        return minusWords;
    }

    private ValidationResult<GdBulkRefineKeywordsWithMinuses, Defect> validateRequest(GdBulkRefineKeywordsWithMinuses request) {
        ModelItemValidationBuilder<GdBulkRefineKeywordsWithMinuses> vb = ModelItemValidationBuilder.of(request);

        vb.item(GdBulkRefineKeywordsWithMinuses.MINUS_WORDS)
                .checkBy(minusKeywordsAreValidBeforeNormalization(
                        GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION,
                        MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD), When.notNull());

        return vb.getResult();
    }

    /**
     * @param keyword    ключевая фраза
     * @param minusWords минус-слова
     * @return обработанная ключевая фраза вместе с минус-словами
     */
    KeywordWithMinuses buildKeywordToAdvq(String keyword, @Nullable Collection<String> minusWords) {
        String cleanedKeyword = cleanKeyword(keyword);
        KeywordWithMinuses keywordWithMinuses = KeywordWithMinuses.fromPhrase(cleanedKeyword);
        if (minusWords == null || minusWords.isEmpty()) {
            return keywordWithMinuses;
        }
        Set<String> cleanedMinusWords = cleanMinusWords(keywordWithMinuses.getPlusKeyword(), minusWords);
        keywordWithMinuses.addMinusKeywords(cleanedMinusWords);
        return keywordWithMinuses;
    }

    /**
     * @param keyword ключевая фраза
     * @return ключевая фраза, очищенная от квадратных скобок
     */
    private String cleanKeyword(String keyword) {
        return SQUARE_QUOTES_WORD_PATTERN.matcher(keyword).replaceAll("$1");
    }

    /**
     * @param keyword    ключевая фраза
     * @param minusWords минус-слова
     * @return минус-слова, очищенные от стоп-слов и тех, которые полностью вычитают ключевую фразу
     */
    private Set<String> cleanMinusWords(String keyword, Collection<String> minusWords) {
        Set<String> cleanedMinusWords = StreamEx.of(minusWords)
                .distinct()
                .map(mw -> {
                    String cleanedMw = SQUARE_QUOTES_MINUS_WORD_PATTERN.matcher(mw).replaceFirst("$1");
                    return StreamEx.split(cleanedMw, SPACE_PATTERN)
                            .map(w -> stopWordService.isStopWord(w) ? '+' + w : w)
                            .joining(" ");
                })
                .toSet();
        Set<String> interWords = getIntersection(keyword, cleanedMinusWords);
        cleanedMinusWords.removeAll(interWords);
        return cleanedMinusWords;
    }

    @SuppressWarnings("ConstantConditions")
    private Map<KeywordWithMinuses, Statistics> getAdvqStatistics(List<KeywordWithMinuses> advqKeywords,
                                                                  List<Long> geo,
                                                                  GdAdGroupType adGroupType) {
        Map<String, KeywordWithMinuses> keywordsWithMinuses =
                listToMap(advqKeywords, KeywordWithMinuses::toString, Function.identity());
        List<String> keywords = new ArrayList<>(keywordsWithMinuses.keySet());
        SearchRequest searchRequest = SearchRequest.fromPhrases(keywords, geo);
        Map<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> advqResults;
        if (adGroupType == GdAdGroupType.CONTENT_PROMOTION_VIDEO) {
            advqResults = advqClient.searchVideo(singletonList(searchRequest), ADVQ_SEARCH_OPTIONS, ADVQ_TIMEOUT);
        } else {
            advqResults = advqClient.search(singletonList(searchRequest), ADVQ_SEARCH_OPTIONS, ADVQ_TIMEOUT);
        }
        Map<AdvqRequestKeyword, SearchKeywordResult> keywordToAdvqResult = advqResults.get(searchRequest);

        Map<KeywordWithMinuses, Statistics> statistics = new HashMap<>();
        searchRequest.getKeywords().forEach(k -> {
            SearchKeywordResult searchKeywordResult = keywordToAdvqResult.get(k);
            if (searchKeywordResult == null || searchKeywordResult.isError()) {
                return;
            }
            Statistics stat = searchKeywordResult.getResult().getStat();
            if (stat.getIncludingPhrases() == null) {
                return;
            }
            KeywordWithMinuses keyword = keywordsWithMinuses.get(k.getPhrase());
            statistics.put(keyword, RefineWordConverter.fixStatisticsForContentPromotionCollections(stat, adGroupType));
        });
        return statistics;
    }

    /**
     * Разбирается каждая вложенная фраза, полученная от Advq:
     * <ul>
     * <li>очищается от стоп-слов,</li>
     * <li>слов, которые пересекаются с ключевой фразой,</li>
     * <li>слов, которые пересекаются с минус-словами,</li>
     * <li>нормализуется.</li>
     * </ul>
     * Подсчитывается количество показов по таким словам.
     *
     * @param advqStats  статистика, полученная от Advq
     * @param minusWords минус-слова
     * @return статистика по уточняющим словам
     */
    private List<RefinedWord> computeWordsStat(Map<KeywordWithMinuses, Statistics> advqStats, Set<String> minusWords) {
        Collection<Optional<RefinedWord>> refineWords = EntryStream.of(advqStats)
                .map(s -> computeWordsStat(s.getKey().getPlusKeyword(), s.getValue(), minusWords))
                .flatMap(m -> m.entrySet().stream())
                .map(e -> new RefinedWord(e.getKey(), e.getValue().getPhrases(), e.getValue().getEffectiveCount()))
                .groupingBy(RefinedWord::getWord, Collectors.reducing(RefinedWord::union))
                .values();

        return StreamEx.of(refineWords)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .sorted(REFINED_WORD_COMPARATOR)
                .toList();
    }

    private Map<String, RefinedWordAcc> computeWordsStat(String plusKeyword,
                                                         Statistics advqStat,
                                                         Set<String> minusWords) {
        Set<String> plusKeywords = StreamEx.split(plusKeyword, SPACE_PATTERN).toSet();
        Map<String, RefinedWordAcc> wordsStat = new HashMap<>();
        for (StatisticsPhraseItem spi : advqStat.getIncludingPhrases()) {
            String phrase = spi.getPhrase();
            List<String> words = StreamEx.split(phrase, SPACE_PATTERN)
                    .remove(x -> x.startsWith("+"))
                    .filter(StringUtils::isNotBlank)
                    .remove(plusKeywords::contains)
                    .remove(minusWords::contains)
                    .remove(stopWordService::isStopWord)
                    .remove(word -> hasIntersection(plusKeywords, word))
                    .map(keywordNormalizer::normalizeKeyword)
                    .toList();

            boolean isExact = words.size() == 1;
            RefinedWordAcc stat = isExact
                    ? newExactRefinedWordAcc(spi.getPhrase(), spi.getCnt())
                    : newRefinedWordAcc(spi.getPhrase(), spi.getCnt());
            words.forEach(word -> wordsStat.merge(word, copyRefinedWordAcc(stat), RefinedWordAcc::add));
        }
        return wordsStat;
    }

    /**
     * @param plusWord  ключевая фраза
     * @param minusWord минус-слово
     * @return {@code true}, если {@code minusWord} вычитает {@code plusWord}
     */
    private boolean hasIntersection(String plusWord, String minusWord) {
        Set<String> intersectionWords = getIntersection(plusWord, singletonList(minusWord));
        return !intersectionWords.isEmpty();
    }

    /**
     * @param plusWords ключевая фраза
     * @param minusWord минус-слово
     * @return {@code true}, если {@code minusWord} вычитает {@code plusWord}
     */
    private boolean hasIntersection(Set<String> plusWords, String minusWord) {
        return StreamEx.of(plusWords).anyMatch(pw -> hasIntersection(pw, minusWord));
    }

    /**
     * @param keyword    ключевая фраза
     * @param minusWords минус-слова
     * @return список минус-фраз, которые вычитают ключевую фразу
     */
    private Set<String> getIntersection(String keyword, Collection<String> minusWords) {
        if (minusWords.isEmpty()) {
            return emptySet();
        }
        return KeywordInclusionUtils.getIncludedMinusKeywords(
                keywordWithLemmasFactory,
                stopWordService::isStopWord,
                singletonList(keyword),
                minusWords
        );
    }

    /**
     * @param advqStats статистика фраз, полученная от Advq
     * @return {@code true}, если хоть по одной фразе есть еще статистики из Advq.
     */
    private boolean anyHasNextPage(Collection<Statistics> advqStats) {
        return StreamEx.of(advqStats).anyMatch(Statistics::isHasNextPage);
    }
}
