package ru.yandex.direct.grid.processing.service.showcondition.keywords;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bsauction.BasicBsRequestPhrase;
import ru.yandex.direct.bsauction.BsRequest;
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.auction.BsRequestPhraseWrapperAdditionalData;
import ru.yandex.direct.core.entity.auction.container.BsRequestPhraseWrapper;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.service.BsAuctionService;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.core.entity.bids.service.PokazometerService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.model.ForecastCtr;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.KeywordBsAuctionService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.ShowConditionFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowCondition;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionFilter;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionPrimaryStatus;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionType;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionWithTotals;
import ru.yandex.direct.grid.core.entity.showcondition.service.GridShowConditionService;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdKeywordAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.utils.GridModerationUtils;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.banner.GdAd;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.forecast.GdDeviceType;
import ru.yandex.direct.grid.processing.model.forecast.GdForecastContainer;
import ru.yandex.direct.grid.processing.model.forecast.GdShowStatContainer;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.showcondition.GdAuctionData;
import ru.yandex.direct.grid.processing.model.showcondition.GdPokazometerData;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowCondition;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionType;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionWithTotals;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionsContainer;
import ru.yandex.direct.grid.processing.service.aggregatedstatuses.KeywordsPrimaryStatusCalculator;
import ru.yandex.direct.grid.processing.service.banner.BannerDataService;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.group.GroupDataService;
import ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordConverter;
import ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionConverter;
import ru.yandex.direct.grid.processing.util.GoalHelper;
import ru.yandex.direct.grid.processing.util.ReasonsFilterUtils;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.pokazometer.GroupRequest;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.Counter;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.emptyIfNull;
import static ru.yandex.direct.core.entity.auction.service.BsAuctionService.PHRASES_CHUNK_LIMIT;
import static ru.yandex.direct.core.entity.auction.utils.BsAuctionUtils.prepareText;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasNoAutotargetingPrefix;
import static ru.yandex.direct.feature.FeatureName.HIDE_OLD_SHOW_CAMPS_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.SHOW_AGGREGATED_STATUS_OPEN_BETA;
import static ru.yandex.direct.feature.FeatureName.SHOW_DNA_BY_DEFAULT;
import static ru.yandex.direct.grid.processing.model.forecast.GdDeviceType.ALL;
import static ru.yandex.direct.grid.processing.model.forecast.GdDeviceType.DESKTOP;
import static ru.yandex.direct.grid.processing.model.forecast.GdDeviceType.PHONE;
import static ru.yandex.direct.grid.processing.model.forecast.GdDeviceType.TABLET;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordConverter.getAuctionsDataItems;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.KeywordConverter.getKeywordsIds;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionConverter.addAuctionDataToShowCondition;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionConverter.hasCampaignContextPricesToShow;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionConverter.toInternalFilter;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.ShowConditionConverter.toOuter;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, возвращающий данные об условиях показа в формате, пригодном для выдачи внешним клиентам
 */
@Service
@ParametersAreNonnullByDefault
public class ShowConditionDataService {

    private final GridShowConditionService showConditionService;
    private final KeywordBsAuctionService keywordBsAuctionService;
    private final PokazometerService pokazometerService;
    private final CampaignRepository campaignRepository;
    private final KeywordRepository keywordRepository;
    private final BannerDataService bannerDataService;
    private final CampaignInfoService campaignInfoService;
    private final GroupDataService groupDataService;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;
    private final FeatureService featureService;
    private final BsAuctionService bsAuctionService;
    private final BannersUrlHelper bannersUrlHelper;
    private final ClientService clientService;

    public ShowConditionDataService(BannerDataService bannerDataService,
                                    GridShowConditionService showConditionService,
                                    KeywordBsAuctionService keywordBsAuctionService,
                                    PokazometerService pokazometerService,
                                    CampaignRepository campaignRepository,
                                    KeywordRepository keywordRepository,
                                    CampaignInfoService campaignInfoService,
                                    GroupDataService groupDataService,
                                    AggregatedStatusesViewService aggregatedStatusesViewService,
                                    FeatureService featureService, BsAuctionService bsAuctionService,
                                    BannersUrlHelper bannersUrlHelper, ClientService clientService) {
        this.bannerDataService = bannerDataService;
        this.showConditionService = showConditionService;
        this.keywordBsAuctionService = keywordBsAuctionService;
        this.pokazometerService = pokazometerService;
        this.campaignRepository = campaignRepository;
        this.keywordRepository = keywordRepository;
        this.campaignInfoService = campaignInfoService;
        this.groupDataService = groupDataService;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.featureService = featureService;
        this.bsAuctionService = bsAuctionService;
        this.bannersUrlHelper = bannersUrlHelper;
        this.clientService = clientService;
    }

    private static List<GdShowCondition> resultPostProcessing(List<GdShowCondition> gdShowConditions,
                                                              GdShowConditionsContainer inputContainer) {
        if (inputContainer.getFilter().getMinPriceContext() != null) {
            //DIRECT-86313: Не показывать фразы если есть фильтрация по priceContext
            return filterList(gdShowConditions, ShowConditionConverter::hasContextPriceToShow);
        }
        return gdShowConditions;
    }

    /**
     * Получить список условий показа в формате, пригодном для возвращения оператору
     * <p>
     * (!) В этом методе не заполняется {@link GdShowCondition#MAIN_AD}.
     * Это делается после того, как все лишние сonditions будут отфильтрованы и обрезаны согласно {@code limitOffset},
     * чтобы не выгружать лишние баннеры.
     * Подробности в https://st.yandex-team.ru/DIRECT-152022#612e07a2b2f2483288ec26b6
     *
     * @param client      клиент, которому принадлежат условия показа
     * @param input       входные параметры для получения условий показа
     * @param context     контекст исполнения GraphQL запроса
     * @param limitOffset запрашиваемый диапазон условий показа из общего списка
     */
    public GdShowConditionWithTotals getShowConditions(GdClientInfo client, GdShowConditionsContainer input,
                                                       GridGraphQLContext context, LimitOffset limitOffset,
                                                       boolean cpcAndCpmOnOneGridEnabled) {
        var operator = context.getOperator();
        int shard = client.getShard();
        ClientId clientId = ClientId.fromLong(client.getId());
        ShowConditionFetchedFieldsResolver showConditionFetchedFieldsResolver =
                context.getFetchedFieldsReslover().getShowCondition();
        List<GdiShowConditionOrderBy> internalOrderByList =
                mapList(input.getOrderBy(), ShowConditionConverter::toInternalOrderBy);

        Set<Long> goalIds =
                GoalHelper.combineGoalIds(input.getStatRequirements().getGoalIds(), input.getFilter().getGoalStats());

        boolean withFilterByAggrStatus = featureService.anyEnabled(operator.getClientId(),
                Set.of(SHOW_DNA_BY_DEFAULT, HIDE_OLD_SHOW_CAMPS_FOR_DNA, SHOW_AGGREGATED_STATUS_OPEN_BETA));
        GdiShowConditionFilter internalFilter = toInternalFilter(input.getFilter());

        boolean addWithTotalsToQuery = featureService.isEnabledForClientId(operator.getClientId(),
                FeatureName.ADD_WITH_TOTALS_TO_SHOW_CONDITION_QUERY);

        GdiShowConditionWithTotals showConditionWithTotals = showConditionService
                .getShowConditions(shard, operator.getUid(), clientId, internalFilter, internalOrderByList,
                        input.getStatRequirements().getFrom(), input.getStatRequirements().getTo(),
                        goalIds,
                        showConditionFetchedFieldsResolver, withFilterByAggrStatus, addWithTotalsToQuery);
        List<GdiShowCondition> conditions = showConditionWithTotals.getGdiShowConditions();
        GdEntityStats gdEntityStats = showConditionWithTotals.getTotalStats() != null
                ? StatHelper.convertInternalStatsToOuter(showConditionWithTotals.getTotalStats())
                : null;
        if (conditions.isEmpty()) {
            return new GdShowConditionWithTotals()
                    .withGdShowConditions(emptyList())
                    .withTotalStats(gdEntityStats);
        }

        Set<Long> campaignIds = listToSet(conditions, GdiShowCondition::getCampaignId);
        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService.getTruncatedCampaigns(clientId, campaignIds);
        conditions = filterList(conditions, c -> campaignsById.containsKey(c.getCampaignId()));
        if (conditions.isEmpty()) {
            return new GdShowConditionWithTotals()
                    .withGdShowConditions(emptyList())
                    .withTotalStats(gdEntityStats);
        }

        Set<Long> adGroupIds = listToSet(conditions, GdiShowCondition::getGroupId);
        List<GdAdGroupTruncated> adGroups = groupDataService
                .getTruncatedAdGroups(client.getShard(), client.getCountryRegionId(), clientId,
                        context.getOperator(), adGroupIds, campaignsById);
        if (adGroups.isEmpty()) {
            return new GdShowConditionWithTotals()
                    .withGdShowConditions(emptyList())
                    .withTotalStats(gdEntityStats);
        }
        Map<Long, GdAdGroupTruncated> adGroupsById = listToMap(adGroups, GdAdGroupTruncated::getId);

        //Берем данные торгов и показометра, только если они запрошены
        //По умолчанию берем данные торгов только для запрошенного диапазона условий показа.
        Map<Long, GdAuctionData> bsAuctionData = showConditionFetchedFieldsResolver.getAuctionData()
                ? getGdAuctionData(shard, clientId, getKeywordsIds(limitOffset, conditions))
                : Collections.emptyMap();

        //для показометра выбираем только те условия показа, по которым нужны показы в сети
        List<GdiShowCondition> conditionsWithContextPriceShow =
                filterList(conditions,
                        c -> c.getType() == GdiShowConditionType.KEYWORD &&
                                hasNoAutotargetingPrefix(c.getIsAutotargeting()) &&
                                hasCampaignContextPricesToShow(campaignsById.get(c.getCampaignId())));
        Map<Long, GdPokazometerData> keywordPokazometerData =
                conditionsWithContextPriceShow.isEmpty() || !showConditionFetchedFieldsResolver.getPokazometrData()
                        ? Collections.emptyMap()
                        : getPokazometerData(conditionsWithContextPriceShow, adGroupsById, client.getWorkCurrency());

        Counter counter = new Counter();

        List<GdShowCondition> gdShowConditions =
                conditions.stream()
                        .filter(e -> adGroupsById.containsKey(e.getGroupId()))
                        .map(c -> toOuter(counter.next(), c, campaignsById.get(c.getCampaignId()),
                                adGroupsById.get(c.getGroupId()),
                                bsAuctionData.get(c.getId()),
                                keywordPokazometerData.getOrDefault(c.getId(), null),
                                cpcAndCpmOnOneGridEnabled))
                        .collect(Collectors.toList());

        List<GdShowCondition> keywordsOrRelevantMatch = gdShowConditions.stream()
                .filter(c -> c.getType().equals(GdShowConditionType.KEYWORD)
                        || c.getType().equals(GdShowConditionType.RELEVANCE_MATCH)).collect(Collectors.toList());
        Set<Long> ids = listToSet(keywordsOrRelevantMatch, GdShowCondition::getId);
        Set<Long> pids = listToSet(keywordsOrRelevantMatch, GdShowCondition::getAdGroupId);
        Map<Long, AggregatedStatusKeywordData> statusesByIds =
                aggregatedStatusesViewService.getKeywordStatusesByIds(shard, pids, ids);

        boolean showAggregatedStatusesDebug = featureService
                .isEnabledForClientId(operator.getClientId(), FeatureName.SHOW_AGGREGATED_STATUS_DEBUG);
        Set<GdiShowConditionPrimaryStatus> filterStatuses =
                new HashSet<>(emptyIfNull(internalFilter.getShowConditionStatusIn()));

        gdShowConditions = gdShowConditions.stream()
                .filter(condition -> !withFilterByAggrStatus
                        || filterStatuses.isEmpty()
                        || filterStatuses.contains(KeywordsPrimaryStatusCalculator
                        .convertToPrimaryStatus(statusesByIds.get(condition.getId()))))
                .filter(condition -> !withFilterByAggrStatus
                        || ReasonsFilterUtils.isValid(internalFilter.getReasonsContainSome(),
                        statusesByIds.get(condition.getId())))
                .peek(condition -> {
                    if (ids.contains(condition.getId())) {
                        AggregatedStatusKeywordData status = statusesByIds.get(condition.getId());
                        if (status != null && status.getStatus().isPresent()) {
                            condition.setAggregatedStatusInfo(new GdKeywordAggregatedStatusInfo()
                                    .withStatus(status.getStatusUnsafe())
                                    .withReasons(status.getReasons())
                                    .withRejectReasons(GridModerationUtils.toGdRejectReasons(status.getRejectReasons()))
                                    .withIsObsolete(status.getIsObsolete()));
                            if (showAggregatedStatusesDebug) {
                                condition.setAggregatedStatus(aggregatedStatusesViewService.toJson(status));
                            }
                        }
                    }
                })
                .collect(Collectors.toList());

        return new GdShowConditionWithTotals()
                .withGdShowConditions(resultPostProcessing(gdShowConditions, input))
                .withTotalStats(gdEntityStats);
    }

    public void setMainAds(GdClientInfo client, List<GdShowCondition> conditions) {
        var adGroupsById =
                StreamEx.of(conditions)
                        .toMap(GdShowCondition::getAdGroupId, GdShowCondition::getAdGroup, (g1, g2) -> g1);
        var mainBannersToGroup = getMainBanners(client, adGroupsById);
        for (var condition : conditions) {
            var mainAd = mainBannersToGroup.get(condition.getAdGroupId());
            condition.setMainAd(mainAd);
        }
    }

    private Map<Long, GdAd> getMainBanners(int shard, ClientId clientId, Map<Long, GdAdGroupTruncated> groups) {
        try (TraceProfile ignore = Trace.current().profile("showConditions:getMainBanners")) {
            return bannerDataService.getMainBanners(shard, clientId, groups);
        }
    }

    private Map<Long, GdAd> getMainBanners(GdClientInfo client, Map<Long, GdAdGroupTruncated> groups) {
        var shard = client.getShard();
        var clientId = ClientId.fromLong(client.getId());
        return getMainBanners(shard, clientId, groups);
    }

    public Map<Long, GdAuctionData> getGdAuctionData(int shard, ClientId clientId, List<Long> keywordIds) {
        List<Keyword> keywords = keywordRepository.getKeywordsByIds(shard, clientId, keywordIds);

        List<Long> campaignIds = mapList(keywords, Keyword::getCampaignId);
        Map<Long, Campaign> campaignByIds = campaignRepository.getCampaignsMap(shard, campaignIds);

        //Нужна мапа id->data
        List<KeywordTrafaretData> keywordTrafaretData =
                keywordBsAuctionService.getTrafaretAuction(clientId, keywords, campaignByIds);
        return StreamEx.of(keywordTrafaretData)
                .mapToEntry(data -> data.getKeyword().getId(), data -> new GdAuctionData()
                        .withAuctionDataItems(getAuctionsDataItems(data)))
                .toMap();
    }

    public Map<String, GdAuctionData> getGdAuctionDataByPhrase(int shard, ClientId clientId, GdForecastContainer input,
                                                               Map<String, ForecastCtr> forecastCtrByPhrase) {
        List<BsRequest<BsRequestPhraseWrapper>> bsRequests = getBsRequests(shard, clientId, input, forecastCtrByPhrase);

        List<KeywordTrafaretData> keywordTrafaretData = bsAuctionService.getBsTrafaretResults(bsRequests);

        return StreamEx.of(keywordTrafaretData)
                .mapToEntry(KeywordTrafaretData::getPhrase, data -> new GdAuctionData()
                        .withAuctionDataItems(getAuctionsDataItems(data)))
                .toMap();
    }

    private List<BsRequest<BsRequestPhraseWrapper>> getBsRequests(int shard, ClientId clientId,
                                                                  GdForecastContainer input,
                                                                  Map<String, ForecastCtr> forecastCtrByPhrase) {
        GdShowStatContainer target = input.getTarget();
        List<Long> geo = target.getGeo();
        var additionalDataForBsAuction = input.getAdditionalDataForBsAuction();
        List<String> phrases = target.getKeywords().getPhrases();
        Currency clientCurrency = clientService.getWorkCurrency(shard, clientId);

        var allBsPhrases = mapList(phrases, phrase -> {
            ForecastCtr forecastCtr = forecastCtrByPhrase.get(phrase);
            // TODO(dimitrovsd): вместо 0L нужно значение keyword.getShowsForecast()
            var bsRequestPhraseStat = bsAuctionService.getRequestPhraseStatForUnknownToBs(forecastCtr, 0L);

            return new BsRequestPhraseWrapper(
                    new BasicBsRequestPhrase()
                            .withText(prepareText(phrase))
                            .withStat(bsRequestPhraseStat))
                    .withAdditionalData(new BsRequestPhraseWrapperAdditionalData(
                            additionalDataForBsAuction.getBannerHref(),
                            false,
                            clientCurrency,
                            phrase
                    ));
        });
        var bsPhrasesChunked = Lists.partition(allBsPhrases, PHRASES_CHUNK_LIMIT);

        // TODO(dimitrovsd): нужно ли работать с filter domain?
        String domain = bannersUrlHelper.extractHostFromHrefWithWwwOrNull(additionalDataForBsAuction.getBannerHref());
        Set<GdDeviceType> deviceTypes = nvl(target.getDeviceTypes(), Set.of(ALL));

        return StreamEx.of(bsPhrasesChunked)
                .map(bsPhrasesChunk -> new BsRequest<BsRequestPhraseWrapper>()
                        .withOrderId(0)
                        .withNoExtendedGeotargeting(false)
                        .withTimetable(false)
                        .withOnlyMobilePages(false)
                        .withRegionIds(geo)
                        .withBannerHead(additionalDataForBsAuction.getBannerTitle())
                        .withBannerBody(additionalDataForBsAuction.getBannerBody())
                        .withCurrency(clientCurrency)
                        .withDomain(domain)
                        .withPhrases(bsPhrasesChunk)
                        .withTargetDeviceTablet(deviceTypes.contains(ALL) || deviceTypes.contains(TABLET))
                        .withTargetDevicePhone(deviceTypes.contains(ALL) || deviceTypes.contains(PHONE))
                        .withTargetDeviceDesktop(deviceTypes.contains(ALL) || deviceTypes.contains(DESKTOP)))
                .toList();
    }

    /**
     * Получить данные об охвате от показометра по указанным условиям показа
     *
     * @param conditionsWithContextPriceShow - условия показа, для которых требуется отображать ставку для сети
     * @param adGroupsById                   мапа, с группами объявлений: adGroupId -> GdAdGroupTruncated
     * @param currencyCode                   - код валюты клиента. В эту валюту будет сконвертирован ответ показометра.
     * @return - возвращаем мапу с keywordId -> [ {coveragePreset, price}, .. ]
     */
    public Map<Long, GdPokazometerData> getPokazometerData(
            List<GdiShowCondition> conditionsWithContextPriceShow,
            Map<Long, GdAdGroupTruncated> adGroupsById,
            CurrencyCode currencyCode) {
        if (conditionsWithContextPriceShow.isEmpty()) {
            return Collections.emptyMap();
        }

        //Группируем условия показа по geo, чтобы сократить количество запросов в показометр
        Map<ImmutableList<Integer>, List<GdiShowCondition>> showConditionByGeo =
                StreamEx.of(conditionsWithContextPriceShow)
                        .mapToEntry(
                                condition -> adGroupsById.get(condition.getGroupId()).getRegionsInfo().getRegionIds(),
                                identity())
                        .mapKeys(ImmutableList::sortedCopyOf)
                        .grouping();

        List<GroupRequest> groupRequests = EntryStream.of(showConditionByGeo)
                .mapKeyValue(ShowConditionConverter::toPokazometerGroupRequest)
                .toList();

        List<KeywordBidPokazometerData> keywordBidPokazometerData =
                pokazometerService.getPokazometerResults(groupRequests, currencyCode);

        return listToMap(keywordBidPokazometerData,
                KeywordBidPokazometerData::getKeywordId, KeywordConverter::toGdPokazometerData);
    }

    public List<GdShowCondition> populateKeywordsWithExternalPriceData(GdClientInfo client, List<Long> keywordIds,
                                                                       List<GdShowCondition> rowset) {
        Map<Long, GdAuctionData> auctionDataMap =
                getGdAuctionData(client.getShard(), ClientId.fromLong(client.getId()), keywordIds);
        rowset.stream()
                .filter(gdShowCondition -> auctionDataMap.containsKey(gdShowCondition.getId()))
                .forEach(gdShowCondition -> addAuctionDataToShowCondition(gdShowCondition,
                        gdShowCondition.getAdGroup().getCampaign(), auctionDataMap.get(gdShowCondition.getId())));
        return rowset;
    }

}
