package ru.yandex.direct.grid.processing.service.showcondition.retargeting;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.dataloader.DataLoader;
import org.dataloader.MappedBatchLoaderWithContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingCondition;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionWithMetrikaFlag;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * {@link DataLoader} для отложенной batch'евой загрузки информации о пользователях
 *
 * @see GridBatchingDataLoader
 * @see ru.yandex.direct.grid.processing.service.dataloader.GridDataLoaderRegistry
 */
@Component
@ParametersAreNonnullByDefault
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
public class RetargetingConditionDataLoader extends GridBatchingDataLoader<Long,
        GdRetargetingConditionWithMetrikaFlag> {
    private static final Logger logger = LoggerFactory.getLogger(RetargetingConditionDataLoader.class);


    public RetargetingConditionDataLoader(GridContextProvider gridContextProvider,
                                          RetargetingDataService retargetingDataService) {
        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(retargetingDataService));
    }

    private MappedBatchLoaderWithContext<Long, GdRetargetingConditionWithMetrikaFlag> getBatchLoadFunction(RetargetingDataService retargetingDataService) {
        return (retargetingConditionIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");
            Map<Long, GdRetargetingConditionWithMetrikaFlag> retargetingConditionById;
            try {
                List<GdRetargetingCondition> retargetingConditions =
                        retargetingDataService.getRetargetingConditionsRowset(queriedClient,
                                new GdRetargetingConditionFilter()
                                        .withRetargetingConditionIdIn(retargetingConditionIds));

                retargetingConditionById =
                        StreamEx.of(retargetingConditions)
                                .mapToEntry(GdRetargetingCondition::getRetargetingConditionId, Function.identity())
                                .mapValues(r -> new GdRetargetingConditionWithMetrikaFlag()
                                        .withIsMetrikaAvailable(true)
                                        .withRetargetingCondition(r))
                                .toMap();
            } catch (MetrikaClientException | InterruptedRuntimeException e) {
                logger.warn("Got an exception when querying for metrika goals for clientId: " +
                        queriedClient.getId(), e);
                retargetingConditionById = StreamEx.of(retargetingConditionIds)
                        .mapToEntry(retargetingConditionId -> new GdRetargetingConditionWithMetrikaFlag()
                                .withRetargetingCondition(null)
                                .withIsMetrikaAvailable(false))
                        .toMap();
            }

            return CompletableFuture.completedFuture(retargetingConditionById);
        };
    }
}
