package ru.yandex.direct.grid.processing.service.showcondition.retargeting;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.AggregatedStatusRetargetingData;
import ru.yandex.direct.core.entity.campaign.model.CampaignExperiment;
import ru.yandex.direct.core.entity.crypta.service.CryptaSuggestService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.ExperimentRetargetingConditions;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionOperationFactory;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingConditionExperimentsValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.bidmodifiers.service.GridBidModifiersService;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.RetargetingFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingBaseStatus;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingCondition;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionFilter;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionType;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingFilter;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiBidsRetargeting;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiBidsRetargetingWithTotals;
import ru.yandex.direct.grid.core.entity.showcondition.service.GridRetargetingConditionService;
import ru.yandex.direct.grid.core.entity.showcondition.service.GridRetargetingService;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdRetargetingAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.utils.GridModerationUtils;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.common.GdResult;
import ru.yandex.direct.grid.processing.model.goal.GdGoalTruncated;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.retargeting.GdCaRetargetingCondition;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargeting;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingCondition;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleItem;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionType;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingWithTotals;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditionForExperiments;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditionForExperimentsPayload;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditionForExperimentsPayloadItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditionPayloadItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditions;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdDeleteRetargetingConditions;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdDeleteRetargetingConditionsPayloadItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdDeleteRetargetings;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdDeleteRetargetingsPayloadItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdResumeRetargetings;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdResumeRetargetingsPayloadItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdSuspendRetargetings;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdSuspendRetargetingsPayloadItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateRetargetingConditionPayloadItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateRetargetingConditions;
import ru.yandex.direct.grid.processing.service.aggregatedstatuses.RetargetingBaseStatusCalculator;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.goal.CryptaGoalsConverter;
import ru.yandex.direct.grid.processing.service.group.GroupDataService;
import ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter;
import ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingMutationConverter;
import ru.yandex.direct.grid.processing.service.showcondition.validation.RetargetingValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.util.ReasonsFilterUtils;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.emptyIfNull;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.OBSOLETE_SHORTCUT_NAME_TO_CURRENT_NAME;
import static ru.yandex.direct.feature.FeatureName.HIDE_OLD_SHOW_CAMPS_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.SHOW_AGGREGATED_STATUS_OPEN_BETA;
import static ru.yandex.direct.feature.FeatureName.SHOW_DNA_BY_DEFAULT;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toCoreRetargetingCondition;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toGdRetargeting;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toGdRetargetingCondition;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toInternalRetargetingConditionFilter;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toInternalRetargetingFilter;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toInternalRetargetingOrderBy;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter.toInternalTypeSet;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingMutationConverter.convertToGdResult;
import static ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingMutationConverter.toGdCreateRetargetingConditionForExperimentsPayloadItem;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@SuppressWarnings("ALL")
@Service
@ParametersAreNonnullByDefault
public class RetargetingDataService {
    public static final int EXPERIMENT_RETARGETING_CONDITION_NUMBER = 2;
    private final CampaignInfoService campaignInfoService;
    private final GridBidModifiersService gridBidModifiersService;
    private final GridRetargetingConditionService gridRetargetingConditionService;
    private final GridValidationService gridValidationService;
    private final GridRetargetingService gridRetargetingService;
    private final GroupDataService groupDataService;
    private final RetargetingConditionOperationFactory retargetingConditionOperationFactory;
    private final RetargetingConditionService retargetingConditionService;
    private final RetargetingService retargetingService;
    private final RetargetingValidationService retargetingValidationService;
    private final RetargetingConditionExperimentsValidationService retargetingConditionExperimentsValidationService;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;
    private final CryptaSuggestService cryptaSuggestService;
    private final FeatureService featureService;
    private final CryptaGoalsConverter cryptaGoalsConverter;

    @Autowired
    RetargetingDataService(GridRetargetingConditionService gridRetargetingConditionService,
                           GridValidationService gridValidationService,
                           GridRetargetingService gridRetargetingService,
                           GroupDataService groupDataService,
                           GridBidModifiersService gridBidModifiersService,
                           CampaignInfoService campaignInfoService,
                           RetargetingConditionOperationFactory retargetingConditionOperationFactory,
                           RetargetingConditionService retargetingConditionService,
                           RetargetingService retargetingService,
                           RetargetingValidationService retargetingValidationService,
                           RetargetingConditionExperimentsValidationService retargetingConditionExperimentsValidationService,
                           AggregatedStatusesViewService aggregatedStatusesViewService,
                           CryptaSuggestService cryptaSuggestService,
                           FeatureService featureService,
                           CryptaGoalsConverter cryptaGoalsConverter) {
        this.gridRetargetingConditionService = gridRetargetingConditionService;
        this.gridValidationService = gridValidationService;
        this.gridRetargetingService = gridRetargetingService;
        this.gridBidModifiersService = gridBidModifiersService;
        this.groupDataService = groupDataService;
        this.campaignInfoService = campaignInfoService;
        this.retargetingConditionOperationFactory = retargetingConditionOperationFactory;
        this.retargetingConditionService = retargetingConditionService;
        this.retargetingService = retargetingService;
        this.retargetingConditionExperimentsValidationService = retargetingConditionExperimentsValidationService;
        this.retargetingValidationService = retargetingValidationService;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.cryptaSuggestService = cryptaSuggestService;
        this.featureService = featureService;
        this.cryptaGoalsConverter = cryptaGoalsConverter;
    }

    /**
     * Возвращение условий ретаргетинга клиента
     *
     * @param clientInfo информация о клиенте
     * @param filter     критерий отбора условий ретаргетинга
     * @return {@link List} условий ретаргетинга
     */
    public List<GdRetargetingCondition> getRetargetingConditionsRowset(GdClientInfo clientInfo,
                                                                       GdRetargetingConditionFilter filter) {
        return getRetargetingConditionsRowset(clientInfo, filter, null);
    }


    /**
     * Возвращение условий ретаргетинга клиента (в том числе сохраненных шорткатов)
     *
     * @param clientInfo         информация о клиенте
     * @param filter             критерий отбора условий ретаргетинга
     * @param shortcutCampaignId кампания, сохраненные шорткаты которой нужно вернуть. Если null, не возвращаем шорткаты
     * @return {@link List} условий ретаргетинга
     */
    private List<GdRetargetingCondition> getRetargetingConditionsRowset(GdClientInfo clientInfo,
                                                                        GdRetargetingConditionFilter filter,
                                                                        @Nullable Long shortcutCampaignId) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        int shard = clientInfo.getShard();

        List<GdiRetargetingCondition> retargetingConditions = getInternalRetargetingConditions(clientInfo, filter, shortcutCampaignId);
        Set<Long> campaignIds = flatMapToSet(retargetingConditions, GdiRetargetingCondition::getCampaignIds);

        Set<Long> retargetingConditionIds = listToSet(retargetingConditions,
                GdiRetargetingCondition::getRetargetingConditionId);

        Map<Long, List<Long>> campaignIdsWithRetargetingBidModifierByConditionId = gridBidModifiersService
                .getCampaignIdsWithRetargetingBidModifiersByConditionId(shard, retargetingConditionIds);
        campaignIds.addAll(flatMapToSet(campaignIdsWithRetargetingBidModifierByConditionId.values(), identity()));

        Map<Long, GdCampaignTruncated> campaignsInfo = campaignInfoService.getTruncatedCampaigns(clientId, campaignIds);

        List<GdRetargetingCondition> gdRetargetingConditions =
                mapList(retargetingConditions,
                    condition -> toGdRetargetingCondition(condition, campaignsInfo,
                            campaignIdsWithRetargetingBidModifierByConditionId));

        if (featureService.isEnabledForClientId(clientId, FeatureName.NEW_CUSTOM_AUDIENCE_ENABLED)) {
            enhanceGoalsForCustomAudience(gdRetargetingConditions);
        }

        return gdRetargetingConditions;
    }

    private void enhanceGoalsForCustomAudience(List<GdRetargetingCondition> gdRetargetingConditions) {
        List<GdGoalTruncated> interestGoals = collectGoalsIdsForRetargingConditions(
                filterList(
                        gdRetargetingConditions,
                        condition -> condition.getType() == GdRetargetingConditionType.INTERESTS
                )
        );

        Map<Long, Goal> enhancedGoalsForCustomAudience = cryptaSuggestService.getGoalsByIds(
                mapList(interestGoals, GdGoalTruncated::getId));

        Map<Long, GdGoalTruncated> translatedGoalsForCustomAudience = listToMap(
                cryptaGoalsConverter.convertToTranslatedGdGoal(enhancedGoalsForCustomAudience.values()),
                GdGoalTruncated::getId
        );

        interestGoals.forEach(
                goal -> {
                        GdGoalTruncated modifiedGoal = translatedGoalsForCustomAudience.get(goal.getId());
                        if (modifiedGoal != null) {
                                goal.setName(modifiedGoal.getName());
                                goal.setDescription(modifiedGoal.getDescription());
                                goal.setAudienceTypeName(modifiedGoal.getAudienceTypeName());
                        }
                }
        );
    }

    private List<GdGoalTruncated> collectGoalsIdsForRetargingConditions(
            List<GdRetargetingCondition> retargetingConditions) {
        return retargetingConditions.stream()
                .map(GdRetargetingCondition::getConditionRules)
                .flatMap(List::stream)
                .map(GdRetargetingConditionRuleItem::getGoals)
                .flatMap(List::stream)
                .collect(Collectors.toList());
    }

    public List<GdCaRetargetingCondition> getCaRetargetingConditions(GdClientInfo clientInfo,
                                                                     GdRetargetingConditionFilter filter) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        var isCustomAudienceEnabled = featureService.isEnabledForClientId(clientId, FeatureName.CUSTOM_AUDIENCE_ENABLED);
        var isNotInternalAd = StringUtils.isEmpty(clientInfo.getInternalAdProductName());
        List<GdiRetargetingCondition> retargetingConditions = List.of();

        if (isCustomAudienceEnabled && isNotInternalAd) {
            retargetingConditions = getInternalRetargetingConditions(clientInfo, filter, null);
        }

        return StreamEx.of(retargetingConditions)
                .map(this::toCaRetargetigCondition)
                .nonNull()
                .toList();
    }

    private GdCaRetargetingCondition toCaRetargetigCondition(@Nullable GdiRetargetingCondition retargetingCondition) {
        if (retargetingCondition == null || CollectionUtils.isEmpty(retargetingCondition.getConditionRules())) {
            return null;
        }
        var rule = retargetingCondition.getConditionRules().get(0);
        var suggestGoals = cryptaSuggestService.getCryptaSuggestFromGoals(rule.getGoals());
        return new GdCaRetargetingCondition()
                .withAdGroupIds(retargetingCondition.getAdGroupIds())
                .withRetargetingConditionId(retargetingCondition.getRetargetingConditionId())
                .withConditionRules(mapList(suggestGoals, CryptaGoalsConverter::toGdCryptaGoalsSuggestItem));
    }

    private List<GdiRetargetingCondition> getInternalRetargetingConditions(GdClientInfo clientInfo,
                                                                           GdRetargetingConditionFilter filter,
                                                                           @Nullable Long shortcutCampaignId) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        int shard = clientInfo.getShard();

        GdiRetargetingConditionFilter internalFilter = toInternalRetargetingConditionFilter(clientId.asLong(), filter);

        List<GdiRetargetingCondition> retargetingConditions = gridRetargetingConditionService
                .getRetargetingConditions(shard, clientId, internalFilter);

        // скрываем ab-эксперименты, если на кампании скрытый Brand-Lift
        List<GdiBaseCampaign> allCampaigns = campaignInfoService.getAllBaseCampaigns(clientId);
        Set<Long> abRetargetingConditions = allCampaigns.stream()
                .filter(GdiBaseCampaign::getIsBrandLiftHidden)
                .flatMap(gdiCampaign -> Stream.of(
                        gdiCampaign.getAbSegmentRetargetingConditionId(),
                        gdiCampaign.getAbSegmentStatisticRetargetingConditionId()))
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        return retargetingConditions.stream()
                .filter(retCond -> !abRetargetingConditions.contains(retCond.getRetargetingConditionId())
                        && (retCond.getType() != GdiRetargetingConditionType.SHORTCUTS
                        || (shortcutCampaignId != null && retCond.getCampaignIds().contains(shortcutCampaignId))))
                .collect(Collectors.toList());
    }

    /**
     * Возвращение условий ретаргетинга клиента (в том числе сохраненных шорткатов)
     *
     * @param clientInfo информация о клиенте
     * @param filter     критерий отбора условий ретаргетинга
     * @param campaignId id кампании, для которой нужно получить доступные шорткаты
     * @return {@link List} условий ретаргетинга
     */
    public List<GdRetargetingCondition> getRetargetingConditionsAndAvailableShortcutsRowset(GdClientInfo clientInfo,
                                                                                            GdRetargetingConditionFilter filter,
                                                                                            @Nullable Long campaignId) {
        ClientId clientId = ClientId.fromLong(clientInfo.getId());
        List<GdRetargetingCondition> retargetingConditionsInUse = getRetargetingConditionsRowset(clientInfo, filter,
                campaignId);

        Set<GdiRetargetingConditionType> typeSet = toInternalTypeSet(filter.getRetargetingConditionTypeIn());
        var areAvailableShortcutsNeeded = typeSet.contains(GdiRetargetingConditionType.SHORTCUTS);

        // отделяем используемые шорткаты от остальных условий, чтобы перенести шорткаты в начало списка
        Map<Boolean, List<GdRetargetingCondition>> conditionsByHavingShortcutType = retargetingConditionsInUse.stream()
                .collect(Collectors.partitioningBy(rc -> rc.getType() == GdRetargetingConditionType.SHORTCUTS));
        List<GdRetargetingCondition> shortcutsInUse = conditionsByHavingShortcutType.get(true);
        List<GdRetargetingCondition> otherRetargetingConditionsInUse = conditionsByHavingShortcutType.get(false);

        // добавляем доступные (и еще неиспользуемые пользователем) шорткаты
        if (areAvailableShortcutsNeeded) {
            List<RetargetingCondition> availableShortcuts =
                    retargetingConditionService.getTruncatedRetargetingConditionShortcuts(clientId, campaignId);
            List<GdRetargetingCondition> result =
                    new ArrayList<>(retargetingConditionsInUse.size() + availableShortcuts.size());
            result.addAll(mapList(availableShortcuts, RetargetingConverter::toGdRetargetingCondition));

            // Подменяем дефолтные шорткаты их сохраненными версиями, если они есть
            // (отличаются поля retargetingConditionId, adGroupIds и campaigns; для сохраненных заполнены rules).
            // У сохраненных шорткатов могут быть устаревшие имена, в таком случае сперва получаем текущую версию имени.

            //TODO на некоторых кампаниях есть дубли шорткатов, комментирую для фикса бага DIRECT-164020
            /*var shortcutsInUseByName = listToMap(shortcutsInUse, shortcutInUse -> {
                return OBSOLETE_SHORTCUT_NAME_TO_CURRENT_NAME.containsKey(shortcutInUse.getName())
                        ? OBSOLETE_SHORTCUT_NAME_TO_CURRENT_NAME.get(shortcutInUse.getName())
                        : shortcutInUse.getName();
            });*/
            var shortcutsInUseByName = new HashMap<String, GdRetargetingCondition>(shortcutsInUse.size());
            for (var shortcutInUse : shortcutsInUse) {
                var shortcutName = OBSOLETE_SHORTCUT_NAME_TO_CURRENT_NAME.containsKey(shortcutInUse.getName())
                        ? OBSOLETE_SHORTCUT_NAME_TO_CURRENT_NAME.get(shortcutInUse.getName())
                        : shortcutInUse.getName();
                // Берем последний из дублей
                shortcutsInUseByName.put(shortcutName, shortcutInUse);
            }

            for (var i = 0; i < result.size(); i++) {
                var availableShortcutName = result.get(i).getName();
                if (shortcutsInUseByName.containsKey(availableShortcutName)) {
                    result.set(i, shortcutsInUseByName.get(availableShortcutName));
                    shortcutsInUseByName.remove(availableShortcutName);
                }
            }

            // Если среди используемых пользователем шорткатов есть устаревшие (== нет среди доступных),
            // то возвращаем их в конце списка шорткатов.
            if (!shortcutsInUseByName.isEmpty()) {
                result.addAll(shortcutsInUseByName.values());
            }

            result.addAll(otherRetargetingConditionsInUse);
            return result;
        } else {
            shortcutsInUse.addAll(otherRetargetingConditionsInUse);
            return shortcutsInUse;
        }
    }


    /**
     * Получить данные о ретаргетингах. Фильтруем и сортируем в YT, достаем вместе со статистикой.
     * <p>
     * При добавлении новых фильтров в коде нужно так же их учесть в методе:
     * {@link ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter#hasAnyCodeFilter}
     */
    public GdRetargetingWithTotals getRetargetingsRowset(GridGraphQLContext context, GdRetargetingsContainer input) {
        GdClientInfo client = context.getQueriedClient();
        ClientId clientId = ClientId.fromLong(client.getId());

        GdStatRequirements statRequirements =
                normalizeStatRequirements(input.getStatRequirements(), context.getInstant(), null);
        input.setStatRequirements(statRequirements);

        RetargetingFetchedFieldsResolver fetchingFieldsResolver = context.getFetchedFieldsReslover().getRetargeting();
        ClientId operatorClientId = context.getOperator().getClientId();
        boolean showAggregatedStatusesDebug = featureService
                .isEnabledForClientId(operatorClientId, FeatureName.SHOW_AGGREGATED_STATUS_DEBUG);

        GdiRetargetingFilter internalFilter = toInternalRetargetingFilter(input.getFilter());
        boolean withFilterByAggrStatus = featureService.isEnabledForClientId(operatorClientId, SHOW_DNA_BY_DEFAULT)
                || featureService.isEnabledForClientId(operatorClientId, HIDE_OLD_SHOW_CAMPS_FOR_DNA)
                || featureService.isEnabledForClientId(operatorClientId, SHOW_AGGREGATED_STATUS_OPEN_BETA);
        boolean addWithTotalsToQuery = featureService.isEnabledForClientId(operatorClientId,
                FeatureName.ADD_WITH_TOTALS_TO_BIDS_RETARGETING_QUERY);

        GdiBidsRetargetingWithTotals gdiBidsRetargetingWithTotals = gridRetargetingService.getRetargetings(clientId,
                internalFilter, fetchingFieldsResolver,
                input.getStatRequirements().getFrom(), input.getStatRequirements().getTo(),
                toInternalRetargetingOrderBy(input.getOrderBy()), withFilterByAggrStatus, addWithTotalsToQuery);
        List<GdiBidsRetargeting> retargetings = gdiBidsRetargetingWithTotals.getGdiBidsRetargetings();
        GdEntityStats gdEntityTotalStats = gdiBidsRetargetingWithTotals.getTotalStats() != null
                ? StatHelper.convertInternalStatsToOuter(gdiBidsRetargetingWithTotals.getTotalStats())
                : null;
        if (retargetings.isEmpty()) {
            return new GdRetargetingWithTotals()
                    .withGdRetargetings(emptyList())
                    .withTotalStats(gdEntityTotalStats);
        }

        var retargetingStatusesByIds = aggregatedStatusesViewService.getRetargetingStatusesByIds(client.getShard(),
                listToSet(retargetings, GdiBidsRetargeting::getRetargetingId));
        Map<Long, String> statusesJsonByIds = showAggregatedStatusesDebug
                ? aggregatedStatusesViewService.statusesToJsonString(retargetingStatusesByIds)
                : emptyMap();

        if (withFilterByAggrStatus) {
            retargetings = filterByAggregatedStatuses(retargetingStatusesByIds, retargetings, internalFilter);
            if (retargetings.isEmpty()) {
                return new GdRetargetingWithTotals()
                        .withGdRetargetings(emptyList())
                        .withTotalStats(gdEntityTotalStats);
            }
        }

        for (var retargeting : retargetings) {
            var retargetingId = retargeting.getRetargetingId();
            var statusData = retargetingStatusesByIds.get(retargetingId);
            if (statusData != null && statusData.getStatus().isPresent()) {
                retargeting.setAggregatedStatusInfo(
                        new GdRetargetingAggregatedStatusInfo()
                                .withStatus(statusData.getStatus().get())
                                .withReasons(statusData.getReasons())
                                .withRejectReasons(GridModerationUtils.toGdRejectReasons(statusData.getRejectReasons()))
                                .withIsObsolete(statusData.getIsObsolete()));
            }
            if (showAggregatedStatusesDebug) {
                retargeting.setAggregatedStatus(statusesJsonByIds
                        .getOrDefault(retargetingId, "{\"status\" : \"no data\"}"));
            }
        }

        Set<Long> campaignIds = listToSet(retargetings, GdiBidsRetargeting::getCampaignId);
        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService.getTruncatedCampaigns(clientId, campaignIds);
        List<GdiBidsRetargeting> availableRetargetings =
                filterList(retargetings, retargeting -> campaignsById.containsKey(retargeting.getCampaignId()));

        Set<Long> adGroupIds = listToSet(availableRetargetings, GdiBidsRetargeting::getAdGroupId);
        List<GdAdGroupTruncated> adGroups =
                groupDataService.getTruncatedAdGroups(client.getShard(), client.getCountryRegionId(), clientId,
                        context.getOperator(), adGroupIds, campaignsById);
        Map<Long, GdAdGroupTruncated> adGroupsById = listToMap(adGroups, GdAdGroupTruncated::getId);
        availableRetargetings = filterList(availableRetargetings,
                retargeting -> adGroupsById.containsKey(retargeting.getAdGroupId()));

        boolean cpcAndCpmOnOneGridEnabled = featureService
                .isEnabledForClientId(operatorClientId, FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED);

        List<GdRetargeting> gdRetargetings = mapList(availableRetargetings, retargeting -> toGdRetargeting(retargeting,
                campaignsById.get(retargeting.getCampaignId()),
                adGroupsById.get(retargeting.getAdGroupId()), cpcAndCpmOnOneGridEnabled));
        return new GdRetargetingWithTotals()
                .withGdRetargetings(gdRetargetings)
                .withTotalStats(gdEntityTotalStats);
    }


    public GdResult<GdCreateRetargetingConditionPayloadItem> createRetargetingConditions(GridGraphQLContext context,
                                                                                         GdCreateRetargetingConditions input) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        ClientId clientId = context.getSubjectUser().getClientId();
        retargetingValidationService.validateGdCreateRetargetingConditions(input);

        Path path = path(field(GdCreateRetargetingConditions.RETARGETING_CONDITIONS));

        List<RetargetingCondition> retargetingConditions =
                mapList(input.getRetargetingConditions(), condition -> toCoreRetargetingCondition(clientId, condition));

        MassResult<Long> result = retargetingConditionService.addRetargetingConditions(retargetingConditions, clientId);

        GdValidationResult validationResult = gridValidationService.getValidationResult(result, path);

        return convertToGdResult(input.getRetargetingConditions().size(), result, validationResult,
                RetargetingMutationConverter::toCreateRetargetingConditionRowset);
    }

    /**
     * Работает не частично.
     * Создаем условия ретаргетингов для экспериментов.
     * Если подходящие условия уже есть отдаем существующие.
     */
    @Deprecated
    public GdCreateRetargetingConditionForExperimentsPayload findOrCreateExperimentsRetargetingConditions(
            ClientId clientId,
            GdCreateRetargetingConditionForExperiments input) {

        List<CampaignExperiment> campaignExperiments = mapList(input.getItems(), item -> new CampaignExperiment()
                .withAbSegmentGoalIds(item.getAbSegmentGoalIds())
                .withSectionIds(item.getSectionIds()));

        ValidationResult<List<CampaignExperiment>, Defect> vr =
                retargetingConditionExperimentsValidationService.validateRetargetingConditionsForExperiments(clientId,
                        campaignExperiments);

        if (vr.hasAnyErrors()) {
            return new GdCreateRetargetingConditionForExperimentsPayload()
                    .withValidationResult(gridValidationService.toGdValidationResult(vr,
                            path(field(GdCreateRetargetingConditionForExperiments.ITEMS))))
                    .withAddedConditions(null);
        }

        List<ExperimentRetargetingConditions> experimentsRetargetingConditions =
                retargetingConditionService.findOrCreateExperimentsRetargetingConditions(clientId, campaignExperiments);

        List<GdCreateRetargetingConditionForExperimentsPayloadItem> result = EntryStream.of(campaignExperiments)
                .keys()
                .map(index -> toGdCreateRetargetingConditionForExperimentsPayloadItem(
                        experimentsRetargetingConditions.get(index).getStatisticRetargetingConditionId(),
                        experimentsRetargetingConditions.get(index).getRetargetingConditionId())
                )
                .toList();

        return new GdCreateRetargetingConditionForExperimentsPayload()
                .withValidationResult(gridValidationService.toGdValidationResult(vr, emptyPath()))
                .withAddedConditions(result);
    }

    public GdResult<GdUpdateRetargetingConditionPayloadItem> updateRetargetingConditions(GridGraphQLContext context,
                                                                                         GdUpdateRetargetingConditions input) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        ClientId clientId = context.getSubjectUser().getClientId();
        retargetingValidationService.validateGdUpdateRetargetingConditions(input);

        Path path = path(field(GdUpdateRetargetingConditions.RETARGETING_CONDITIONS));

        List<ModelChanges<RetargetingCondition>> modelChanges = mapList(input.getRetargetingConditions(),
                RetargetingConverter::toRetargetingConditionModelChanges);

        MassResult<Long> result =
                retargetingConditionOperationFactory.updateRetargetingConditions(clientId, modelChanges);

        GdValidationResult validationResult = gridValidationService.getValidationResult(result, path);

        return convertToGdResult(input.getRetargetingConditions().size(), result, validationResult,
                RetargetingMutationConverter::toUpdateRetargetingConditionRowset);
    }

    public GdResult<GdDeleteRetargetingConditionsPayloadItem> deleteRetargetingConditions(GridGraphQLContext context,
                                                                                          GdDeleteRetargetingConditions input) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        ClientId clientId = context.getSubjectUser().getClientId();
        retargetingValidationService.validateGdDeleteRetargetingConditions(input);

        MassResult<Long> operationResult =
                retargetingConditionOperationFactory
                        .deleteRetargetingConditions(input.getRetargetingConditionIds(), clientId, Applicability.FULL);

        Path path = path(field(GdDeleteRetargetingConditions.RETARGETING_CONDITION_IDS));
        GdValidationResult validationResult = gridValidationService.getValidationResult(operationResult, path);

        return convertToGdResult(input.getRetargetingConditionIds().size(), operationResult, validationResult,
                RetargetingMutationConverter::toDeleteRetargetingConditionRowset);
    }

    public GdResult<GdSuspendRetargetingsPayloadItem> suspendRetargetings(GridGraphQLContext context,
                                                                          GdSuspendRetargetings input) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        ClientId clientId = context.getSubjectUser().getClientId();
        retargetingValidationService.validateGdSuspendRetargetings(input);

        MassResult<Long> result = retargetingService
                .suspendRetargetings(input.getRetargetingIds(), clientId, context.getOperator().getUid());

        Path path = path(field(GdSuspendRetargetings.RETARGETING_IDS));
        GdValidationResult validationResult = gridValidationService.getValidationResult(result, path);

        return convertToGdResult(input.getRetargetingIds().size(), result, validationResult,
                RetargetingMutationConverter::toSuspendRetargetingRowset);
    }

    public GdResult<GdResumeRetargetingsPayloadItem> resumeRetargetings(GridGraphQLContext context,
                                                                        GdResumeRetargetings input) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        ClientId clientId = context.getSubjectUser().getClientId();
        retargetingValidationService.validateGdResumeRetargetings(input);

        MassResult<Long> result = retargetingService
                .resumeRetargetings(input.getRetargetingIds(), clientId, context.getOperator().getUid());

        Path path = path(field(GdResumeRetargetings.RETARGETING_IDS));
        GdValidationResult validationResult = gridValidationService.getValidationResult(result, path);

        return convertToGdResult(input.getRetargetingIds().size(), result, validationResult,
                RetargetingMutationConverter::toResumeRetargetingRowset);
    }

    public GdResult<GdDeleteRetargetingsPayloadItem> deleteRetargetings(GridGraphQLContext context,
                                                                        GdDeleteRetargetings input) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        ClientId clientId = context.getSubjectUser().getClientId();
        retargetingValidationService.validateGdDeleteRetargetings(input);

        MassResult<Long> result = retargetingService
                .deleteRetargetings(input.getRetargetingIds(), clientId, context.getOperator().getUid());

        Path path = path(field(GdResumeRetargetings.RETARGETING_IDS));
        GdValidationResult validationResult = gridValidationService.getValidationResult(result, path);

        return convertToGdResult(input.getRetargetingIds().size(), result, validationResult,
                RetargetingMutationConverter::toDeleteRetargetingRowset);
    }

    private List<GdiBidsRetargeting> filterByAggregatedStatuses(Map<Long, AggregatedStatusRetargetingData> retargetingStatusesByIds,
                                                                List<GdiBidsRetargeting> retargetings,
                                                                GdiRetargetingFilter filter) {
        Set<GdiRetargetingBaseStatus> statuses = new HashSet<>(emptyIfNull(filter.getStatusIn()));
        return retargetings.stream()
                .filter(r -> statuses.isEmpty()
                        || statuses.contains(RetargetingBaseStatusCalculator
                        .convertToRetargetingBaseStatus(retargetingStatusesByIds.get(r.getRetargetingId()))))
                .filter(r -> ReasonsFilterUtils.isValid(filter.getReasonsContainSome(),
                        retargetingStatusesByIds.get(r.getRetargetingId())))
                .collect(Collectors.toList());
    }
}
