package ru.yandex.direct.grid.processing.service.showcondition.validation;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleItemReq;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionsAndAvailableShortcutsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingFilter;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingsContainer;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditionItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdCreateRetargetingConditions;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdDeleteRetargetingConditions;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdDeleteRetargetings;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdEstimateRetargetingConditionContainer;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdResumeRetargetings;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdSuspendRetargetings;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateCpmRetargetingConditionItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateRetargetingConditionItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateRetargetingConditions;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.IDS_COLLECTION_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.LIMIT_OFFSET_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.STATS_FILTER_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.STAT_REQUIREMENTS_VALIDATOR;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Service
@ParametersAreNonnullByDefault
public class RetargetingValidationService {

    private GridValidationService gridValidationService;
    private final PathNodeConverterProvider pathNodeConverterProvider;

    @Autowired
    RetargetingValidationService(GridValidationService gridValidationService) {

        this.gridValidationService = gridValidationService;

        pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                //.register() //todo: но это еще не точно. посмотреть что тут за ошибки.
                .build();
    }

    public void validateGdRetargetingsContainer(GdRetargetingsContainer input) {
        gridValidationService.applyValidator(RETARGETINGS_CONTAINER_WITH_STATS_VALIDATOR, input, false);
    }

    public void validateGdRetargetingsContainerForSearch(GdRetargetingsContainer input) {
        gridValidationService.applyValidator(RETARGETINGS_CONTAINER_VALIDATOR_GENERAL, input, false);
    }

    public void validateGdRetargetingConditionsContainer(GdRetargetingConditionsContainer input) {
        //todo:
        gridValidationService.applyValidator(RETARGETING_CONDITIONS_CONTAINER_VALIDATOR, input, false);

    }

    public void validateGdRetargetingConditionsAndAvailableShortcutsContainer(
            GdRetargetingConditionsAndAvailableShortcutsContainer input) {
        gridValidationService.applyValidator(RETARGETING_CONDITIONS_AND_AVAILABLE_SHORTCUTS_VALIDATOR, input, false);

    }

    public void validateGdCreateRetargetingConditions(GdCreateRetargetingConditions input) {
        gridValidationService.applyValidator(CREATE_RETARGETING_CONDITIONS_INPUT_VALIDATOR, input, false);
    }

    public void validateGdUpdateRetargetingConditions(GdUpdateRetargetingConditions input) {
        gridValidationService.applyValidator(UPDATE_RETARGETING_CONDITIONS_INPUT_VALIDATOR, input, false);
    }

    public void validateGdDeleteRetargetingConditions(GdDeleteRetargetingConditions input) {
        gridValidationService.applyValidator(DELETE_RETARGETING_CONDITIONS_INPUT_VALIDATOR, input, false);
    }

    public void validateGdSuspendRetargetings(GdSuspendRetargetings input) {
        gridValidationService.applyValidator(SUSPEND_RETARGETING_INPUT_VALIDATOR, input, false);
    }

    public void validateGdResumeRetargetings(GdResumeRetargetings input) {
        gridValidationService.applyValidator(RESUME_RETARGETING_INPUT_VALIDATOR, input, false);
    }

    public void validateEstimateRetargetingCondition(GdEstimateRetargetingConditionContainer input) {
        gridValidationService.applyValidator(ESTIMATE_RETARGETING_CONDITION_INPUT_VALIDATOR, input, false);
    }

    public void validateGdDeleteRetargetings(GdDeleteRetargetings input) {
        gridValidationService.applyValidator(DELETE_RETARGETING_INPUT_VALIDATOR, input, false);
    }

    private static final Validator<GdRetargetingFilter, Defect> RETARGETING_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdRetargetingFilter> vb = ModelItemValidationBuilder.of(filter);

        vb.item(GdRetargetingFilter.RETARGETING_CONDITION_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdRetargetingFilter.RETARGETING_CONDITION_ID_NOT_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdRetargetingFilter.RETARGETING_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdRetargetingFilter.RETARGETING_ID_NOT_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdRetargetingFilter.CAMPAIGN_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdRetargetingFilter.AD_GROUP_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdRetargetingFilter.STATS)
                .checkBy(STATS_FILTER_VALIDATOR, When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdRetargetingsContainer, Defect> RETARGETINGS_CONTAINER_VALIDATOR_GENERAL = input ->
    {
        ModelItemValidationBuilder<GdRetargetingsContainer> vb = ModelItemValidationBuilder.of(input);

        vb.item(GdRetargetingsContainer.FILTER)
                .check(notNull(), When.isTrue(input.getFilterKey() == null));
        vb.item(GdRetargetingsContainer.FILTER_KEY)
                .check(notNull(), When.isTrue(input.getFilter() == null));

        vb.item(GdRetargetingsContainer.FILTER)
                .checkBy(RETARGETING_FILTER_VALIDATOR, When.notNull());
        vb.item(GdRetargetingsContainer.FILTER_KEY)
                .check(notBlank(), When.notNull());

        vb.item(GdRetargetingsContainer.LIMIT_OFFSET)
                .checkBy(LIMIT_OFFSET_VALIDATOR);
        vb.item(GdRetargetingsContainer.CACHE_KEY)
                .check(notBlank(), When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdRetargetingsContainer, Defect> RETARGETINGS_CONTAINER_WITH_STATS_VALIDATOR =
            input -> {
                ModelItemValidationBuilder<GdRetargetingsContainer> vb = ModelItemValidationBuilder.of(input);

                vb.checkBy(RETARGETINGS_CONTAINER_VALIDATOR_GENERAL);

                vb.item(GdRetargetingsContainer.STAT_REQUIREMENTS)
                        .checkBy(STAT_REQUIREMENTS_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdRetargetingConditionFilter, Defect> RETARGETING_CONDITION_FILTER_VALIDATOR =
            filter -> {
                ModelItemValidationBuilder<GdRetargetingConditionFilter> vb = ModelItemValidationBuilder.of(filter);

                vb.item(GdRetargetingConditionFilter.RETARGETING_CONDITION_ID_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
                vb.item(GdRetargetingConditionFilter.RETARGETING_CONDITION_ID_NOT_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
                vb.item(GdRetargetingConditionFilter.RETARGETING_CONDITION_TYPE_IN)
                        .check(notEmptyCollection());

                return vb.getResult();
            };

    private static final Validator<GdRetargetingConditionsContainer, Defect>
            RETARGETING_CONDITIONS_CONTAINER_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdRetargetingConditionsContainer> vb = ModelItemValidationBuilder.of(input);

        vb.item(GdRetargetingConditionsContainer.FILTER)
                .checkBy(RETARGETING_CONDITION_FILTER_VALIDATOR);
        return vb.getResult();
    };

    private static final Validator<GdRetargetingConditionsAndAvailableShortcutsContainer, Defect>
            RETARGETING_CONDITIONS_AND_AVAILABLE_SHORTCUTS_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdRetargetingConditionsAndAvailableShortcutsContainer> vb =
                ModelItemValidationBuilder.of(input);

        vb.item(GdRetargetingConditionsAndAvailableShortcutsContainer.FILTER)
                .checkBy(RETARGETING_CONDITION_FILTER_VALIDATOR);
        return vb.getResult();
    };

    private static final Validator<GdRetargetingConditionRuleItemReq, Defect>
            RETARGETING_CONDITION_RULE_ITEM_VALIDATOR = retargetingConditionRule -> {

        ModelItemValidationBuilder<GdRetargetingConditionRuleItemReq> vb =
                ModelItemValidationBuilder.of(retargetingConditionRule);

        vb.list(GdRetargetingConditionRuleItemReq.GOALS)
                .check(notEmptyCollection());

        return vb.getResult();
    };

    private static final Validator<GdEstimateRetargetingConditionContainer, Defect>
            ESTIMATE_RETARGETING_CONDITION_INPUT_VALIDATOR = input -> {

        ModelItemValidationBuilder<GdEstimateRetargetingConditionContainer> vb = ModelItemValidationBuilder.of(input);

        vb.list(GdEstimateRetargetingConditionContainer.CONDITION_RULES)
                .check(notEmptyCollection())
                .checkEachBy(RETARGETING_CONDITION_RULE_ITEM_VALIDATOR);

        return vb.getResult();
    };

    private static final Validator<GdCreateRetargetingConditionItem, Defect>
            CREATE_RETARGETING_CONDITION_ITEM_VALIDATOR = retargetingConditionItem -> {

        ModelItemValidationBuilder<GdCreateRetargetingConditionItem> vb =
                ModelItemValidationBuilder.of(retargetingConditionItem);

        vb.list(GdCreateRetargetingConditionItem.CONDITION_RULES)
                .checkEachBy(RETARGETING_CONDITION_RULE_ITEM_VALIDATOR);

        return vb.getResult();
    };

    public static final Validator<GdUpdateCpmRetargetingConditionItem, Defect>
            UPDATE_CPM_RETARGETING_CONDITION_ITEM_VALIDATOR = retargetingConditionItem -> {

        ModelItemValidationBuilder<GdUpdateCpmRetargetingConditionItem> vb =
                ModelItemValidationBuilder.of(retargetingConditionItem);

        vb.item(GdUpdateCpmRetargetingConditionItem.NAME)
                .check(notBlank());

        vb.list(GdUpdateCpmRetargetingConditionItem.CONDITION_RULES)
                .checkEachBy(RETARGETING_CONDITION_RULE_ITEM_VALIDATOR);

        return vb.getResult();
    };

    private static final Validator<GdCreateRetargetingConditions, Defect>
            CREATE_RETARGETING_CONDITIONS_INPUT_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdCreateRetargetingConditions> vb = ModelItemValidationBuilder.of(input);

        vb.list(GdCreateRetargetingConditions.RETARGETING_CONDITIONS)
                .check(notEmptyCollection())
                .checkEachBy(CREATE_RETARGETING_CONDITION_ITEM_VALIDATOR);

        return vb.getResult();
    };

    private static final Validator<GdUpdateRetargetingConditionItem, Defect>
            UPDATE_RETARGETING_CONDITION_ITEM_VALIDATOR = retargetingConditionItem -> {

        ModelItemValidationBuilder<GdUpdateRetargetingConditionItem> vb =
                ModelItemValidationBuilder.of(retargetingConditionItem);
        vb.item(GdUpdateRetargetingConditionItem.RETARGETING_CONDITION_ID)
                .check(validId());
        vb.list(GdUpdateRetargetingConditionItem.CONDITION_RULES)
                .checkEachBy(RETARGETING_CONDITION_RULE_ITEM_VALIDATOR);

        return vb.getResult();
    };

    private static final Validator<GdUpdateRetargetingConditions, Defect>
            UPDATE_RETARGETING_CONDITIONS_INPUT_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdUpdateRetargetingConditions> vb = ModelItemValidationBuilder.of(input);

        vb.list(GdUpdateRetargetingConditions.RETARGETING_CONDITIONS)
                .check(notEmptyCollection())
                .checkEachBy(UPDATE_RETARGETING_CONDITION_ITEM_VALIDATOR);

        return vb.getResult();
    };

    private static final Validator<GdDeleteRetargetingConditions, Defect>
            DELETE_RETARGETING_CONDITIONS_INPUT_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdDeleteRetargetingConditions> vb = ModelItemValidationBuilder.of(input);

        vb.list(GdDeleteRetargetingConditions.RETARGETING_CONDITION_IDS)
                .check(notEmptyCollection())
                .checkEach(validId());

        return vb.getResult();
    };

    private static final Validator<GdResumeRetargetings, Defect>
            RESUME_RETARGETING_INPUT_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdResumeRetargetings> vb = ModelItemValidationBuilder.of(input);

        vb.list(GdResumeRetargetings.RETARGETING_IDS)
                .check(notEmptyCollection())
                .checkEach(validId());

        return vb.getResult();
    };

    private static final Validator<GdSuspendRetargetings, Defect>
            SUSPEND_RETARGETING_INPUT_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdSuspendRetargetings> vb = ModelItemValidationBuilder.of(input);

        vb.list(GdSuspendRetargetings.RETARGETING_IDS)
                .check(notEmptyCollection())
                .checkEach(validId());

        return vb.getResult();
    };

    private static final Validator<GdDeleteRetargetings, Defect>
            DELETE_RETARGETING_INPUT_VALIDATOR = input -> {
        ModelItemValidationBuilder<GdDeleteRetargetings> vb = ModelItemValidationBuilder.of(input);

        vb.list(GdDeleteRetargetings.RETARGETING_IDS)
                .check(notEmptyCollection())
                .checkEach(validId());

        return vb.getResult();
    };

}
