package ru.yandex.direct.grid.processing.service.showcondition.validation;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import io.leangen.graphql.annotations.GraphQLNonNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionFilter;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionsContainer;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsItem;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdAddKeywordsOperators;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdChangeKeywordsCase;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdFindAndReplaceKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdKeywordIds;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdMoveSearchQueriesToMinusKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetAutoBids;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetAutoBidsNetworkByCoverage;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetAutoBidsSearchByTrafficVolume;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdSetBids;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywords;
import ru.yandex.direct.grid.processing.model.showcondition.mutation.GdUpdateKeywordsItem;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bids.validation.SetAutoBidValidator.MAX_CONTEXT_COVERAGE;
import static ru.yandex.direct.core.entity.bids.validation.SetAutoBidValidator.MAX_INCREASE_PERCENT;
import static ru.yandex.direct.core.entity.bids.validation.SetAutoBidValidator.MAX_TARGET_TRAFFIC_VOLUME;
import static ru.yandex.direct.core.entity.bids.validation.SetAutoBidValidator.MIN_CONTEXT_COVERAGE;
import static ru.yandex.direct.core.entity.bids.validation.SetAutoBidValidator.MIN_INCREASE_PERCENT;
import static ru.yandex.direct.core.entity.bids.validation.SetAutoBidValidator.MIN_TARGET_TRAFFIC_VOLUME;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.grid.processing.service.showcondition.bids.BidsDataService.MAX_IDS_PER_REQUEST;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.IDS_COLLECTION_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.LIMIT_OFFSET_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.STATS_FILTER_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.STAT_REQUIREMENTS_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.checkFieldsContainNonNullValue;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.checkOrderByForGoalsHasGoalIdsInRequest;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachInSetNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class ShowConditionValidationService {

    private final PathNodeConverterProvider pathConverterForRequestWithKeywordIds;
    private final GridValidationService gridValidationService;

    @Autowired
    public ShowConditionValidationService(GridValidationService gridValidationService) {
        this.gridValidationService = gridValidationService;
        //Нужен, чтобы фронту отдавать только поля из запроса.
        this.pathConverterForRequestWithKeywordIds = DefaultPathNodeConverterProvider.builder()
                .register(Keyword.class, SkipByDefaultMappingPathNodeConverter.builder().build())
                .register(Long.class, SkipByDefaultMappingPathNodeConverter.builder().build())
                .build();
    }

    @Nullable
    public GdValidationResult getValidationResultForRequestWithKeywordIds(Result<?> result) {
        return getValidationResultForRequestWithKeywordIds(result.getValidationResult());
    }

    @Nullable
    public GdValidationResult getValidationResultForRequestWithKeywordIds(ValidationResult<?, Defect> vr) {
        if (hasValidationIssues(vr)) {
            return GridValidationResultConversionService
                    .buildGridValidationResult(vr, path(field(GdKeywordIds.KEYWORD_IDS)),
                            pathConverterForRequestWithKeywordIds);
        }

        return null;
    }

    @Nullable
    public GdValidationResult getValidationResultForRequestWithSearchQuery(ValidationResult<?, Defect> vr) {
        if (hasValidationIssues(vr)) {
            return GridValidationResultConversionService
                    .buildGridValidationResult(
                            vr,
                            path(field(GdMoveSearchQueriesToMinusKeywords.SEARCH_QUERY)),
                            null
                    );
        }

        return null;
    }

    private static final Validator<List<Long>, Defect> COMMON_KEYWORD_IDS_VALIDATOR = req -> {
        ListValidationBuilder<Long, Defect> vb = ListValidationBuilder.of(req);
        vb
                .check(notNull())
                .check(maxListSize(MAX_IDS_PER_REQUEST), When.isValid())
                .checkEach(notNull(), When.isValid())
                .checkEach(validId(), When.isValid());
        return vb.getResult();
    };

    private static final Validator<GdKeywordIds, Defect> KEYWORD_IDS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdKeywordIds> vb = ModelItemValidationBuilder.of(req);

        vb.list(GdKeywordIds.KEYWORD_IDS)
                .checkBy(COMMON_KEYWORD_IDS_VALIDATOR)
                .check(notEmptyCollection(), When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdKeywordIds, Defect> KEYWORD_IDS_VALIDATOR_ALLOW_EMPTY = req -> {
        ModelItemValidationBuilder<GdKeywordIds> vb = ModelItemValidationBuilder.of(req);

        vb.list(GdKeywordIds.KEYWORD_IDS)
                .checkBy(COMMON_KEYWORD_IDS_VALIDATOR);
        return vb.getResult();
    };

    private static final Validator<GdSetBids, Defect> SET_BIDS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdSetBids> vb = ModelItemValidationBuilder.of(req);
        vb.list(GdSetBids.SHOW_CONDITION_IDS)
                .check(notEmptyCollection())
                .check(maxListSize(MAX_IDS_PER_REQUEST), When.isValid())
                .checkEach(validId(), When.isValid());

        return vb.getResult();
    };

    private static final Validator<GdSetAutoBidsNetworkByCoverage, Defect> NETWORK_BY_COVERAGE_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdSetAutoBidsNetworkByCoverage> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdSetAutoBidsNetworkByCoverage.CONTEXT_COVERAGE)
                        .check(inRange(MIN_CONTEXT_COVERAGE, MAX_CONTEXT_COVERAGE));
                vb.item(GdSetAutoBidsNetworkByCoverage.INCREASE_PERCENT)
                        .check(inRange(MIN_INCREASE_PERCENT, MAX_INCREASE_PERCENT));
                return vb.getResult();
            };

    private static final Validator<GdSetAutoBidsSearchByTrafficVolume, Defect> SEARCH_BY_TRAFFIC_VOLUME_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdSetAutoBidsSearchByTrafficVolume> vb =
                        ModelItemValidationBuilder.of(req);
                vb.item(GdSetAutoBidsSearchByTrafficVolume.TARGET_TRAFFIC_VOLUME)
                        .check(inRange(MIN_TARGET_TRAFFIC_VOLUME, MAX_TARGET_TRAFFIC_VOLUME));
                vb.item(GdSetAutoBidsSearchByTrafficVolume.INCREASE_PERCENTAGE)
                        .check(inRange(MIN_INCREASE_PERCENT, MAX_INCREASE_PERCENT));
                return vb.getResult();
            };

    private static final Validator<GdSetAutoBids, Defect> SET_AUTO_BIDS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdSetAutoBids> vb = ModelItemValidationBuilder.of(req);

        vb.check(checkFieldsContainNonNullValue(
                List.of(GdSetAutoBids.NETWORK_BY_COVERAGE, GdSetAutoBids.SEARCH_BY_TRAFFIC_VOLUME)));
        vb.item(GdSetAutoBids.NETWORK_BY_COVERAGE)
                .checkBy(NETWORK_BY_COVERAGE_VALIDATOR, When.notNull());
        vb.item(GdSetAutoBids.SEARCH_BY_TRAFFIC_VOLUME)
                .checkBy(SEARCH_BY_TRAFFIC_VOLUME_VALIDATOR, When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdUpdateKeywords, Defect> UPDATE_KEYWORDS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdUpdateKeywords> vb = ModelItemValidationBuilder.of(req);

        vb.list(GdUpdateKeywords.KEYWORD_UPDATE_ITEMS)
                .check(notNull())
                .check(notEmptyCollection(), When.notNull())
                .checkEach(notNull(), When.isValid())
                .checkEach(checkFieldsContainNonNullValue(ImmutableList.of(GdUpdateKeywordsItem.ID)), When.isValid())
                .checkEach(checkFieldsContainNonNullValue(
                        ImmutableList.of(GdUpdateKeywordsItem.KEYWORD,
                                GdUpdateKeywordsItem.PRICE, GdUpdateKeywordsItem.PRICE_CONTEXT,
                                GdUpdateKeywordsItem.AUTOBUDGET_PRIORITY)), When.isValid()
                );
        return vb.getResult();
    };

    private static final Validator<GdAddKeywordsItem, Defect> ADD_KEYWORDS_ITEM_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdAddKeywordsItem> vb = ModelItemValidationBuilder.of(req);

        vb.item(GdAddKeywordsItem.AD_GROUP_ID)
                .check(validId());
        vb.item(GdAddKeywordsItem.KEYWORD)
                .check(notBlank());
        return vb.getResult();
    };
    private static final Validator<GdAddKeywords, Defect> ADD_KEYWORDS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdAddKeywords> vb = ModelItemValidationBuilder.of(req);

        vb.list(GdAddKeywords.ADD_ITEMS)
                .check(notEmptyCollection())
                .checkEachBy(ADD_KEYWORDS_ITEM_VALIDATOR, When.isValid());
        return vb.getResult();
    };

    private static final Validator<GdAddKeywordsOperators, Defect> ADD_KEYWORDS_OPERATORS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdAddKeywordsOperators> vb = ModelItemValidationBuilder.of(req);

        vb.list(GdAddKeywordsOperators.KEYWORD_IDS)
                .check(notNull())
                .check(notEmptyCollection(), When.notNull())
                .checkEach(validId(), When.isValid());
        vb.list(GdAddKeywordsOperators.OPERATORS_MODE)
                .check(notNull())
                .check(notEmptyCollection(), When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdFindAndReplaceKeywords, Defect> FIND_AND_REPLACE_KEYWORDS_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdFindAndReplaceKeywords> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdFindAndReplaceKeywords.LIMIT_OFFSET)
                        .checkBy(LIMIT_OFFSET_VALIDATOR, When.notNull());
                vb.item(GdFindAndReplaceKeywords.SEARCH_TEXT)
                        .check(notNull());
                PhraseSyntaxValidator.setCommonChecks(vb.item(GdFindAndReplaceKeywords.CHANGE_TEXT));

                vb.item(GdFindAndReplaceKeywords.CHANGE_MODE)
                        .check(notNull());
                vb.item(GdFindAndReplaceKeywords.SEARCH_OPTIONS)
                        .check(notNull());
                vb.item(GdFindAndReplaceKeywords.FIELDS)
                        .check(notNull())
                        .check(notEmptyCollection(), When.notNull());

                return vb.getResult();
            };

    private static final Validator<GdChangeKeywordsCase, Defect> CHANGE_KEYWORDS_CASE_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdChangeKeywordsCase> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdChangeKeywordsCase.CASE_MODE)
                        .check(notNull());

                vb.item(GdChangeKeywordsCase.FIELDS)
                        .check(notNull())
                        .check(notEmptyCollection(), When.notNull());

                return vb.getResult();
            };

    private static final Validator<GdShowConditionFilter, Defect> SHOW_CONDITION_FILTER_VALIDATOR =
            filter -> {
                ModelItemValidationBuilder<GdShowConditionFilter> vb = ModelItemValidationBuilder.of(filter);
                vb.item(GdShowConditionFilter.CAMPAIGN_ID_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
                vb.item(GdShowConditionFilter.AD_GROUP_ID_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());

                vb.item(GdShowConditionFilter.SHOW_CONDITION_ID_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
                vb.item(GdShowConditionFilter.SHOW_CONDITION_ID_NOT_IN)
                        .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
                vb.item(GdShowConditionFilter.MIN_PRICE)
                        .check(notLessThan(BigDecimal.ZERO), When.notNull());
                vb.item(GdShowConditionFilter.MIN_PRICE_CONTEXT)
                        .check(notLessThan(BigDecimal.ZERO), When.notNull());
                vb.item(GdShowConditionFilter.AUTOBUDGET_PRIORITY_IN)
                        .check(eachInSetNotNull(), When.notNull());
                vb.item(GdShowConditionFilter.STATUS_IN)
                        .check(eachInSetNotNull(), When.notNull());
                vb.item(GdShowConditionFilter.SHOW_CONDITION_STATUS_IN)
                        .check(eachInSetNotNull(), When.notNull());
                vb.item(GdShowConditionFilter.TYPE_IN)
                        .check(eachInSetNotNull(), When.notNull());
                vb.item(GdShowConditionFilter.STATS)
                        .checkBy(STATS_FILTER_VALIDATOR, When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdShowConditionsContainer, Defect> SHOW_CONDITIONS_CONTAINER_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdShowConditionsContainer> vb = ModelItemValidationBuilder.of(req);

                vb.item(GdShowConditionsContainer.FILTER)
                        .check(notNull(), When.isTrue(req.getFilterKey() == null));
                vb.item(GdShowConditionsContainer.FILTER_KEY)
                        .check(notNull(), When.isTrue(req.getFilter() == null));

                vb.item(GdShowConditionsContainer.FILTER)
                        .checkBy(SHOW_CONDITION_FILTER_VALIDATOR, When.notNull());
                vb.item(GdShowConditionsContainer.FILTER_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdShowConditionsContainer.STAT_REQUIREMENTS)
                        .checkBy(STAT_REQUIREMENTS_VALIDATOR);
                vb.item(GdShowConditionsContainer.LIMIT_OFFSET)
                        .checkBy(LIMIT_OFFSET_VALIDATOR);
                vb.item(GdShowConditionsContainer.CACHE_KEY)
                        .check(notBlank(), When.notNull());

                vb.item(GdShowConditionsContainer.ORDER_BY)
                        .checkByFunction(orderByList ->
                                        checkOrderByForGoalsHasGoalIdsInRequest(orderByList, req.getStatRequirements()),
                                When.notNull());

                return vb.getResult();
            };

    public void validateRequestWithKeywordIds(GdKeywordIds requestWithGdKeywordIds) {
        gridValidationService.applyValidator(KEYWORD_IDS_VALIDATOR, requestWithGdKeywordIds, false);
    }

    public void validateUpdateKeywordsRequest(GdUpdateKeywords updateKeywordsRequest) {
        gridValidationService.applyValidator(UPDATE_KEYWORDS_VALIDATOR, updateKeywordsRequest, false);
    }

    public void validateAddKeywordsOperators(GdAddKeywordsOperators addKeywordsOperators) {
        gridValidationService.applyValidator(ADD_KEYWORDS_OPERATORS_VALIDATOR, addKeywordsOperators, false);
    }

    public void validateAddKeywordsRequest(GdAddKeywords addKeywordsRequest) {
        gridValidationService.applyValidator(ADD_KEYWORDS_VALIDATOR, addKeywordsRequest, false);
    }

    public void validateFindAndReplaceKeywordsRequest(GdFindAndReplaceKeywords findAndReplaceKeywords) {
        gridValidationService.applyValidator(KEYWORD_IDS_VALIDATOR, findAndReplaceKeywords, false);
        gridValidationService.applyValidator(FIND_AND_REPLACE_KEYWORDS_VALIDATOR, findAndReplaceKeywords, false);
    }

    public void validateChangeKeywordsCaseRequest(@GraphQLNonNull GdChangeKeywordsCase changeKeywordsCase) {
        gridValidationService.applyValidator(KEYWORD_IDS_VALIDATOR, changeKeywordsCase, false);
        gridValidationService.applyValidator(CHANGE_KEYWORDS_CASE_VALIDATOR, changeKeywordsCase, false);
    }

    public void validateGdShowConditionsContainer(GdShowConditionsContainer input) {
        gridValidationService.applyValidator(SHOW_CONDITIONS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateSetBidsRequest(GdSetBids input) {
        gridValidationService.applyValidator(SET_BIDS_VALIDATOR, input, false);
    }

    public void validateSetAutoBidsRequest(GdSetAutoBids input) {
        gridValidationService.applyValidator(KEYWORD_IDS_VALIDATOR_ALLOW_EMPTY, input, false);
        gridValidationService.applyValidator(SET_AUTO_BIDS_VALIDATOR, input, false);
    }
}
