package ru.yandex.direct.grid.processing.service.smartfilter;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFilterCondition;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFilters;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFiltersItem;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdAutobudgetPriority;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdConditionOperator;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdSmartFilterTab;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdTargetFunnel;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Component
@ParametersAreNonnullByDefault
public class AddSmartFiltersRequestConverter {

    private static final PathNodeConverter VALIDATE_GD_ADD_SMART_FILTERS_ITEM_CONVERTER =
            SkipByDefaultMappingPathNodeConverter.builder()
                    .replace(PerformanceFilter.PID.name(), GdAddSmartFiltersItem.AD_GROUP_ID.name())
                    .replace(PerformanceFilter.NAME.name(), GdAddSmartFiltersItem.NAME.name())
                    .replace(PerformanceFilter.CONDITIONS.name(), GdAddSmartFiltersItem.CONDITIONS.name())
                    .replace(PerformanceFilter.PRICE_CPC.name(), GdAddSmartFiltersItem.PRICE_CPC.name())
                    .replace(PerformanceFilter.PRICE_CPA.name(), GdAddSmartFiltersItem.PRICE_CPA.name())
                    .replace(PerformanceFilter.AUTOBUDGET_PRIORITY.name(),
                            GdAddSmartFiltersItem.AUTOBUDGET_PRIORITY.name())
                    .replace(PerformanceFilter.TARGET_FUNNEL.name(), GdAddSmartFiltersItem.TARGET_FUNNEL.name())
                    .replace(PerformanceFilter.IS_SUSPENDED.name(), GdAddSmartFiltersItem.IS_SUSPENDED.name())
                    .build();

    private static final PathNodeConverter VALIDATE_GD_ADD_SMART_FILTERS_CONDITIONS_CONVERTER =
            new PathNodeConverter() {
                private PathNodeConverter innerConverter = SkipByDefaultMappingPathNodeConverter.builder()
                        .replace(PerformanceFilterCondition.FIELD_NAME.name(),
                                GdAddSmartFilterCondition.FIELD.name())
                        .replace(PerformanceFilterCondition.OPERATOR.name(),
                                GdAddSmartFilterCondition.OPERATOR.name())
                        .replace(PerformanceFilterCondition.STRING_VALUE.name(),
                                GdAddSmartFilterCondition.STRING_VALUE.name())
                        .replace(PerformanceFilterCondition.PARSED_VALUE,
                                GdAddSmartFilterCondition.STRING_VALUE.name())
                        .build();

                @Override
                public Path convert(PathNode.Field field) {
                    return innerConverter.convert(field);
                }

                @Override
                public Path convert(PathNode.Field field, PathNode.Index index) {
                    return emptyPath();
                }
            };

    private static final PathNodeConverterProvider PATH_NODE_CONVERTER_PROVIDER =
            DefaultPathNodeConverterProvider.builder()
                    .register(PerformanceFilter.class, VALIDATE_GD_ADD_SMART_FILTERS_ITEM_CONVERTER)
                    .register(PerformanceFilterCondition.class, VALIDATE_GD_ADD_SMART_FILTERS_CONDITIONS_CONVERTER)
                    .build();

    private final PerformanceFilterStorage performanceFilterStorage;

    @Autowired
    public AddSmartFiltersRequestConverter(
            PerformanceFilterStorage performanceFilterStorage) {
        this.performanceFilterStorage = performanceFilterStorage;
    }

    PerformanceFilter convertGdAddItem(GdAddSmartFiltersItem gdSmartFilter, Feed feed) {
        checkNotNull(feed, "feed is required");

        BusinessType businessType = feed.getBusinessType();
        FeedType feedType = feed.getFeedType();
        Source source = feed.getSource();
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType, source);

        List<PerformanceFilterCondition> conditions = mapList(gdSmartFilter.getConditions(),
                gdSmartFilterCondition -> convertGdCondition(gdSmartFilterCondition, filterSchema));
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, conditions);

        return new PerformanceFilter()
                .withPid(gdSmartFilter.getAdGroupId())
                .withName(gdSmartFilter.getName())
                .withTargetFunnel(fromGdTargetFunnel(gdSmartFilter.getTargetFunnel()))
                .withPriceCpc(gdSmartFilter.getPriceCpc())
                .withPriceCpa(gdSmartFilter.getPriceCpa())
                .withAutobudgetPriority(fromGdAutobudgetPriority(gdSmartFilter.getAutobudgetPriority()))
                .withIsSuspended(gdSmartFilter.getIsSuspended())
                .withBusinessType(businessType)
                .withFeedType(feedType)
                .withSource(source)
                .withTab(fromGdSmartFilterTab(gdSmartFilter.getTab()))
                .withConditions(conditions);
    }

    private static PerformanceFilterCondition<?> convertGdCondition(GdAddSmartFilterCondition gdSmartFilterCondition,
                                                                    FilterSchema filterSchema) {
        return new PerformanceFilterCondition<>(
                filterSchema.translateFieldNameFromGd(gdSmartFilterCondition.getField()),
                fromGdConditionOperator(gdSmartFilterCondition.getOperator()),
                gdSmartFilterCondition.getStringValue());
    }

    @Nullable
    public static GdValidationResult toGdValidationResult(ValidationResult<?, Defect> vr) {
        if (hasValidationIssues(vr)) {
            return GridValidationResultConversionService
                    .buildGridValidationResult(vr, path(field(GdAddSmartFilters.ADD_ITEMS)),
                            PATH_NODE_CONVERTER_PROVIDER);
        }
        return null;
    }

}
