package ru.yandex.direct.grid.processing.service.smartfilter;

import java.util.EnumMap;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.model.NowOptimizingBy;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridShowConditionMapping;
import ru.yandex.direct.grid.core.entity.smartfilter.model.GdiSmartFilter;
import ru.yandex.direct.grid.core.entity.smartfilter.model.GdiSmartFilterFilter;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.showcondition.GdShowConditionAutobudgetPriority;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterCondition;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterConditionOperator;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterNowOptimizingBy;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterPrimaryStatus;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterStatus;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterTab;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterTargetFunnel;
import ru.yandex.direct.grid.processing.util.StatHelper;

import static ru.yandex.direct.core.entity.keyword.repository.KeywordMapping.priceFromDbFormat;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class SmartFilterConverter {

    private static final EnumMap<Operator, GdSmartFilterConditionOperator> gdOperatorByCoreOperator;
    private static final EnumMap<GdSmartFilterConditionOperator, Operator> coreOperatorByGdOperator;

    static {
        gdOperatorByCoreOperator = new EnumMap<>(Operator.class);
        gdOperatorByCoreOperator.put(Operator.GREATER, GdSmartFilterConditionOperator.GREATER_THAN);
        gdOperatorByCoreOperator.put(Operator.LESS, GdSmartFilterConditionOperator.LESS_THAN);
        gdOperatorByCoreOperator.put(Operator.EQUALS, GdSmartFilterConditionOperator.EQUALS_ANY);
        gdOperatorByCoreOperator.put(Operator.RANGE, GdSmartFilterConditionOperator.RANGE);
        gdOperatorByCoreOperator.put(Operator.CONTAINS, GdSmartFilterConditionOperator.CONTAINS_ANY);
        gdOperatorByCoreOperator.put(Operator.NOT_CONTAINS, GdSmartFilterConditionOperator.NOT_CONTAINS_ALL);
        gdOperatorByCoreOperator.put(Operator.EXISTS, GdSmartFilterConditionOperator.EXISTS);
        coreOperatorByGdOperator = new EnumMap<>(
                EntryStream.of(gdOperatorByCoreOperator)
                        .invert()
                        .toMap());
    }

    static GdSmartFilter toGdSmartFilter(PerformanceFilter performanceFilter, GdAdGroupTruncated group,
                                         PerformanceFilterStorage performanceFilterStorage) {
        BusinessType businessType = performanceFilter.getBusinessType();
        FeedType feedType = performanceFilter.getFeedType();
        Source source = performanceFilter.getSource();
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType, source);

        List<GdSmartFilterCondition> conditions = mapList(performanceFilter.getConditions(),
                performanceFilterCondition -> toGdCondition(performanceFilterCondition, filterSchema));
        GdCampaignTruncated campaign = group.getCampaign();
        GdSmartFilterStatus status = calcStatus(performanceFilter, campaign, group);

        return new GdSmartFilter()
                .withId(performanceFilter.getId())
                .withAdGroupId(performanceFilter.getPid())
                .withCampaignId(campaign.getId())
                .withName(performanceFilter.getName())
                .withNowOptimizingBy(toGdNowOptimizingBy(performanceFilter.getNowOptimizingBy()))
                .withTargetFunnel(toGdTargetFunnel(performanceFilter.getTargetFunnel()))
                .withPriceCpc(priceFromDbFormat(performanceFilter.getPriceCpc()))
                .withPriceCpa(priceFromDbFormat(performanceFilter.getPriceCpa()))
                .withAutobudgetPriority(toGdAutobudgetPriority(performanceFilter.getAutobudgetPriority()))
                .withIsSuspended(performanceFilter.getIsSuspended())
                .withStatus(status)
                .withConditions(conditions)
                .withTab(toGdSmartFilterTab(performanceFilter.getTab()));
    }

    private static GdSmartFilterStatus calcStatus(PerformanceFilter performanceFilter, GdCampaignTruncated campaign,
                                                  GdAdGroupTruncated group) {
        return new GdSmartFilterStatus()
                .withReadOnly(campaign.getStatus().getReadOnly() || group.getStatus().getArchived())
                .withSuspended(performanceFilter.getIsSuspended())
                .withPrimaryStatus(calcPrimaryStatus(performanceFilter));
    }

    private static GdSmartFilterPrimaryStatus calcPrimaryStatus(PerformanceFilter performanceFilter) {
        if (performanceFilter.getIsSuspended()) {
            return GdSmartFilterPrimaryStatus.STOPPED;
        }
        return GdSmartFilterPrimaryStatus.ACTIVE;
    }

    static GdSmartFilterNowOptimizingBy toGdNowOptimizingBy(NowOptimizingBy nowOptimizingBy) {
        return GdSmartFilterNowOptimizingBy.fromSource(nowOptimizingBy);
    }

    static GdSmartFilterTargetFunnel toGdTargetFunnel(TargetFunnel targetFunnel) {
        return GdSmartFilterTargetFunnel.fromSource(targetFunnel);
    }

    public static TargetFunnel fromGdTargetFunnel(GdSmartFilterTargetFunnel targetFunnel) {
        return GdSmartFilterTargetFunnel.toSource(targetFunnel);
    }

    @Nullable
    public static GdShowConditionAutobudgetPriority toGdAutobudgetPriority(@Nullable Integer autobudgetPriority) {
        return Optional.ofNullable(autobudgetPriority)
                .map(Integer::longValue)
                .map(GridShowConditionMapping::autobudgetPriorityFromNum)
                .map(GdShowConditionAutobudgetPriority::fromSource)
                .orElse(null);
    }

    @Nullable
    public static Integer fromGdAutobudgetPriority(@Nullable GdShowConditionAutobudgetPriority autobudgetPriority) {
        return Optional.ofNullable(autobudgetPriority)
                .map(GdShowConditionAutobudgetPriority::toSource)
                .map(GridShowConditionMapping::numFromAutobudgetPriority)
                .map(Long::intValue)
                .orElse(null);
    }

    private static GdSmartFilterCondition toGdCondition(PerformanceFilterCondition performanceFilterCondition,
                                                        FilterSchema filterSchema) {
        return new GdSmartFilterCondition()
                .withField(filterSchema.translateFieldNameForGd(performanceFilterCondition.getFieldName()))
                .withOperator(toGdConditionOperator(performanceFilterCondition.getOperator()))
                .withStringValue(performanceFilterCondition.getStringValue());
    }

    static GdSmartFilterConditionOperator toGdConditionOperator(Operator operator) {
        GdSmartFilterConditionOperator gdSmartFilterConditionOperator = gdOperatorByCoreOperator.get(operator);
        if (gdSmartFilterConditionOperator == null) {
            throw new IllegalArgumentException("Unknown operator type: " + operator);
        }
        return gdSmartFilterConditionOperator;
    }

    static GdiSmartFilter toGdiSmartFilter(GdSmartFilter smartFilter) {
        return new GdiSmartFilter()
                .withSmartFilterId(smartFilter.getId())
                .withAdGroupId(smartFilter.getAdGroupId())
                .withCampaignId(smartFilter.getCampaignId());
    }

    static GdiSmartFilterFilter toInternalFilter(GdSmartFilterFilter filter) {
        return new GdiSmartFilterFilter()
                .withSmartFilterIdIn((filter.getSmartFilterIdIn()))
                .withCampaignIdIn(filter.getCampaignIdIn())
                .withAdGroupIdIn(filter.getAdGroupIdIn())

                .withMinPriceCpc(filter.getMinPriceCpc())
                .withMaxPriceCpc(filter.getMaxPriceCpc())
                .withMinPriceCpa(filter.getMinPriceCpa())
                .withMaxPriceCpa(filter.getMaxPriceCpa())

                .withAutobudgetPriorityIn(
                        mapSet(filter.getAutobudgetPriorityIn(), GdShowConditionAutobudgetPriority::toSource))
                .withStats(StatHelper.toInternalStatsFilter(filter.getStats()))
                .withGoalStats(mapList(filter.getGoalStats(), StatHelper::toInternalGoalStatsFilter));
    }

    public static Operator fromGdConditionOperator(GdSmartFilterConditionOperator gdSmartFilterConditionOperator) {
        Operator operator = coreOperatorByGdOperator.get(gdSmartFilterConditionOperator);
        if (operator == null) {
            throw new IllegalArgumentException("Unknown operator type: " + gdSmartFilterConditionOperator);
        }
        return operator;
    }

    private static GdSmartFilterTab toGdSmartFilterTab(@Nullable PerformanceFilterTab tab) {
        if (tab == null) {
            return GdSmartFilterTab.CONDITION;
        }
        return GdSmartFilterTab.fromSource(tab);
    }

    @Nullable
    public static PerformanceFilterTab fromGdSmartFilterTab(@Nullable GdSmartFilterTab tab) {
        return GdSmartFilterTab.toSource(tab);
    }
}
