package ru.yandex.direct.grid.processing.service.smartfilter;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFiltersQueryFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFiltersAddOperation;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.SmartFilterFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.smartfilter.GridSmartFilterYtRepository;
import ru.yandex.direct.grid.core.entity.smartfilter.model.GdiSmartFilter;
import ru.yandex.direct.grid.core.entity.smartfilter.model.GdiSmartFilterStats;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.model.GdStatPreset;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterBaseStatus;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFiltersContainer;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFiltersContext;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFilters;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFiltersItem;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFiltersPayload;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFiltersPayloadItem;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdDeleteSmartFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdDeleteSmartFilterItem;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdDeleteSmartFilterPayload;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdDeleteSmartFilterPayloadItem;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFilters;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFiltersItem;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFiltersPayload;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFiltersPayloadItem;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.group.GroupDataService;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.grid.processing.service.smartfilter.converter.GdDeleteSmartFiltersConverter;
import ru.yandex.direct.grid.processing.service.smartfilter.converter.GdUpdateSmartFiltersConverter;
import ru.yandex.direct.grid.processing.util.GoalHelper;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.feature.FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.toGdSmartFilter;
import static ru.yandex.direct.grid.processing.service.smartfilter.converter.GdUpdateSmartFiltersConverter.fromGd;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.grid.processing.util.StatHelper.applyEntityStatsFilter;
import static ru.yandex.direct.grid.processing.util.StatHelper.applyGoalsStatFilters;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalGoalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;
import static ru.yandex.direct.grid.processing.util.StatHelper.recalcTotalStatsForUnitedGrid;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
@ParametersAreNonnullByDefault
public class SmartFilterDataService {

    private final ShardHelper shardHelper;
    private final AddSmartFiltersRequestConverter addSmartFiltersRequestConverter;
    private final PerformanceFilterService performanceFilterService;
    private final FeedService feedService;
    private final PerformanceFilterStorage performanceFilterStorage;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final GridSmartFilterYtRepository gridSmartFilterYtRepository;
    private final GridShortenerService gridShortenerService;
    private final CampaignInfoService campaignInfoService;
    private final GroupDataService groupDataService;
    private final FeatureService featureService;
    private final MetrikaGoalsService metrikaGoalsService;

    private static final GdStatPreset DEFAULT_PRESET = GdStatPreset.TODAY;

    @Autowired
    public SmartFilterDataService(
            ShardHelper shardHelper,
            AddSmartFiltersRequestConverter addSmartFiltersRequestConverter,
            PerformanceFilterService performanceFilterService,
            FeedService feedService,
            PerformanceFilterStorage performanceFilterStorage,
            AdGroupRepository adGroupRepository,
            CampaignRepository campaignRepository,
            GridSmartFilterYtRepository gridSmartFilterYtRepository,
            GridShortenerService gridShortenerService, CampaignInfoService campaignInfoService,
            FeatureService featureService,
            GroupDataService groupDataService,
            MetrikaGoalsService metrikaGoalsService) {
        this.shardHelper = shardHelper;
        this.addSmartFiltersRequestConverter = addSmartFiltersRequestConverter;
        this.performanceFilterService = performanceFilterService;
        this.feedService = feedService;
        this.performanceFilterStorage = performanceFilterStorage;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.gridSmartFilterYtRepository = gridSmartFilterYtRepository;
        this.gridShortenerService = gridShortenerService;
        this.campaignInfoService = campaignInfoService;
        this.featureService = featureService;
        this.groupDataService = groupDataService;
        this.metrikaGoalsService = metrikaGoalsService;
    }

    public GdAddSmartFiltersPayload addSmartFilters(ClientId clientId, Long operatorUid, GdAddSmartFilters request) {
        checkNotNull(clientId, "clientId is required");
        List<PerformanceFilter> performanceFiltersToAdd =
                convertGdAddItems(request.getAddItems(), clientId);

        PerformanceFiltersAddOperation addOperation =
                performanceFilterService.createPartialAddOperation(clientId, operatorUid, performanceFiltersToAdd);

        MassResult<Long> result = addOperation.prepareAndApply();
        List<GdAddSmartFiltersPayloadItem> addedItems =
                getResults(result, id -> new GdAddSmartFiltersPayloadItem().withId(id));

        ValidationResult<?, Defect> vr = result.getValidationResult();
        GdValidationResult validationResult = AddSmartFiltersRequestConverter.toGdValidationResult(vr);

        return new GdAddSmartFiltersPayload()
                .withAddedItems(addedItems)
                .withValidationResult(validationResult);
    }

    private List<PerformanceFilter> convertGdAddItems(List<GdAddSmartFiltersItem> addItems, ClientId clientId) {
        Set<Long> adGroupIds = listToSet(addItems, GdAddSmartFiltersItem::getAdGroupId);
        Map<Long, Feed> feedByAdGroupId = feedService.getFeedByPerformanceAdGroupId(clientId, adGroupIds);

        return mapList(addItems, item -> {
            Long adGroupId = item.getAdGroupId();
            Feed feed = feedByAdGroupId.get(adGroupId);
            return addSmartFiltersRequestConverter.convertGdAddItem(item, feed);
        });
    }

    /**
     * Обновление перформанс-фильтров
     */
    GdUpdateSmartFiltersPayload updateFilter(ClientId clientId, Long operatorUid, GdUpdateSmartFilters input) {

        List<GdUpdateSmartFiltersItem> updateItems = input.getUpdateItems();
        List<Long> filterIds = mapList(updateItems, GdUpdateSmartFiltersItem::getId);
        Map<Long, PerformanceFilter> oldFilterById = getFilterById(clientId, filterIds);

        List<PerformanceFilter> performanceFilters =
                mapList(updateItems, item -> fromGd(item, oldFilterById, performanceFilterStorage));
        MassResult<Long> longMassResult =
                performanceFilterService.updatePerformanceFilters(clientId, operatorUid, performanceFilters);
        return convertToGdUpdateAdGroupPayload(longMassResult);
    }

    private GdUpdateSmartFiltersPayload convertToGdUpdateAdGroupPayload(MassResult<Long> result) {
        ValidationResult<?, Defect> vr = result.getValidationResult();
        GdValidationResult validationResult = GdUpdateSmartFiltersConverter.toGdValidationResult(vr);

        List<GdUpdateSmartFiltersPayloadItem> gdUpdateSmartFilterPayloadItems =
                getResults(result, id -> new GdUpdateSmartFiltersPayloadItem().withId(id));

        return new GdUpdateSmartFiltersPayload()
                .withUpdatedItems(gdUpdateSmartFilterPayloadItems)
                .withValidationResult(validationResult);
    }

    private Map<Long, PerformanceFilter> getFilterById(ClientId clientId, List<Long> filterIds) {
        PerformanceFiltersQueryFilter filtersQuery = PerformanceFiltersQueryFilter.newBuilder()
                .withPerfFilterIds(filterIds).build();
        List<PerformanceFilter> filters = performanceFilterService.getPerformanceFilters(clientId, filtersQuery);
        return listToMap(filters, PerformanceFilter::getId);
    }


    /**
     * Удаление перформанс-фильтров
     */
    GdDeleteSmartFilterPayload deleteFilters(ClientId clientId, Long operatorUid, GdDeleteSmartFilter input) {
        List<Long> deleteFilterIds = mapList(input.getDeleteItems(), GdDeleteSmartFilterItem::getId);
        MassResult<Long> longMassResult =
                performanceFilterService.deletePerformanceFilters(clientId, operatorUid, deleteFilterIds);
        return convertToGdDeleteAdGroupPayload(longMassResult);
    }

    private GdDeleteSmartFilterPayload convertToGdDeleteAdGroupPayload(MassResult<Long> result) {
        ValidationResult<?, Defect> vr = result.getValidationResult();
        GdValidationResult validationResult = GdDeleteSmartFiltersConverter.toGdValidationResult(vr);

        List<GdDeleteSmartFilterPayloadItem> gdDeleteSmartFilterPayloadItems =
                getResults(result, id -> new GdDeleteSmartFilterPayloadItem().withId(id));

        return new GdDeleteSmartFilterPayload()
                .withDeletedItems(gdDeleteSmartFilterPayloadItems)
                .withValidationResult(validationResult);
    }

    GdSmartFiltersContext getSmartFilters(ClientId clientId, GdSmartFiltersContainer input,
                                          GridGraphQLContext context) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        if (input.getFilterKey() != null) {
            GdSmartFilterFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    clientId,
                    GdSmartFilterFilter.class,
                    () -> new GdSmartFilterFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }

        GdSmartFilterFilter filter = input.getFilter();
        PerformanceFiltersQueryFilter queryFilter = PerformanceFiltersQueryFilter.newBuilder()
                .withClientId(clientId)
                .withCampaignIds(filter.getCampaignIdIn())
                .withAdGroupIds(filter.getAdGroupIdIn())
                .withPerfFilterIds(filter.getSmartFilterIdIn())
                .withoutDeleted()
                .withBaseStatuses(mapSet(filter.getStatusIn(), GdSmartFilterBaseStatus::toSource))
                .withAutobudgetPriorities(
                        mapSet(filter.getAutobudgetPriorityIn(), SmartFilterConverter::fromGdAutobudgetPriority))
                .withTargetFunnels(mapSet(filter.getTargetFunnelIn(), SmartFilterConverter::fromGdTargetFunnel))
                .withNameContains(filter.getNameContains())
                .withNameNotContains(filter.getNameNotContains())
                .withMinPriceCpc(filter.getMinPriceCpc())
                .withMaxPriceCpc(filter.getMaxPriceCpc())
                .withMinPriceCpa(filter.getMinPriceCpa())
                .withMaxPriceCpa(filter.getMaxPriceCpa())
                .build();
        List<PerformanceFilter> performanceFilters = performanceFilterService.getPerformanceFilters(clientId, queryFilter);

        List<Long> filterIds = mapList(performanceFilters, PerformanceFilter::getId);
        Map<Long, DbStrategy> strategyByFilterIds = campaignRepository.getStrategyByFilterIds(shard, filterIds);
        Map<Long, StrategyName> strategyBySmartFilterIds = EntryStream.of(strategyByFilterIds)
                .mapValues(DbStrategyBase::getStrategyName)
                .toMap();

        List<GdSmartFilter> gdSmartFilters = convertToGdSmartFilters(shard, context, performanceFilters);

        SmartFilterFetchedFieldsResolver resolver = context.getFetchedFieldsReslover().getSmartFilter();

        boolean cpcAndCpmOnOneGridEnabled = featureService
                .isEnabledForClientId(clientId, CPC_AND_CPM_ON_ONE_GRID_ENABLED);

        if (resolver.getStats() || filter.getStats() != null || filter.getGoalStats() != null) {
            GdStatRequirements statRequirements = input.getStatRequirements();
            statRequirements = normalizeStatRequirements(statRequirements, context.getInstant());
            input.setStatRequirements(statRequirements);

            Set<Long> goalIds = GoalHelper.combineGoalIds(
                    input.getStatRequirements().getGoalIds(), input.getFilter().getGoalStats());
            boolean getRevenueOnlyByAvailableGoals = featureService
                    .isEnabledForClientId(clientId, GET_REVENUE_ONLY_BY_AVAILABLE_GOALS);
            Set<Long> availableGoalIds = null;
            if (CollectionUtils.isNotEmpty(goalIds) && getRevenueOnlyByAvailableGoals) {
                availableGoalIds = metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                        context.getOperator().getUid(), clientId);
            }

            List<GdiSmartFilter> gdiSmartFilters = mapList(gdSmartFilters, SmartFilterConverter::toGdiSmartFilter);

            Map<Long, GdiSmartFilterStats> statsFromYt = gridSmartFilterYtRepository
                    .getStatistic(gdiSmartFilters, statRequirements.getFrom(),
                            statRequirements.getTo(), goalIds, availableGoalIds);

            Map<Long, GdiSmartFilterStats> statsMap = StreamEx.of(gdSmartFilters)
                    .toMap(GdSmartFilter::getId, gd -> nvl(statsFromYt.get(gd.getId()),
                            new GdiSmartFilterStats()
                                    .withStat(GridStatNew.addZeros(new GdiEntityStats()))
                                    .withGoalStats(emptyList())));

            for (GdSmartFilter gd : gdSmartFilters) {
                gd.setStats(StatHelper.internalStatsToOuter(statsMap.get(gd.getId()).getStat(),
                        GdCampaignType.PERFORMANCE, cpcAndCpmOnOneGridEnabled));
                gd.setGoalStats(ifNotNull(statsMap.get(gd.getId()).getGoalStats(),
                        goalStats -> mapList(goalStats, StatHelper::internalGoalStatToOuter)));
            }

            gdSmartFilters = StreamEx.of(gdSmartFilters)
                    .filter(gd -> applyEntityStatsFilter(filter.getStats(), gd.getStats()))
                    .filter(gd -> applyGoalsStatFilters(filter.getGoalStats(), gd.getGoalStats()))
                    .toList();
        }

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        List<GdSmartFilter> cut = gdSmartFilters.stream()
                .sorted(SmartFilterUtils.getComparator(input.getOrderBy(), strategyBySmartFilterIds))
                .skip(range.offset()).limit(range.limit())
                .collect(toList());
        String cacheKey = ""; // todo maxlog: подумать о том, чтобы включить redis'ное кэширование

        var totalStats = calcTotalStats(mapList(cut, GdSmartFilter::getStats));
        if (cpcAndCpmOnOneGridEnabled) {
            var stats = StreamEx.of(gdSmartFilters)
                    .map(GdSmartFilter::getStats)
                    .toList();
            recalcTotalStatsForUnitedGrid(totalStats, Map.of(GdCampaignType.PERFORMANCE, stats));
        }

        return new GdSmartFiltersContext()
                .withTotalCount(gdSmartFilters.size())
                .withSmartFilterIds(listToSet(gdSmartFilters, GdSmartFilter::getId))
                .withFilter(input.getFilter())
                .withRowset(cut)
                .withTotalStats(totalStats)
                .withTotalGoalStats(calcTotalGoalStats(totalStats, mapList(cut, GdSmartFilter::getGoalStats)))
                .withCacheKey(cacheKey);
    }

    private List<GdSmartFilter> convertToGdSmartFilters(int shard, GridGraphQLContext context,
                                                        List<PerformanceFilter> performanceFilters) {
        User operator = context.getOperator();
        GdClientInfo client = context.getQueriedClient();
        ClientId clientId = ClientId.fromLong(client.getId());

        Set<Long> adGroupIds = listToSet(performanceFilters, PerformanceFilter::getPid);
        Map<Long, Long> campaignIdsByAdGroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);

        Set<Long> campaignIds = listToSet(campaignIdsByAdGroupIds.values(), Function.identity());
        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService.getTruncatedCampaigns(clientId, campaignIds);

        List<GdAdGroupTruncated> adGroups = groupDataService
                .getTruncatedAdGroups(shard, client.getCountryRegionId(), clientId,
                        operator, adGroupIds, campaignsById);
        Map<Long, GdAdGroupTruncated> adGroupsById = listToMap(adGroups, GdAdGroupTruncated::getId);
        performanceFilters = filterList(performanceFilters,
                smartFilter -> adGroupsById.containsKey(smartFilter.getPid()));

        return StreamEx.of(performanceFilters)
                .map(smartFilter -> toGdSmartFilter(smartFilter, adGroupsById.get(smartFilter.getPid()), performanceFilterStorage))
                .toList();
    }
}
