package ru.yandex.direct.grid.processing.service.smartfilter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.feed.GdSource;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.constants.GdDirectConstants;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterSchema;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterSchemaQuery;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFiltersContainer;
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFiltersContext;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFilters;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdAddSmartFiltersPayload;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdDeleteSmartFilter;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdDeleteSmartFilterPayload;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFilters;
import ru.yandex.direct.grid.processing.model.smartfilter.mutation.GdUpdateSmartFiltersPayload;
import ru.yandex.direct.grid.processing.service.smartfilter.loader.AdGroupBySmartFilterDataLoader;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterSchemaConverter.toBusinessType;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterSchemaConverter.toFeedType;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterSchemaConverter.toSource;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class SmartFilterGraphQlService {

    public static final String SMART_FILTERS_RESOLVER_NAME = "smartFilters";

    private final PerformanceFilterStorage performanceFilterStorage;
    private final SmartFilterDataService smartFilterDataService;
    private final GridValidationService gridValidationService;
    private final AdGroupBySmartFilterDataLoader adGroupBySmartFilterDataLoader;

    @Autowired
    public SmartFilterGraphQlService(
            PerformanceFilterStorage performanceFilterStorage,
            SmartFilterDataService smartFilterDataService,
            GridValidationService gridValidationService,
            AdGroupBySmartFilterDataLoader adGroupBySmartFilterDataLoader) {
        this.performanceFilterStorage = performanceFilterStorage;
        this.smartFilterDataService = smartFilterDataService;
        this.gridValidationService = gridValidationService;
        this.adGroupBySmartFilterDataLoader = adGroupBySmartFilterDataLoader;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = SMART_FILTERS_RESOLVER_NAME)
    public GdSmartFiltersContext getSmartFilters(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSmartFiltersContainer input) {
        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());
        gridValidationService.validateGdSmartFilterContainer(input);

        return smartFilterDataService.getSmartFilters(clientId, input, context);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "adGroup")
    public CompletableFuture<GdAdGroupTruncated> getAdGroup(
            @GraphQLContext GdSmartFilter smartFilter
    ) {
        return adGroupBySmartFilterDataLoader.get().load(smartFilter);
    }

    @GraphQLQuery(name = "smartFilterSchemas")
    public List<GdSmartFilterSchema> smartFilterSchemas(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdDirectConstants ignore,
            @Nullable @GraphQLArgument(name = "query") GdSmartFilterSchemaQuery query
    ) {
        Collection<FilterSchema> schemas = new ArrayList<>();
        if (query == null || query.getBusinessType() == null || (query.getSource() != GdSource.SITE && query.getFeedType() == null)) {
            schemas = performanceFilterStorage.getAllSchemas();
        } else {
            BusinessType businessType = toBusinessType(query.getBusinessType());
            FeedType feedType = toFeedType(query.getFeedType());
            Source source = toSource(query.getSource());
            FilterSchema filterSchemaOrNull =
                    performanceFilterStorage.getFilterSchemaOrNull(businessType, feedType, source);
            if (filterSchemaOrNull == null) {
                return emptyList();
            }
            schemas.add(filterSchemaOrNull);
        }

        return StreamEx.of(schemas)
                .map(SmartFilterSchemaConverter::getFilterSchema)
                .toList();
    }

    /**
     * Мутация для массового добавления smart фильтров
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addSmartFilters")
    public GdAddSmartFiltersPayload addSmartFilters(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddSmartFilters input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return smartFilterDataService.addSmartFilters(clientId, context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateFilters")
    public GdUpdateSmartFiltersPayload updateFilter(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateSmartFilters input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return smartFilterDataService.updateFilter(clientId, context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "deleteFilters")
    public GdDeleteSmartFilterPayload deleteFilters(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdDeleteSmartFilter input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return smartFilterDataService.deleteFilters(clientId, context.getOperator().getUid(), input);
    }

}
